declare module Tone {
    /**
     *  @class  Tone.AmplitudeEnvelope is a Tone.Envelope connected to a gain node.
     *          Unlike Tone.Envelope, which outputs the envelope's value, Tone.AmplitudeEnvelope accepts
     *          an audio signal as the input and will apply the envelope to the amplitude
     *          of the signal. Read more about ADSR Envelopes on [Wikipedia](https://en.wikipedia.org/wiki/Synthesizer#ADSR_envelope).
     *
     *  @constructor
     *  @extends {Tone.Envelope}
     *  @param {Time|Object} [attack] The amount of time it takes for the envelope to go from
     *                               0 to it's maximum value.
     *  @param {Time} [decay]	The period of time after the attack that it takes for the envelope
     *                       	to fall to the sustain value.
     *  @param {NormalRange} [sustain]	The percent of the maximum value that the envelope rests at until
     *                                	the release is triggered.
     *  @param {Time} [release]	The amount of time after the release is triggered it takes to reach 0.
     *  @example
     * var ampEnv = new Tone.AmplitudeEnvelope({
     * 	"attack": 0.1,
     * 	"decay": 0.2,
     * 	"sustain": 1.0,
     * 	"release": 0.8
     * }).toMaster();
     * //create an oscillator and connect it
     * var osc = new Tone.Oscillator().connect(ampEnv).start();
     * //trigger the envelopes attack and release "8t" apart
     * ampEnv.triggerAttackRelease("8t");
     */
    class AmplitudeEnvelope extends Tone.Envelope {
        constructor(attack?: Time | any, decay?: Time, sustain?: NormalRange, release?: Time);
        /**
         *  Clean up
         *  @return  {Tone.AmplitudeEnvelope}  this
         */
        dispose(): Tone.AmplitudeEnvelope;
        /**
         *  When triggerAttack is called, the attack time is the amount of
         *  time it takes for the envelope to reach it's maximum value.
         *  @type {Time}
         */
        attack: Time;
        /**
         *  After the attack portion of the envelope, the value will fall
         *  over the duration of the decay time to it's sustain value.
         *  @type {Time}
         */
        decay: Time;
        /**
         * 	The sustain value is the value
         * 	which the envelope rests at after triggerAttack is
         * 	called, but before triggerRelease is invoked.
         *  @type {NormalRange}
         */
        sustain: NormalRange;
        /**
         *  After triggerRelease is called, the envelope's
         *  value will fall to it's miminum value over the
         *  duration of the release time.
         *  @type {Time}
         */
        release: Time;
        /**
         * Read the current value of the envelope. Useful for
         * syncronizing visual output to the envelope.
         * @memberOf Tone.Envelope#
         * @type {Number}
         * @name value
         * @readOnly
         */
        readonly value: number;
        /**
         * The shape of the attack.
         * Can be any of these strings:
         * <ul>
         *   <li>linear</li>
         *   <li>exponential</li>
         *   <li>sine</li>
         *   <li>cosine</li>
         *   <li>bounce</li>
         *   <li>ripple</li>
         *   <li>step</li>
         * </ul>
         * Can also be an array which describes the curve. Values
         * in the array are evenly subdivided and linearly
         * interpolated over the duration of the attack.
         * @memberOf Tone.Envelope#
         * @type {String|Array}
         * @name attackCurve
         * @example
         * env.attackCurve = "linear";
         * @example
         * //can also be an array
         * env.attackCurve = [0, 0.2, 0.3, 0.4, 1]
         */
        attackCurve: string | Array;
        /**
         * The shape of the release. See the attack curve types.
         * @memberOf Tone.Envelope#
         * @type {String|Array}
         * @name releaseCurve
         * @example
         * env.releaseCurve = "linear";
         */
        releaseCurve: string | Array;
        /**
         * The shape of the decay either "linear" or "exponential"
         * @memberOf Tone.Envelope#
         * @type {String}
         * @name decayCurve
         * @example
         * env.decayCurve = "linear";
         */
        decayCurve: string;
        /**
         *  Trigger the attack/decay portion of the ADSR envelope.
         *  @param  {Time} [time=now] When the attack should start.
         *  @param {NormalRange} [velocity=1] The velocity of the envelope scales the vales.
         *                               number between 0-1
         *  @returns {Tone.Envelope} this
         *  @example
         *  //trigger the attack 0.5 seconds from now with a velocity of 0.2
         *  env.triggerAttack("+0.5", 0.2);
         */
        triggerAttack(time?: Time, velocity?: NormalRange): Tone.Envelope;
        /**
         *  Triggers the release of the envelope.
         *  @param  {Time} [time=now] When the release portion of the envelope should start.
         *  @returns {Tone.Envelope} this
         *  @example
         *  //trigger release immediately
         *  env.triggerRelease();
         */
        triggerRelease(time?: Time): Tone.Envelope;
        /**
         *  Get the scheduled value at the given time. This will
         *  return the unconverted (raw) value.
         *  @param  {Number}  time  The time in seconds.
         *  @return  {Number}  The scheduled value at the given time.
         */
        getValueAtTime(time: number): number;
        /**
         *  triggerAttackRelease is shorthand for triggerAttack, then waiting
         *  some duration, then triggerRelease.
         *  @param {Time} duration The duration of the sustain.
         *  @param {Time} [time=now] When the attack should be triggered.
         *  @param {number} [velocity=1] The velocity of the envelope.
         *  @returns {Tone.Envelope} this
         *  @example
         * //trigger the attack and then the release after 0.6 seconds.
         * env.triggerAttackRelease(0.6);
         */
        triggerAttackRelease(duration: Time, time?: Time, velocity?: number): Tone.Envelope;
        /**
         *  Cancels all scheduled envelope changes after the given time.
         *  @param  {Time} after
         *  @returns {Tone.Envelope} this
         */
        cancel(after: Time): Tone.Envelope;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Wrapper around the native Web Audio's
     *          [AnalyserNode](http://webaudio.github.io/web-audio-api/#idl-def-AnalyserNode).
     *          Extracts FFT or Waveform data from the incoming signal.
     *  @extends {Tone.AudioNode}
     *  @param {String=} type The return type of the analysis, either "fft", or "waveform".
     *  @param {Number=} size The size of the FFT. Value must be a power of
     *                       two in the range 16 to 16384.
     */
    class Analyser extends Tone.AudioNode {
        constructor(type?: string, size?: number);
        /**
         *  The default values.
         *  @type {Object}
         *  @const
         */
        static readonly defaults: any;
        /**
         *  Run the analysis given the current settings and return the
         *  result as a TypedArray of length [size](#size).
         *  @returns {TypedArray}
         */
        getValue(): TypedArray;
        /**
         *  The size of analysis. This must be a power of two in the range 16 to 16384.
         *  @memberOf Tone.Analyser#
         *  @type {Number}
         *  @name size
         */
        size: number;
        /**
         *  The analysis function returned by analyser.getValue(), either "fft" or "waveform".
         *  @memberOf Tone.Analyser#
         *  @type {String}
         *  @name type
         */
        type: string;
        /**
         *  0 represents no time averaging with the last analysis frame.
         *  @memberOf Tone.Analyser#
         *  @type {NormalRange}
         *  @name smoothing
         */
        smoothing: NormalRange;
        /**
         *  Clean up.
         *  @return  {Tone.Analyser}  this
         */
        dispose(): Tone.Analyser;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    module Analyser {
        /**
         *  Possible return types of analyser.getValue()
         *  @enum {String}
         */
        enum Type {
            Waveform,
            FFT
        }
    }
    /**
     *  @class Tone.Channel provides a channel strip interface with
     *  volume, pan, solo and mute controls.
     *
     *  @extends {Tone.AudioNode}
     *  @constructor
     *  @param {Decibels} volume The output volume.
     *  @param {AudioRange} pan the initial pan
     *  @example
     * //pan the incoming signal left and drop the volume
     * var channel = new Tone.Channel(-0.25, -12);
     */
    class Channel extends Tone.AudioNode {
        constructor(volume: Decibels, pan: AudioRange);
        /**
         *  The L/R panning control.
         *  @type {AudioRange}
         *  @signal
         */
        pan: AudioRange;
        /**
         *  The volume control in decibels.
         *  @type {Decibels}
         *  @signal
         */
        volume: Decibels;
        /**
         *  The defaults
         *  @type  {Object}
         *  @const
         *  @static
         */
        static readonly defaults: any;
        /**
         * Solo/unsolo the channel. Soloing is only relative to other
         * Tone.Channels and Tone.Solos.
         * @memberOf Tone.Channel#
         * @name solo
         * @type {Boolean}
         */
        solo: boolean;
        /**
         *  If the current instance is muted, i.e. another instance is soloed,
         *  or the channel is muted
         *  @memberOf Tone.Channel#
         *  @type {Boolean}
         *  @name muted
         *  @readOnly
         */
        readonly muted: boolean;
        /**
         * Mute/unmute the volume
         * @memberOf Tone.Channel#
         * @name mute
         * @type {Boolean}
         */
        mute: boolean;
        /**
         *  clean up
         *  @returns {Tone.Channel} this
         */
        dispose(): Tone.Channel;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.Compressor is a thin wrapper around the Web Audio
     *         [DynamicsCompressorNode](http://webaudio.github.io/web-audio-api/#the-dynamicscompressornode-interface).
     *         Compression reduces the volume of loud sounds or amplifies quiet sounds
     *         by narrowing or "compressing" an audio signal's dynamic range.
     *         Read more on [Wikipedia](https://en.wikipedia.org/wiki/Dynamic_range_compression).
     *
     *  @extends {Tone.AudioNode}
     *  @constructor
     *  @param {Decibels|Object} [threshold] The value above which the compression starts to be applied.
     *  @param {Positive} [ratio] The gain reduction ratio.
     *  @example
     * var comp = new Tone.Compressor(-30, 3);
     */
    class Compressor extends Tone.AudioNode {
        constructor(threshold?: Decibels | any, ratio?: Positive);
        /**
         *  the threshold value
         *  @type {Decibels}
         *  @signal
         */
        threshold: Decibels;
        /**
         *  The attack parameter
         *  @type {Time}
         *  @signal
         */
        attack: Time;
        /**
         *  The release parameter
         *  @type {Time}
         *  @signal
         */
        release: Time;
        /**
         *  The knee parameter
         *  @type {Decibels}
         *  @signal
         */
        knee: Decibels;
        /**
         *  The ratio value
         *  @type {Number}
         *  @signal
         */
        ratio: number;
        /**
         *  @static
         *  @const
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         *  clean up
         *  @returns {Tone.Compressor} this
         */
        dispose(): Tone.Compressor;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     * @class  Tone.Crossfade provides equal power fading between two inputs.
     *         More on crossfading technique [here](https://en.wikipedia.org/wiki/Fade_(audio_engineering)#Crossfading).
     *
     * @constructor
     * @extends {Tone.AudioNode}
     * @param {NormalRange} [initialFade=0.5]
     * @example
     * var crossFade = new Tone.CrossFade(0.5);
     * //connect effect A to crossfade from
     * //effect output 0 to crossfade input 0
     * effectA.connect(crossFade, 0, 0);
     * //connect effect B to crossfade from
     * //effect output 0 to crossfade input 1
     * effectB.connect(crossFade, 0, 1);
     * crossFade.fade.value = 0;
     * // ^ only effectA is output
     * crossFade.fade.value = 1;
     * // ^ only effectB is output
     * crossFade.fade.value = 0.5;
     * // ^ the two signals are mixed equally.
     */
    class CrossFade extends Tone.AudioNode {
        constructor(initialFade?: NormalRange);
        /**
         *  Alias for <code>input[0]</code>.
         *  @type {Tone.Gain}
         */
        a: Tone.Gain;
        /**
         *  Alias for <code>input[1]</code>.
         *  @type {Tone.Gain}
         */
        b: Tone.Gain;
        /**
         * 	The mix between the two inputs. A fade value of 0
         * 	will output 100% <code>input[0]</code> and
         * 	a value of 1 will output 100% <code>input[1]</code>.
         *  @type {NormalRange}
         *  @signal
         */
        fade: NormalRange;
        /**
         *  clean up
         *  @returns {Tone.CrossFade} this
         */
        dispose(): Tone.CrossFade;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.EQ3 is a three band EQ with control over low, mid, and high gain as
     *         well as the low and high crossover frequencies.
     *
     *  @constructor
     *  @extends {Tone.AudioNode}
     *
     *  @param {Decibels|Object} [lowLevel] The gain applied to the lows.
     *  @param {Decibels} [midLevel] The gain applied to the mid.
     *  @param {Decibels} [highLevel] The gain applied to the high.
     *  @example
     * var eq = new Tone.EQ3(-10, 3, -20);
     */
    class EQ3 extends Tone.AudioNode {
        constructor(lowLevel?: Decibels | any, midLevel?: Decibels, highLevel?: Decibels);
        /**
         * The gain in decibels of the low part
         * @type {Decibels}
         * @signal
         */
        low: Decibels;
        /**
         * The gain in decibels of the mid part
         * @type {Decibels}
         * @signal
         */
        mid: Decibels;
        /**
         * The gain in decibels of the high part
         * @type {Decibels}
         * @signal
         */
        high: Decibels;
        /**
         *  The Q value for all of the filters.
         *  @type {Positive}
         *  @signal
         */
        Q: Positive;
        /**
         *  The low/mid crossover frequency.
         *  @type {Frequency}
         *  @signal
         */
        lowFrequency: Frequency;
        /**
         *  The mid/high crossover frequency.
         *  @type {Frequency}
         *  @signal
         */
        highFrequency: Frequency;
        /**
         *  the default values
         */
        static defaults: any;
        /**
         *  clean up
         *  @returns {Tone.EQ3} this
         */
        dispose(): Tone.EQ3;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Tone.Envelope is an [ADSR](https://en.wikipedia.org/wiki/Synthesizer#ADSR_envelope)
     *          envelope generator. Tone.Envelope outputs a signal which
     *          can be connected to an AudioParam or Tone.Signal.
     *          <img src="https://upload.wikimedia.org/wikipedia/commons/e/ea/ADSR_parameter.svg">
     *
     *  @constructor
     *  @extends {Tone.AudioNode}
     *  @param {Time} [attack] The amount of time it takes for the envelope to go from
     *                         0 to it's maximum value.
     *  @param {Time} [decay]	The period of time after the attack that it takes for the envelope
     *                       	to fall to the sustain value. Value must be greater than 0.
     *  @param {NormalRange} [sustain]	The percent of the maximum value that the envelope rests at until
     *                                	the release is triggered.
     *  @param {Time} [release]	The amount of time after the release is triggered it takes to reach 0.
     *                         	Value must be greater than 0.
     *  @example
     * //an amplitude envelope
     * var gainNode = Tone.context.createGain();
     * var env = new Tone.Envelope({
     * 	"attack" : 0.1,
     * 	"decay" : 0.2,
     * 	"sustain" : 1,
     * 	"release" : 0.8,
     * });
     * env.connect(gainNode.gain);
     */
    class Envelope extends Tone.AudioNode {
        constructor(attack?: Time, decay?: Time, sustain?: NormalRange, release?: Time);
        /**
         *  When triggerAttack is called, the attack time is the amount of
         *  time it takes for the envelope to reach it's maximum value.
         *  @type {Time}
         */
        attack: Time;
        /**
         *  After the attack portion of the envelope, the value will fall
         *  over the duration of the decay time to it's sustain value.
         *  @type {Time}
         */
        decay: Time;
        /**
         * 	The sustain value is the value
         * 	which the envelope rests at after triggerAttack is
         * 	called, but before triggerRelease is invoked.
         *  @type {NormalRange}
         */
        sustain: NormalRange;
        /**
         *  After triggerRelease is called, the envelope's
         *  value will fall to it's miminum value over the
         *  duration of the release time.
         *  @type {Time}
         */
        release: Time;
        /**
         *  the default parameters
         *  @static
         *  @const
         */
        static readonly defaults: any;
        /**
         * Read the current value of the envelope. Useful for
         * syncronizing visual output to the envelope.
         * @memberOf Tone.Envelope#
         * @type {Number}
         * @name value
         * @readOnly
         */
        readonly value: number;
        /**
         * The shape of the attack.
         * Can be any of these strings:
         * <ul>
         *   <li>linear</li>
         *   <li>exponential</li>
         *   <li>sine</li>
         *   <li>cosine</li>
         *   <li>bounce</li>
         *   <li>ripple</li>
         *   <li>step</li>
         * </ul>
         * Can also be an array which describes the curve. Values
         * in the array are evenly subdivided and linearly
         * interpolated over the duration of the attack.
         * @memberOf Tone.Envelope#
         * @type {String|Array}
         * @name attackCurve
         * @example
         * env.attackCurve = "linear";
         * @example
         * //can also be an array
         * env.attackCurve = [0, 0.2, 0.3, 0.4, 1]
         */
        attackCurve: string | Array;
        /**
         * The shape of the release. See the attack curve types.
         * @memberOf Tone.Envelope#
         * @type {String|Array}
         * @name releaseCurve
         * @example
         * env.releaseCurve = "linear";
         */
        releaseCurve: string | Array;
        /**
         * The shape of the decay either "linear" or "exponential"
         * @memberOf Tone.Envelope#
         * @type {String}
         * @name decayCurve
         * @example
         * env.decayCurve = "linear";
         */
        decayCurve: string;
        /**
         *  Trigger the attack/decay portion of the ADSR envelope.
         *  @param  {Time} [time=now] When the attack should start.
         *  @param {NormalRange} [velocity=1] The velocity of the envelope scales the vales.
         *                               number between 0-1
         *  @returns {Tone.Envelope} this
         *  @example
         *  //trigger the attack 0.5 seconds from now with a velocity of 0.2
         *  env.triggerAttack("+0.5", 0.2);
         */
        triggerAttack(time?: Time, velocity?: NormalRange): Tone.Envelope;
        /**
         *  Triggers the release of the envelope.
         *  @param  {Time} [time=now] When the release portion of the envelope should start.
         *  @returns {Tone.Envelope} this
         *  @example
         *  //trigger release immediately
         *  env.triggerRelease();
         */
        triggerRelease(time?: Time): Tone.Envelope;
        /**
         *  Get the scheduled value at the given time. This will
         *  return the unconverted (raw) value.
         *  @param  {Number}  time  The time in seconds.
         *  @return  {Number}  The scheduled value at the given time.
         */
        getValueAtTime(time: number): number;
        /**
         *  triggerAttackRelease is shorthand for triggerAttack, then waiting
         *  some duration, then triggerRelease.
         *  @param {Time} duration The duration of the sustain.
         *  @param {Time} [time=now] When the attack should be triggered.
         *  @param {number} [velocity=1] The velocity of the envelope.
         *  @returns {Tone.Envelope} this
         *  @example
         * //trigger the attack and then the release after 0.6 seconds.
         * env.triggerAttackRelease(0.6);
         */
        triggerAttackRelease(duration: Time, time?: Time, velocity?: number): Tone.Envelope;
        /**
         *  Cancels all scheduled envelope changes after the given time.
         *  @param  {Time} after
         *  @returns {Tone.Envelope} this
         */
        cancel(after: Time): Tone.Envelope;
        /**
         *  Disconnect and dispose.
         *  @returns {Tone.Envelope} this
         */
        dispose(): Tone.Envelope;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Get the current frequency data of the connected audio source
     *          using a fast Fourier transform.
     *  @extends {Tone.AudioNode}
     *  @param {Number=} size The size of the FFT. Value must be a power of
     *                       two in the range 16 to 16384.
     */
    class FFT extends Tone.AudioNode {
        constructor(size?: number);
        /**
         *  The default values.
         *  @type {Object}
         *  @const
         */
        static readonly defaults: any;
        /**
         *  Gets the current frequency data from the connected audio source.
         *  Returns the frequency data of length [size](#size) as a Float32Array of decibel values.
         *  @returns {TypedArray}
         */
        getValue(): TypedArray;
        /**
         *  The size of analysis. This must be a power of two in the range 16 to 16384.
         *  Determines the size of the array returned by [getValue](#getvalue) (i.e. the number of
         *  frequency bins). Large FFT sizes may be costly to compute.
         *  @memberOf Tone.FFT#
         *  @type {Number}
         *  @name size
         */
        size: number;
        /**
         *  Clean up.
         *  @return  {Tone.FFT}  this
         */
        dispose(): Tone.FFT;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Comb filters are basic building blocks for physical modeling. Read more
     *         about comb filters on [CCRMA's website](https://ccrma.stanford.edu/~jos/pasp/Feedback_Comb_Filters.html).
     *
     *  @extends {Tone.AudioNode}
     *  @constructor
     *  @param {Time|Object} [delayTime] The delay time of the filter.
     *  @param {NormalRange=} resonance The amount of feedback the filter has.
     */
    class FeedbackCombFilter extends Tone.AudioNode {
        constructor(delayTime?: Time | any, resonance?: NormalRange);
        /**
         *  The amount of delay of the comb filter.
         *  @type {Time}
         *  @signal
         */
        delayTime: Time;
        /**
         *  The amount of feedback of the delayed signal.
         *  @type {NormalRange}
         *  @signal
         */
        resonance: NormalRange;
        /**
         *  the default parameters
         *  @static
         *  @const
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         *  clean up
         *  @returns {Tone.FeedbackCombFilter} this
         */
        dispose(): Tone.FeedbackCombFilter;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Tone.Filter is a filter which allows for all of the same native methods
     *          as the [BiquadFilterNode](http://webaudio.github.io/web-audio-api/#the-biquadfilternode-interface).
     *          Tone.Filter has the added ability to set the filter rolloff at -12
     *          (default), -24 and -48.
     *
     *  @constructor
     *  @extends {Tone.AudioNode}
     *  @param {Frequency|Object} [frequency] The cutoff frequency of the filter.
     *  @param {string=} type The type of filter.
     *  @param {number=} rolloff The drop in decibels per octave after the cutoff frequency.
     *                            3 choices: -12, -24, and -48
     *  @example
     *  var filter = new Tone.Filter(200, "highpass");
     */
    class Filter extends Tone.AudioNode {
        constructor(frequency?: Frequency | any, type?: string, rolloff?: number);
        /**
         *  The cutoff frequency of the filter.
         *  @type {Frequency}
         *  @signal
         */
        frequency: Frequency;
        /**
         *  The detune parameter
         *  @type {Cents}
         *  @signal
         */
        detune: Cents;
        /**
         *  The gain of the filter, only used in certain filter types
         *  @type {Number}
         *  @signal
         */
        gain: number;
        /**
         *  The Q or Quality of the filter
         *  @type {Positive}
         *  @signal
         */
        Q: Positive;
        /**
         *  the default parameters
         *
         *  @static
         *  @type {Object}
         */
        static defaults: any;
        /**
         * The type of the filter. Types: "lowpass", "highpass",
         * "bandpass", "lowshelf", "highshelf", "notch", "allpass", or "peaking".
         * @memberOf Tone.Filter#
         * @type {string}
         * @name type
         */
        type: string;
        /**
         * The rolloff of the filter which is the drop in db
         * per octave. Implemented internally by cascading filters.
         * Only accepts the values -12, -24, -48 and -96.
         * @memberOf Tone.Filter#
         * @type {number}
         * @name rolloff
         */
        rolloff: number;
        /**
         * Get the frequency response curve. This curve represets how the filter
         * responses to frequencies between 20hz-20khz.
         * @param  {Number} [len=128] The number of values to return
         * @return {Float32Array}     The frequency response curve between 20-20k
         */
        getFrequencyResponse(len?: number): Float32Array;
        /**
         *  Clean up.
         *  @return {Tone.Filter} this
         */
        dispose(): Tone.Filter;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Tone.Follower is a  crude envelope follower which will follow
     *          the amplitude of an incoming signal. Read more about envelope followers (also known
     *          as envelope detectors) on [Wikipedia](https://en.wikipedia.org/wiki/Envelope_detector).
     *
     *  @constructor
     *  @extends {Tone.AudioNode}
     *  @param {Time} [smoothing=0.05] The rate of change of the follower.
     *  @example
     * var follower = new Tone.Follower(0.3);
     */
    class Follower extends Tone.AudioNode {
        constructor(smoothing?: Time);
        /**
         *  @static
         *  @type {Object}
         */
        static defaults: any;
        /**
         * The attack time.
         * @memberOf Tone.Follower#
         * @type {Time}
         * @name smoothing
         */
        smoothing: Time;
        /**
         *  Borrows the connect method from Signal so that the output can be used
         *  as a Tone.Signal control signal.
         *  @function
         */
        connect(): void;
        /**
         *  dispose
         *  @returns {Tone.Follower} this
         */
        dispose(): Tone.Follower;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.FrequencyEnvelope is a Tone.ScaledEnvelope, but instead of `min` and `max`
     *         it's got a `baseFrequency` and `octaves` parameter.
     *
     *  @extends {Tone.Envelope}
     *  @constructor
     *  @param {Time|Object} [attack]	the attack time in seconds
     *  @param {Time} [decay]	the decay time in seconds
     *  @param {number} [sustain] 	a percentage (0-1) of the full amplitude
     *  @param {Time} [release]	the release time in seconds
     *  @example
     *  var freqEnv = new Tone.FrequencyEnvelope({
     *  	"attack" : 0.2,
     *  	"baseFrequency" : "C2",
     *  	"octaves" : 4
     *  });
     *  freqEnv.connect(oscillator.frequency);
     */
    class FrequencyEnvelope extends Tone.Envelope {
        constructor(attack?: Time | any, decay?: Time, sustain?: number, release?: Time);
        /**
         *  the default parameters
         *  @static
         */
        static defaults: any;
        /**
         * The envelope's mininum output value. This is the value which it
         * starts at.
         * @memberOf Tone.FrequencyEnvelope#
         * @type {Frequency}
         * @name baseFrequency
         */
        baseFrequency: Frequency;
        /**
         * The number of octaves above the baseFrequency that the
         * envelope will scale to.
         * @memberOf Tone.FrequencyEnvelope#
         * @type {Positive}
         * @name octaves
         */
        octaves: Positive;
        /**
         * The envelope's exponent value.
         * @memberOf Tone.FrequencyEnvelope#
         * @type {number}
         * @name exponent
         */
        exponent: number;
        /**
         *  clean up
         *  @returns {Tone.FrequencyEnvelope} this
         */
        dispose(): Tone.FrequencyEnvelope;
        /**
         *  When triggerAttack is called, the attack time is the amount of
         *  time it takes for the envelope to reach it's maximum value.
         *  @type {Time}
         */
        attack: Time;
        /**
         *  After the attack portion of the envelope, the value will fall
         *  over the duration of the decay time to it's sustain value.
         *  @type {Time}
         */
        decay: Time;
        /**
         * 	The sustain value is the value
         * 	which the envelope rests at after triggerAttack is
         * 	called, but before triggerRelease is invoked.
         *  @type {NormalRange}
         */
        sustain: NormalRange;
        /**
         *  After triggerRelease is called, the envelope's
         *  value will fall to it's miminum value over the
         *  duration of the release time.
         *  @type {Time}
         */
        release: Time;
        /**
         * Read the current value of the envelope. Useful for
         * syncronizing visual output to the envelope.
         * @memberOf Tone.Envelope#
         * @type {Number}
         * @name value
         * @readOnly
         */
        readonly value: number;
        /**
         * The shape of the attack.
         * Can be any of these strings:
         * <ul>
         *   <li>linear</li>
         *   <li>exponential</li>
         *   <li>sine</li>
         *   <li>cosine</li>
         *   <li>bounce</li>
         *   <li>ripple</li>
         *   <li>step</li>
         * </ul>
         * Can also be an array which describes the curve. Values
         * in the array are evenly subdivided and linearly
         * interpolated over the duration of the attack.
         * @memberOf Tone.Envelope#
         * @type {String|Array}
         * @name attackCurve
         * @example
         * env.attackCurve = "linear";
         * @example
         * //can also be an array
         * env.attackCurve = [0, 0.2, 0.3, 0.4, 1]
         */
        attackCurve: string | Array;
        /**
         * The shape of the release. See the attack curve types.
         * @memberOf Tone.Envelope#
         * @type {String|Array}
         * @name releaseCurve
         * @example
         * env.releaseCurve = "linear";
         */
        releaseCurve: string | Array;
        /**
         * The shape of the decay either "linear" or "exponential"
         * @memberOf Tone.Envelope#
         * @type {String}
         * @name decayCurve
         * @example
         * env.decayCurve = "linear";
         */
        decayCurve: string;
        /**
         *  Trigger the attack/decay portion of the ADSR envelope.
         *  @param  {Time} [time=now] When the attack should start.
         *  @param {NormalRange} [velocity=1] The velocity of the envelope scales the vales.
         *                               number between 0-1
         *  @returns {Tone.Envelope} this
         *  @example
         *  //trigger the attack 0.5 seconds from now with a velocity of 0.2
         *  env.triggerAttack("+0.5", 0.2);
         */
        triggerAttack(time?: Time, velocity?: NormalRange): Tone.Envelope;
        /**
         *  Triggers the release of the envelope.
         *  @param  {Time} [time=now] When the release portion of the envelope should start.
         *  @returns {Tone.Envelope} this
         *  @example
         *  //trigger release immediately
         *  env.triggerRelease();
         */
        triggerRelease(time?: Time): Tone.Envelope;
        /**
         *  Get the scheduled value at the given time. This will
         *  return the unconverted (raw) value.
         *  @param  {Number}  time  The time in seconds.
         *  @return  {Number}  The scheduled value at the given time.
         */
        getValueAtTime(time: number): number;
        /**
         *  triggerAttackRelease is shorthand for triggerAttack, then waiting
         *  some duration, then triggerRelease.
         *  @param {Time} duration The duration of the sustain.
         *  @param {Time} [time=now] When the attack should be triggered.
         *  @param {number} [velocity=1] The velocity of the envelope.
         *  @returns {Tone.Envelope} this
         *  @example
         * //trigger the attack and then the release after 0.6 seconds.
         * env.triggerAttackRelease(0.6);
         */
        triggerAttackRelease(duration: Time, time?: Time, velocity?: number): Tone.Envelope;
        /**
         *  Cancels all scheduled envelope changes after the given time.
         *  @param  {Time} after
         *  @returns {Tone.Envelope} this
         */
        cancel(after: Time): Tone.Envelope;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Tone.Gate only passes a signal through when the incoming
     *          signal exceeds a specified threshold. To do this, Gate uses
     *          a Tone.Follower to follow the amplitude of the incoming signal.
     *          A common implementation of this class is a [Noise Gate](https://en.wikipedia.org/wiki/Noise_gate).
     *
     *  @constructor
     *  @extends {Tone.AudioNode}
     *  @param {Decibels|Object} [threshold] The threshold above which the gate will open.
     *  @param {Time=} smoothing The follower's smoothing time
     *  @example
     * var gate = new Tone.Gate(-30, 0.2, 0.3).toMaster();
     * var mic = new Tone.UserMedia().connect(gate);
     * //the gate will only pass through the incoming
     * //signal when it's louder than -30db
     */
    class Gate extends Tone.AudioNode {
        constructor(threshold?: Decibels | any, smoothing?: Time);
        /**
         *  @const
         *  @static
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         * The threshold of the gate in decibels
         * @memberOf Tone.Gate#
         * @type {Decibels}
         * @name threshold
         */
        threshold: Decibels;
        /**
         * The attack/decay speed of the gate
         * @memberOf Tone.Gate#
         * @type {Time}
         * @name smoothing
         */
        smoothing: Time;
        /**
         *  Clean up.
         *  @returns {Tone.Gate} this
         */
        dispose(): Tone.Gate;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  LFO stands for low frequency oscillator. Tone.LFO produces an output signal
     *          which can be attached to an AudioParam or Tone.Signal
     *          in order to modulate that parameter with an oscillator. The LFO can
     *          also be synced to the transport to start/stop and change when the tempo changes.
     *
     *  @constructor
     *  @extends {Tone.AudioNode}
     *  @param {Frequency|Object} [frequency] The frequency of the oscillation. Typically, LFOs will be
     *                               in the frequency range of 0.1 to 10 hertz.
     *  @param {number=} min The minimum output value of the LFO.
     *  @param {number=} max The maximum value of the LFO.
     *  @example
     * var lfo = new Tone.LFO("4n", 400, 4000);
     * lfo.connect(filter.frequency);
     */
    class LFO extends Tone.AudioNode {
        constructor(frequency?: Frequency | any, min?: number, max?: number);
        /**
         *  the lfo's frequency
         *  @type {Frequency}
         *  @signal
         */
        frequency: Frequency;
        /**
         * The amplitude of the LFO, which controls the output range between
         * the min and max output. For example if the min is -10 and the max
         * is 10, setting the amplitude to 0.5 would make the LFO modulate
         * between -5 and 5.
         * @type {Number}
         * @signal
         */
        amplitude: number;
        /**
         *  the default parameters
         *
         *  @static
         *  @const
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         *  Start the LFO.
         *  @param  {Time} [time=now] the time the LFO will start
         *  @returns {Tone.LFO} this
         */
        start(time?: Time): Tone.LFO;
        /**
         *  Stop the LFO.
         *  @param  {Time} [time=now] the time the LFO will stop
         *  @returns {Tone.LFO} this
         */
        stop(time?: Time): Tone.LFO;
        /**
         *  Sync the start/stop/pause to the transport
         *  and the frequency to the bpm of the transport
         *  @returns {Tone.LFO} this
         *  @example
         *  lfo.frequency.value = "8n";
         *  lfo.sync().start(0)
         *  //the rate of the LFO will always be an eighth note,
         *  //even as the tempo changes
         */
        sync(): Tone.LFO;
        /**
         *  unsync the LFO from transport control
         *  @returns {Tone.LFO} this
         */
        unsync(): Tone.LFO;
        /**
         * The miniumum output of the LFO.
         * @memberOf Tone.LFO#
         * @type {number}
         * @name min
         */
        min: number;
        /**
         * The maximum output of the LFO.
         * @memberOf Tone.LFO#
         * @type {number}
         * @name max
         */
        max: number;
        /**
         * The type of the oscillator: sine, square, sawtooth, triangle.
         * @memberOf Tone.LFO#
         * @type {string}
         * @name type
         */
        type: string;
        /**
         * The phase of the LFO.
         * @memberOf Tone.LFO#
         * @type {number}
         * @name phase
         */
        phase: number;
        /**
         * The output units of the LFO.
         * @memberOf Tone.LFO#
         * @type {Tone.Type}
         * @name units
         */
        units: Tone.Type;
        /**
         *  Returns the playback state of the source, either "started" or "stopped".
         *  @type {Tone.State}
         *  @readOnly
         *  @memberOf Tone.LFO#
         *  @name state
         */
        readonly state: Tone.State;
        /**
         *  disconnect and dispose
         *  @returns {Tone.LFO} this
         */
        dispose(): Tone.LFO;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.Limiter will limit the loudness of an incoming signal.
     *         It is composed of a Tone.Compressor with a fast attack
     *         and release. Limiters are commonly used to safeguard against
     *         signal clipping. Unlike a compressor, limiters do not provide
     *         smooth gain reduction and almost completely prevent
     *         additional gain above the threshold.
     *
     *  @extends {Tone.AudioNode}
     *  @constructor
     *  @param {number} threshold The theshold above which the limiting is applied.
     *  @example
     *  var limiter = new Tone.Limiter(-6);
     */
    class Limiter extends Tone.AudioNode {
        constructor(threshold: number);
        /**
         * The threshold of of the limiter
         * @type {Decibel}
         * @signal
         */
        threshold: Decibel;
        /**
         *  The default value
         *  @type {Object}
         *  @const
         *  @static
         */
        static readonly defaults: any;
        /**
         *  Clean up.
         *  @returns {Tone.Limiter} this
         */
        dispose(): Tone.Limiter;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.Lowpass is a lowpass feedback comb filter. It is similar to
     *         Tone.FeedbackCombFilter, but includes a lowpass filter.
     *
     *  @extends {Tone.AudioNode}
     *  @constructor
     *  @param {Time|Object} [delayTime] The delay time of the comb filter
     *  @param {NormalRange=} resonance The resonance (feedback) of the comb filter
     *  @param {Frequency=} dampening The cutoff of the lowpass filter dampens the
     *                                signal as it is fedback.
     */
    class LowpassCombFilter extends Tone.AudioNode {
        constructor(delayTime?: Time | any, resonance?: NormalRange, dampening?: Frequency);
        /**
         *  The delayTime of the comb filter.
         *  @type {Time}
         *  @signal
         */
        delayTime: Time;
        /**
         *  The dampening control of the feedback
         *  @type {Frequency}
         *  @signal
         */
        dampening: Frequency;
        /**
         *  The amount of feedback of the delayed signal.
         *  @type {NormalRange}
         *  @signal
         */
        resonance: NormalRange;
        /**
         *  the default parameters
         *  @static
         *  @const
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         *  Clean up.
         *  @returns {Tone.LowpassCombFilter} this
         */
        dispose(): Tone.LowpassCombFilter;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Tone.Merge brings two signals into the left and right
     *          channels of a single stereo channel.
     *
     *  @constructor
     *  @extends {Tone.AudioNode}
     *  @param {number} [channels=2] The number of channels to merge.
     *  @example
     * var merge = new Tone.Merge().toMaster();
     * //routing a sine tone in the left channel
     * //and noise in the right channel
     * var osc = new Tone.Oscillator().connect(merge.left);
     * var noise = new Tone.Noise().connect(merge.right);
     * //starting our oscillators
     * noise.start();
     * osc.start();
     */
    class Merge extends Tone.AudioNode {
        constructor(channels?: number);
        /**
         *  The left input channel.
         *  Alias for <code>input[0]</code>
         *  @type {GainNode}
         */
        left: GainNode;
        /**
         *  The right input channel.
         *  Alias for <code>input[1]</code>.
         *  @type {GainNode}
         */
        right: GainNode;
        /**
         *  Clean up.
         *  @returns {Tone.Merge} this
         */
        dispose(): Tone.Merge;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Tone.Meter gets the [RMS](https://en.wikipedia.org/wiki/Root_mean_square)
     *          of an input signal. It can also get the raw
     *          value of the input signal.
     *
     *  @constructor
     *  @param {Number} smoothing The amount of smoothing applied between frames.
     *  @extends {Tone.AudioNode}
     *  @example
     * var meter = new Tone.Meter();
     * var mic = new Tone.UserMedia().open();
     * //connect mic to the meter
     * mic.connect(meter);
     * //the current level of the mic input in decibels
     * var level = meter.getLevel();
     */
    class Meter extends Tone.AudioNode {
        constructor(smoothing: number);
        /**
         * A value from 0 -> 1 where 0 represents no time averaging with the last analysis frame.
         * @type {Number}
         */
        smoothing: number;
        /**
         *  The defaults
         *  @type {Object}
         *  @static
         *  @const
         */
        static readonly defaults: any;
        /**
         *  Get the current decibel value of the incoming signal
         *  @returns {Decibels}
         */
        getLevel(): Decibels;
        /**
         *  Get the signal value of the incoming signal
         *  @returns {Number}
         */
        getValue(): number;
        /**
         *  Clean up.
         *  @returns {Tone.Meter} this
         */
        dispose(): Tone.Meter;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.MidSideCompressor applies two different compressors to the mid
     *         and side signal components. See Tone.MidSideSplit.
     *
     *  @extends {Tone.AudioNode}
     *  @param {Object} options The options that are passed to the mid and side
     *                          compressors.
     *  @constructor
     */
    class MidSideCompressor extends Tone.AudioNode {
        constructor(options: any);
        /**
         *  The compressor applied to the mid signal
         *  @type  {Tone.Compressor}
         */
        mid: Tone.Compressor;
        /**
         *  The compressor applied to the side signal
         *  @type  {Tone.Compressor}
         */
        side: Tone.Compressor;
        /**
         *  @const
         *  @static
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         *  Clean up.
         *  @returns {Tone.MidSideCompressor} this
         */
        dispose(): Tone.MidSideCompressor;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Mid/Side processing separates the the 'mid' signal
     *         (which comes out of both the left and the right channel)
     *         and the 'side' (which only comes out of the the side channels).
     *         MidSideMerge merges the mid and side signal after they've been seperated
     *         by Tone.MidSideSplit.<br><br>
     *         <code>
     *         Left = (Mid+Side)/sqrt(2);   // obtain left signal from mid and side<br>
     *         Right = (Mid-Side)/sqrt(2);   // obtain right signal from mid and side<br>
     *         </code>
     *
     *  @extends {Tone.AudioNode}
     *  @constructor
     */
    class MidSideMerge extends Tone.AudioNode {
        /**
         *  The mid signal input. Alias for
         *  <code>input[0]</code>
         *  @type  {Tone.Gain}
         */
        mid: Tone.Gain;
        /**
         * Multiply the left by sqrt(1/2)
         * @type {Tone.Multiply}
         */
        _timesTwoLeft: Tone.Multiply;
        /**
         *  The side signal input. Alias for
         *  <code>input[1]</code>
         *  @type  {Tone.Gain}
         */
        side: Tone.Gain;
        /**
         * Multiply the right by sqrt(1/2)
         * @type {Tone.Multiply}
         */
        _timesTwoRight: Tone.Multiply;
        /**
         *  clean up
         *  @returns {Tone.MidSideMerge} this
         */
        dispose(): Tone.MidSideMerge;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Mid/Side processing separates the the 'mid' signal
     *         (which comes out of both the left and the right channel)
     *         and the 'side' (which only comes out of the the side channels). <br><br>
     *         <code>
     *         Mid = (Left+Right)/sqrt(2);   // obtain mid-signal from left and right<br>
     *         Side = (Left-Right)/sqrt(2);   // obtain side-signal from left and righ<br>
     *         </code>
     *
     *  @extends {Tone.AudioNode}
     *  @constructor
     */
    class MidSideSplit extends Tone.AudioNode {
        /**
         *  The mid send. Connect to mid processing. Alias for
         *  <code>output[0]</code>
         *  @type {Tone.Add}
         */
        _midAdd: Tone.Add;
        /**
         * Multiply the _midAdd by sqrt(1/2)
         * @type {Tone.Multiply}
         */
        mid: Tone.Multiply;
        /**
         *  The side output. Connect to side processing. Also Output 1
         *  @type {Tone.Subtract}
         */
        _sideSubtract: Tone.Subtract;
        /**
         * Multiply the _midAdd by sqrt(1/2)
         * @type {Tone.Multiply}
         */
        side: Tone.Multiply;
        /**
         *  clean up
         *  @returns {Tone.MidSideSplit} this
         */
        dispose(): Tone.MidSideSplit;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.Mono coerces the incoming mono or stereo signal into a mono signal
     *         where both left and right channels have the same value. This can be useful
     *         for [stereo imaging](https://en.wikipedia.org/wiki/Stereo_imaging).
     *
     *  @extends {Tone.AudioNode}
     *  @constructor
     */
    class Mono extends Tone.AudioNode {
        /**
         *  clean up
         *  @returns {Tone.Mono} this
         */
        dispose(): Tone.Mono;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class A compressor with seperate controls over low/mid/high dynamics
     *
     *  @extends {Tone.AudioNode}
     *  @constructor
     *  @param {Object} options The low/mid/high compressor settings.
     *  @example
     *  var multiband = new Tone.MultibandCompressor({
     *  	"lowFrequency" : 200,
     *  	"highFrequency" : 1300
     *  	"low" : {
     *  		"threshold" : -12
     *  	}
     *  })
     */
    class MultibandCompressor extends Tone.AudioNode {
        constructor(options: any);
        /**
         *  low/mid crossover frequency.
         *  @type {Frequency}
         *  @signal
         */
        lowFrequency: Frequency;
        /**
         *  mid/high crossover frequency.
         *  @type {Frequency}
         *  @signal
         */
        highFrequency: Frequency;
        /**
         *  The compressor applied to the low frequencies.
         *  @type {Tone.Compressor}
         */
        low: Tone.Compressor;
        /**
         *  The compressor applied to the mid frequencies.
         *  @type {Tone.Compressor}
         */
        mid: Tone.Compressor;
        /**
         *  The compressor applied to the high frequencies.
         *  @type {Tone.Compressor}
         */
        high: Tone.Compressor;
        /**
         *  @const
         *  @static
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         *  clean up
         *  @returns {Tone.MultibandCompressor} this
         */
        dispose(): Tone.MultibandCompressor;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Split the incoming signal into three bands (low, mid, high)
     *         with two crossover frequency controls.
     *
     *  @extends {Tone.AudioNode}
     *  @constructor
     *  @param {Frequency|Object} [lowFrequency] the low/mid crossover frequency
     *  @param {Frequency} [highFrequency] the mid/high crossover frequency
     */
    class MultibandSplit extends Tone.AudioNode {
        constructor(lowFrequency?: Frequency | any, highFrequency?: Frequency);
        /**
         *  The low band. Alias for <code>output[0]</code>
         *  @type {Tone.Filter}
         */
        low: Tone.Filter;
        /**
         *  The mid band output. Alias for <code>output[1]</code>
         *  @type {Tone.Filter}
         */
        mid: Tone.Filter;
        /**
         *  The high band output. Alias for <code>output[2]</code>
         *  @type {Tone.Filter}
         */
        high: Tone.Filter;
        /**
         *  The low/mid crossover frequency.
         *  @type {Frequency}
         *  @signal
         */
        lowFrequency: Frequency;
        /**
         *  The mid/high crossover frequency.
         *  @type {Frequency}
         *  @signal
         */
        highFrequency: Frequency;
        /**
         *  The quality of all the filters
         *  @type {Number}
         *  @signal
         */
        Q: number;
        /**
         *  Clean up.
         *  @returns {Tone.MultibandSplit} this
         */
        dispose(): Tone.MultibandSplit;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.PanVol is a Tone.Panner and Tone.Volume in one.
     *
     *  @extends {Tone.AudioNode}
     *  @constructor
     *  @param {AudioRange} pan the initial pan
     *  @param {number} volume The output volume.
     *  @example
     * //pan the incoming signal left and drop the volume
     * var panVol = new Tone.PanVol(-0.25, -12);
     */
    class PanVol extends Tone.AudioNode {
        constructor(pan: AudioRange, volume: number);
        /**
         *  The L/R panning control.
         *  @type {AudioRange}
         *  @signal
         */
        pan: AudioRange;
        /**
         *  The volume control in decibels.
         *  @type {Decibels}
         *  @signal
         */
        volume: Decibels;
        /**
         *  The defaults
         *  @type  {Object}
         *  @const
         *  @static
         */
        static readonly defaults: any;
        /**
         * Mute/unmute the volume
         * @memberOf Tone.PanVol#
         * @name mute
         * @type {Boolean}
         */
        mute: boolean;
        /**
         *  clean up
         *  @returns {Tone.PanVol} this
         */
        dispose(): Tone.PanVol;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Tone.Panner is an equal power Left/Right Panner and does not
     *          support 3D. Panner uses the StereoPannerNode when available.
     *
     *  @constructor
     *  @extends {Tone.AudioNode}
     *  @param {NormalRange} [initialPan=0] The initail panner value (center).
     *  @example
     *  //pan the input signal hard right.
     *  var panner = new Tone.Panner(1);
     */
    class Panner extends Tone.AudioNode {
        constructor(initialPan?: NormalRange);
        /**
         *  The pan control. -1 = hard left, 1 = hard right.
         *  @type {AudioRange}
         *  @signal
         */
        pan: AudioRange;
        /**
         *  Defaults
         *  @type  {Object}
         *  @const
         *  @static
         */
        static readonly defaults: any;
        /**
         *  Clean up.
         *  @returns {Tone.Panner} this
         */
        dispose(): Tone.Panner;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  A spatialized panner node which supports equalpower or HRTF panning.
     *          Tries to normalize the API across various browsers. See Tone.Listener
     *
     *  @constructor
     *  @extends {Tone.AudioNode}
     *  @param {Number} positionX The initial x position.
     *  @param {Number} positionY The initial y position.
     *  @param {Number} positionZ The initial z position.
     */
    class Panner3D extends Tone.AudioNode {
        constructor(positionX: number, positionY: number, positionZ: number);
        /**
         *  Defaults according to the specification
         *  @static
         *  @const
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         *  Sets the position of the source in 3d space.
         *  @param  {Number}  x
         *  @param  {Number}  y
         *  @param  {Number}  z
         *  @return {Tone.Panner3D} this
         */
        setPosition(x: number, y: number, z: number): Tone.Panner3D;
        /**
         *  Sets the orientation of the source in 3d space.
         *  @param  {Number}  x
         *  @param  {Number}  y
         *  @param  {Number}  z
         *  @return {Tone.Panner3D} this
         */
        setOrientation(x: number, y: number, z: number): Tone.Panner3D;
        /**
         *  The x position of the panner object.
         *  @type {Number}
         *  @memberOf Tone.Panner3D#
         *  @name positionX
         */
        positionX: number;
        /**
         *  The y position of the panner object.
         *  @type {Number}
         *  @memberOf Tone.Panner3D#
         *  @name positionY
         */
        positionY: number;
        /**
         *  The z position of the panner object.
         *  @type {Number}
         *  @memberOf Tone.Panner3D#
         *  @name positionZ
         */
        positionZ: number;
        /**
         *  The x orientation of the panner object.
         *  @type {Number}
         *  @memberOf Tone.Panner3D#
         *  @name orientationX
         */
        orientationX: number;
        /**
         *  The y orientation of the panner object.
         *  @type {Number}
         *  @memberOf Tone.Panner3D#
         *  @name orientationY
         */
        orientationY: number;
        /**
         *  The z orientation of the panner object.
         *  @type {Number}
         *  @memberOf Tone.Panner3D#
         *  @name orientationZ
         */
        orientationZ: number;
        /**
         *  The panning model. Either "equalpower" or "HRTF".
         *  @type {String}
         *  @memberOf Tone.Panner3D#
         *  @name panningModel
         */
        panningModel: string;
        /**
         *  A reference distance for reducing volume as source move further from the listener
         *  @type {Number}
         *  @memberOf Tone.Panner3D#
         *  @name refDistance
         */
        refDistance: number;
        /**
         *  Describes how quickly the volume is reduced as source moves away from listener.
         *  @type {Number}
         *  @memberOf Tone.Panner3D#
         *  @name rolloffFactor
         */
        rolloffFactor: number;
        /**
         *  The distance model used by,  "linear", "inverse", or "exponential".
         *  @type {String}
         *  @memberOf Tone.Panner3D#
         *  @name distanceModel
         */
        distanceModel: string;
        /**
         *  The angle, in degrees, inside of which there will be no volume reduction
         *  @type {Degrees}
         *  @memberOf Tone.Panner3D#
         *  @name coneInnerAngle
         */
        coneInnerAngle: Degrees;
        /**
         *  The angle, in degrees, outside of which the volume will be reduced
         *  to a constant value of coneOuterGain
         *  @type {Degrees}
         *  @memberOf Tone.Panner3D#
         *  @name coneOuterAngle
         */
        coneOuterAngle: Degrees;
        /**
         *  The gain outside of the coneOuterAngle
         *  @type {Gain}
         *  @memberOf Tone.Panner3D#
         *  @name coneOuterGain
         */
        coneOuterGain: Gain;
        /**
         *  The maximum distance between source and listener,
         *  after which the volume will not be reduced any further.
         *  @type {Positive}
         *  @memberOf Tone.Panner3D#
         *  @name maxDistance
         */
        maxDistance: Positive;
        /**
         *  Clean up.
         *  @returns {Tone.Panner3D} this
         */
        dispose(): Tone.Panner3D;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.ScaledEnvelop is an envelope which can be scaled
     *         to any range. It's useful for applying an envelope
     *         to a frequency or any other non-NormalRange signal
     *         parameter.
     *
     *  @extends {Tone.Envelope}
     *  @constructor
     *  @param {Time|Object} [attack]	the attack time in seconds
     *  @param {Time} [decay]	the decay time in seconds
     *  @param {number} [sustain] 	a percentage (0-1) of the full amplitude
     *  @param {Time} [release]	the release time in seconds
     *  @example
     *  var scaledEnv = new Tone.ScaledEnvelope({
     *  	"attack" : 0.2,
     *  	"min" : 200,
     *  	"max" : 2000
     *  });
     *  scaledEnv.connect(oscillator.frequency);
     */
    class ScaledEnvelope extends Tone.Envelope {
        constructor(attack?: Time | any, decay?: Time, sustain?: number, release?: Time);
        /**
         *  the default parameters
         *  @static
         */
        static defaults: any;
        /**
         * The envelope's min output value. This is the value which it
         * starts at.
         * @memberOf Tone.ScaledEnvelope#
         * @type {number}
         * @name min
         */
        min: number;
        /**
         * The envelope's max output value. In other words, the value
         * at the peak of the attack portion of the envelope.
         * @memberOf Tone.ScaledEnvelope#
         * @type {number}
         * @name max
         */
        max: number;
        /**
         * The envelope's exponent value.
         * @memberOf Tone.ScaledEnvelope#
         * @type {number}
         * @name exponent
         */
        exponent: number;
        /**
         *  clean up
         *  @returns {Tone.ScaledEnvelope} this
         */
        dispose(): Tone.ScaledEnvelope;
        /**
         *  When triggerAttack is called, the attack time is the amount of
         *  time it takes for the envelope to reach it's maximum value.
         *  @type {Time}
         */
        attack: Time;
        /**
         *  After the attack portion of the envelope, the value will fall
         *  over the duration of the decay time to it's sustain value.
         *  @type {Time}
         */
        decay: Time;
        /**
         * 	The sustain value is the value
         * 	which the envelope rests at after triggerAttack is
         * 	called, but before triggerRelease is invoked.
         *  @type {NormalRange}
         */
        sustain: NormalRange;
        /**
         *  After triggerRelease is called, the envelope's
         *  value will fall to it's miminum value over the
         *  duration of the release time.
         *  @type {Time}
         */
        release: Time;
        /**
         * Read the current value of the envelope. Useful for
         * syncronizing visual output to the envelope.
         * @memberOf Tone.Envelope#
         * @type {Number}
         * @name value
         * @readOnly
         */
        readonly value: number;
        /**
         * The shape of the attack.
         * Can be any of these strings:
         * <ul>
         *   <li>linear</li>
         *   <li>exponential</li>
         *   <li>sine</li>
         *   <li>cosine</li>
         *   <li>bounce</li>
         *   <li>ripple</li>
         *   <li>step</li>
         * </ul>
         * Can also be an array which describes the curve. Values
         * in the array are evenly subdivided and linearly
         * interpolated over the duration of the attack.
         * @memberOf Tone.Envelope#
         * @type {String|Array}
         * @name attackCurve
         * @example
         * env.attackCurve = "linear";
         * @example
         * //can also be an array
         * env.attackCurve = [0, 0.2, 0.3, 0.4, 1]
         */
        attackCurve: string | Array;
        /**
         * The shape of the release. See the attack curve types.
         * @memberOf Tone.Envelope#
         * @type {String|Array}
         * @name releaseCurve
         * @example
         * env.releaseCurve = "linear";
         */
        releaseCurve: string | Array;
        /**
         * The shape of the decay either "linear" or "exponential"
         * @memberOf Tone.Envelope#
         * @type {String}
         * @name decayCurve
         * @example
         * env.decayCurve = "linear";
         */
        decayCurve: string;
        /**
         *  Trigger the attack/decay portion of the ADSR envelope.
         *  @param  {Time} [time=now] When the attack should start.
         *  @param {NormalRange} [velocity=1] The velocity of the envelope scales the vales.
         *                               number between 0-1
         *  @returns {Tone.Envelope} this
         *  @example
         *  //trigger the attack 0.5 seconds from now with a velocity of 0.2
         *  env.triggerAttack("+0.5", 0.2);
         */
        triggerAttack(time?: Time, velocity?: NormalRange): Tone.Envelope;
        /**
         *  Triggers the release of the envelope.
         *  @param  {Time} [time=now] When the release portion of the envelope should start.
         *  @returns {Tone.Envelope} this
         *  @example
         *  //trigger release immediately
         *  env.triggerRelease();
         */
        triggerRelease(time?: Time): Tone.Envelope;
        /**
         *  Get the scheduled value at the given time. This will
         *  return the unconverted (raw) value.
         *  @param  {Number}  time  The time in seconds.
         *  @return  {Number}  The scheduled value at the given time.
         */
        getValueAtTime(time: number): number;
        /**
         *  triggerAttackRelease is shorthand for triggerAttack, then waiting
         *  some duration, then triggerRelease.
         *  @param {Time} duration The duration of the sustain.
         *  @param {Time} [time=now] When the attack should be triggered.
         *  @param {number} [velocity=1] The velocity of the envelope.
         *  @returns {Tone.Envelope} this
         *  @example
         * //trigger the attack and then the release after 0.6 seconds.
         * env.triggerAttackRelease(0.6);
         */
        triggerAttackRelease(duration: Time, time?: Time, velocity?: number): Tone.Envelope;
        /**
         *  Cancels all scheduled envelope changes after the given time.
         *  @param  {Time} after
         *  @returns {Tone.Envelope} this
         */
        cancel(after: Time): Tone.Envelope;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.Solo lets you isolate a specific audio stream. When
     *         an instance is set to `solo=true`, it will mute all other instances.
     *  @extends {Tone.AudioNode}
     *  @example
     * var soloA = new Tone.Solo()
     * var soloB = new Tone.Solo()
     * soloA.solo = true
     * //no audio will pass through soloB
     */
    class Solo extends Tone.AudioNode {
        /**
         *  The input and output node
         *  @type  {Tone.Gain}
         */
        input: Tone.Gain;
        /**
         *  The defaults
         *  @type  {Object}
         *  @static
         */
        static defaults: any;
        /**
         *  Isolates this instance and mutes all other instances of Tone.Solo.
         *  Only one instance can be soloed at a time. A soloed
         *  instance will report `solo=false` when another instance is soloed.
         *  @memberOf Tone.Solo#
         *  @type {Boolean}
         *  @name solo
         */
        solo: boolean;
        /**
         *  If the current instance is muted, i.e. another instance is soloed
         *  @memberOf Tone.Solo#
         *  @type {Boolean}
         *  @name muted
         *  @readOnly
         */
        readonly muted: boolean;
        /**
         *  Clean up
         *  @return  {Tone.Solo}  this
         */
        dispose(): Tone.Solo;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *	@class  Tone.Split splits an incoming signal into left and right channels.
     *
     *  @constructor
     *  @extends {Tone.AudioNode}
     *  @param {number} [channels=2] The number of channels to merge.
     *  @example
     * var split = new Tone.Split();
     * stereoSignal.connect(split);
     */
    class Split extends Tone.AudioNode {
        constructor(channels?: number);
        /**
         *  Left channel output.
         *  Alias for <code>output[0]</code>
         *  @type {Tone.Gain}
         */
        left: Tone.Gain;
        /**
         *  Right channel output.
         *  Alias for <code>output[1]</code>
         *  @type {Tone.Gain}
         */
        right: Tone.Gain;
        /**
         *  Clean up.
         *  @returns {Tone.Split} this
         */
        dispose(): Tone.Split;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.Volume is a simple volume node, useful for creating a volume fader.
     *
     *  @extends {Tone.AudioNode}
     *  @constructor
     *  @param {Decibels} [volume=0] the initial volume
     *  @example
     * var vol = new Tone.Volume(-12);
     * instrument.chain(vol, Tone.Master);
     */
    class Volume extends Tone.AudioNode {
        constructor(volume?: Decibels);
        /**
         *  The volume control in decibels.
         *  @type {Decibels}
         *  @signal
         */
        volume: Decibels;
        /**
         *  Defaults
         *  @type  {Object}
         *  @const
         *  @static
         */
        static readonly defaults: any;
        /**
         * Mute the output.
         * @memberOf Tone.Volume#
         * @type {boolean}
         * @name mute
         * @example
         * //mute the output
         * volume.mute = true;
         */
        mute: boolean;
        /**
         *  clean up
         *  @returns {Tone.Volume} this
         */
        dispose(): Tone.Volume;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Get the current waveform data of the connected audio source.
     *  @extends {Tone.AudioNode}
     *  @param {Number=} size The size of the FFT. Value must be a power of
     *                       two in the range 32 to 32768.
     */
    class Waveform extends Tone.AudioNode {
        constructor(size?: number);
        /**
         *  The default values.
         *  @type {Object}
         *  @const
         */
        static readonly defaults: any;
        /**
         *  Gets the waveform of the audio source. Returns the waveform data
         *  of length [size](#size) as a Float32Array with values between -1 and 1.
         *  @returns {TypedArray}
         */
        getValue(): TypedArray;
        /**
         *  The size of analysis. This must be a power of two in the range 32 to 32768.
         *  @memberOf Tone.Waveform#
         *  @type {Number}
         *  @name size
         */
        size: number;
        /**
         *  Clean up.
         *  @return  {Tone.Waveform}  this
         */
        dispose(): Tone.Waveform;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.CtrlInterpolate will interpolate between given values based
     *         on the "index" property. Passing in an array or object literal
     *         will interpolate each of the parameters. Note (i.e. "C3")
     *         and Time (i.e. "4n + 2") can be interpolated. All other values are
     *         assumed to be numbers.
     *  @example
     * var interp = new Tone.CtrlInterpolate([0, 2, 9, 4]);
     * interp.index = 0.75;
     * interp.value; //returns 1.5
     *
     *  @example
     * var interp = new Tone.CtrlInterpolate([
     * 	[2, 4, 5],
     * 	[9, 3, 2],
     * ]);
     * @param {Array} values The array of values to interpolate over
     * @param {Positive} index The initial interpolation index.
     * @extends {Tone}
     */
    class CtrlInterpolate extends Tone {
        constructor(values: Array, index: Positive);
        /**
         *  The values to interpolate between
         *  @type  {Array}
         */
        values: Array;
        /**
         *  The interpolated index between values. For example: a value of 1.5
         *  would interpolate equally between the value at index 1
         *  and the value at index 2.
         *  @example
         * interp.index = 0;
         * interp.value; //returns the value at 0
         * interp.index = 0.5;
         * interp.value; //returns the value between indices 0 and 1.
         *  @type  {Positive}
         */
        index: Positive;
        /**
         *  The defaults
         *  @const
         *  @type  {Object}
         */
        static readonly defaults: any;
        /**
         *  The current interpolated value based on the index
         *  @readOnly
         *  @memberOf Tone.CtrlInterpolate#
         *  @type {*}
         *  @name value
         */
        readonly value: any;
        /**
         *  Clean up
         *  @return  {Tone.CtrlInterpolate}  this
         */
        dispose(): Tone.CtrlInterpolate;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.CtrlMarkov represents a Markov Chain where each call
     *         to Tone.CtrlMarkov.next will move to the next state. If the next
     *         state choice is an array, the next state is chosen randomly with
     *         even probability for all of the choices. For a weighted probability
     *         of the next choices, pass in an object with "state" and "probability" attributes.
     *         The probabilities will be normalized and then chosen. If no next options
     *         are given for the current state, the state will stay there.
     *  @extends {Tone}
     *  @example
     * var chain = new Tone.CtrlMarkov({
     * 	"beginning" : ["end", "middle"],
     * 	"middle" : "end"
     * });
     * chain.value = "beginning";
     * chain.next(); //returns "end" or "middle" with 50% probability
     *
     *  @example
     * var chain = new Tone.CtrlMarkov({
     * 	"beginning" : [{"value" : "end", "probability" : 0.8},
     * 					{"value" : "middle", "probability" : 0.2}],
     * 	"middle" : "end"
     * });
     * chain.value = "beginning";
     * chain.next(); //returns "end" with 80% probability or "middle" with 20%.
     *  @param {Object} values An object with the state names as the keys
     *                         and the next state(s) as the values.
     */
    class CtrlMarkov extends Tone {
        constructor(values: any);
        /**
         *  The Markov values with states as the keys
         *  and next state(s) as the values.
         *  @type {Object}
         */
        values: any;
        /**
         *  The current state of the Markov values. The next
         *  state will be evaluated and returned when Tone.CtrlMarkov.next
         *  is invoked.
         *  @type {String}
         */
        value: string;
        /**
         *  Returns the next state of the Markov values.
         *  @return  {String}
         */
        next(): string;
        /**
         *  Clean up
         *  @return  {Tone.CtrlMarkov}  this
         */
        dispose(): Tone.CtrlMarkov;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Generate patterns from an array of values.
     *         Has a number of arpeggiation and randomized
     *         selection patterns.
     *           <ul>
     *  	        <li>"up" - cycles upward</li>
     *  			<li>"down" - cycles downward</li>
     *  			<li>"upDown" - up then and down</li>
     *  			<li>"downUp" - cycles down then and up</li>
     *  			<li>"alternateUp" - jump up two and down one</li>
     *  			<li>"alternateDown" - jump down two and up one</li>
     *  			<li>"random" - randomly select an index</li>
     *  			<li>"randomWalk" - randomly moves one index away from the current position</li>
     *  			<li>"randomOnce" - randomly select an index without repeating until all values have been chosen.</li>
     *     		</ul>
     *  @param  {Array}  values   An array of options to choose from.
     *  @param  {Tone.CtrlPattern.Type=}  type  The name of the pattern.
     *  @extends {Tone}
     */
    class CtrlPattern extends Tone {
        constructor(values: Array, type?: Tone.CtrlPattern.Type);
        /**
         *  The array of values to arpeggiate over
         *  @type {Array}
         */
        values: Array;
        /**
         *  The current position in the values array
         *  @type  {Number}
         */
        index: number;
        /**
         *  The Control Patterns
         *  @type  {Object}
         *  @static
         */
        static Type: any;
        /**
         *  The default values.
         *  @type  {Object}
         */
        static defaults: any;
        /**
         *  The value at the current index of the pattern.
         *  @readOnly
         *  @memberOf Tone.CtrlPattern#
         *  @type {*}
         *  @name value
         */
        readonly value: any;
        /**
         *  The pattern used to select the next
         *  item from the values array
         *  @memberOf Tone.CtrlPattern#
         *  @type {Tone.CtrlPattern.Type}
         *  @name type
         */
        type: Tone.CtrlPattern.Type;
        /**
         *  Return the next value given the current position
         *  and pattern.
         *  @return {*} The next value
         */
        next(): any;
        /**
         *  Clean up
         *  @returns {Tone.CtrlPattern} this
         */
        dispose(): Tone.CtrlPattern;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Choose a random value.
     *  @extends {Tone}
     *  @example
     * var randomWalk = new Tone.CtrlRandom({
     * 	"min" : 0,
     * 	"max" : 10,
     * 	"integer" : true
     * });
     * randomWalk.eval();
     *
     *  @param {Number|Time=} min The minimum return value.
     *  @param {Number|Time=} max The maximum return value.
     */
    class CtrlRandom extends Tone {
        constructor(min: number | Time, max: number | Time);
        /**
         *  The minimum return value
         *  @type  {Number|Time}
         */
        min: number | Time;
        /**
         *  The maximum return value
         *  @type  {Number|Time}
         */
        max: number | Time;
        /**
         *  If the return value should be an integer
         *  @type  {Boolean}
         */
        integer: boolean;
        /**
         *  The defaults
         *  @const
         *  @type  {Object}
         */
        static readonly defaults: any;
        /**
         *  Return a random value between min and max.
         *  @readOnly
         *  @memberOf Tone.CtrlRandom#
         *  @type {*}
         *  @name value
         */
        readonly value: any;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.AudioNode is the base class for classes which process audio.
     *         AudioNodes have inputs and outputs.
     *  @param	{AudioContext=} context	The audio context to use with the class
     *  @extends {Tone}
     */
    class AudioNode extends Tone {
        constructor(context?: AudioContext);
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         * Dispose and disconnect
         * @return {Tone.AudioNode} this
         */
        dispose(): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Buffer loading and storage. Tone.Buffer is used internally by all
     *          classes that make requests for audio files such as Tone.Player,
     *          Tone.Sampler and Tone.Convolver.
     *
     *          Aside from load callbacks from individual buffers, Tone.Buffer
     *  		provides events which keep track of the loading progress
     *  		of _all_ of the buffers. These are Tone.Buffer.on("load" / "progress" / "error")
     *
     *  @constructor
     *  @extends {Tone}
     *  @param {AudioBuffer|String} url The url to load, or the audio buffer to set.
     *  @param {Function=} onload A callback which is invoked after the buffer is loaded.
     *                            It's recommended to use `Tone.Buffer.on('load', callback)` instead
     *                            since it will give you a callback when _all_ buffers are loaded.
     *  @param {Function=} onerror The callback to invoke if there is an error
     *  @example
     * var buffer = new Tone.Buffer("path/to/sound.mp3", function(){
     * 	//the buffer is now available.
     * 	var buff = buffer.get();
     * });
     *  @example
     * //can load provide fallback extension types if the first type is not supported.
     * var buffer = new Tone.Buffer("path/to/sound.[mp3|ogg|wav]");
     */
    class Buffer extends Tone {
        constructor(url: AudioBuffer | string, onload?: (...params: any[]) => any, onerror?: (...params: any[]) => any);
        /**
         *  the default parameters
         *  @type {Object}
         */
        static defaults: any;
        /**
         *  Pass in an AudioBuffer or Tone.Buffer to set the value
         *  of this buffer.
         *  @param {AudioBuffer|Tone.Buffer} buffer the buffer
         *  @returns {Tone.Buffer} this
         */
        set(buffer: AudioBuffer | Tone.Buffer): Tone.Buffer;
        /**
         *  @return {AudioBuffer} The audio buffer stored in the object.
         */
        get(): AudioBuffer;
        /**
         *  Makes an xhr reqest for the selected url then decodes
         *  the file as an audio buffer. Invokes
         *  the callback once the audio buffer loads.
         *  @param {String} url The url of the buffer to load.
         *                      filetype support depends on the
         *                      browser.
         *  @returns {Promise} returns a Promise which resolves with the Tone.Buffer
         */
        load(url: string): Promise;
        /**
         *  dispose and disconnect
         *  @returns {Tone.Buffer} this
         */
        dispose(): Tone.Buffer;
        /**
         * If the buffer is loaded or not
         * @memberOf Tone.Buffer#
         * @type {Boolean}
         * @name loaded
         * @readOnly
         */
        readonly loaded: boolean;
        /**
         * The duration of the buffer.
         * @memberOf Tone.Buffer#
         * @type {Number}
         * @name duration
         * @readOnly
         */
        readonly duration: number;
        /**
         * The length of the buffer in samples
         * @memberOf Tone.Buffer#
         * @type {Number}
         * @name length
         * @readOnly
         */
        readonly length: number;
        /**
         * The number of discrete audio channels. Returns 0 if no buffer
         * is loaded.
         * @memberOf Tone.Buffer#
         * @type {Number}
         * @name numberOfChannels
         * @readOnly
         */
        readonly numberOfChannels: number;
        /**
         *  Set the audio buffer from the array. To create a multichannel AudioBuffer,
         *  pass in a multidimensional array.
         *  @param {Float32Array} array The array to fill the audio buffer
         *  @return {Tone.Buffer} this
         */
        fromArray(array: Float32Array): Tone.Buffer;
        /**
         * 	Sums muliple channels into 1 channel
         *  @param {Number=} channel Optionally only copy a single channel from the array.
         *  @return {Array}
         */
        toMono(channel?: number): Array;
        /**
         * 	Get the buffer as an array. Single channel buffers will return a 1-dimensional
         * 	Float32Array, and multichannel buffers will return multidimensional arrays.
         *  @param {Number=} channel Optionally only copy a single channel from the array.
         *  @return {Array}
         */
        toArray(channel?: number): Array;
        /**
         *  Returns the Float32Array representing the PCM audio data for the specific channel.
         *  @param  {Number}  channel  The channel number to return
         *  @return  {Float32Array}  The audio as a TypedArray
         */
        getChannelData(channel: number): Float32Array;
        /**
         *  Cut a subsection of the array and return a buffer of the
         *  subsection. Does not modify the original buffer
         *  @param {Time} start The time to start the slice
         *  @param {Time=} end The end time to slice. If none is given
         *                     will default to the end of the buffer
         *  @return {Tone.Buffer} this
         */
        slice(start: Time, end?: Time): Tone.Buffer;
        /**
         * Reverse the buffer.
         * @memberOf Tone.Buffer#
         * @type {Boolean}
         * @name reverse
         */
        reverse: boolean;
        /**
         *  A path which is prefixed before every url.
         *  @type  {String}
         *  @static
         */
        static baseUrl: string;
        /**
         *  Create a Tone.Buffer from the array. To create a multichannel AudioBuffer,
         *  pass in a multidimensional array.
         *  @param {Float32Array} array The array to fill the audio buffer
         *  @return {Tone.Buffer} A Tone.Buffer created from the array
         */
        static fromArray(array: Float32Array): Tone.Buffer;
        /**
         * Creates a Tone.Buffer from a URL, returns a promise
         * which resolves to a Tone.Buffer
         * @param  {String} url The url to load.
         * @return {Promise<Tone.Buffer>}     A promise which resolves to a Tone.Buffer
         */
        static fromUrl(url: string): Promise<Tone.Buffer>;
        /**
         *  Loads a url using XMLHttpRequest.
         *  @param {String} url
         *  @param {Function} onload
         *  @param {Function} onerror
         *  @param {Function} onprogress
         *  @return {XMLHttpRequest}
         */
        static load(url: string, onload: (...params: any[]) => any, onerror: (...params: any[]) => any, onprogress: (...params: any[]) => any): XMLHttpRequest;
        /**
         *  Stop all of the downloads in progress
         *  @return {Tone.Buffer}
         *  @static
         */
        static cancelDownloads(): Tone.Buffer;
        /**
         *  Checks a url's extension to see if the current browser can play that file type.
         *  @param {String} url The url/extension to test
         *  @return {Boolean} If the file extension can be played
         *  @static
         *  @example
         * Tone.Buffer.supportsType("wav"); //returns true
         * Tone.Buffer.supportsType("path/to/file.wav"); //returns true
         */
        static supportsType(url: string): boolean;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class A data structure for holding multiple buffers.
     *
     *  @param  {Object|Array}    urls      An object literal or array
     *                                      of urls to load.
     *  @param  {Function=}  callback  The callback to invoke when
     *                                 the buffers are loaded.
     *  @extends {Tone}
     *  @example
     * //load a whole bank of piano samples
     * var pianoSamples = new Tone.Buffers({
     * 	"C4" : "path/to/C4.mp3"
     * 	"C#4" : "path/to/C#4.mp3"
     * 	"D4" : "path/to/D4.mp3"
     * 	"D#4" : "path/to/D#4.mp3"
     * 	...
     * }, function(){
     * 	//play one of the samples when they all load
     * 	player.buffer = pianoSamples.get("C4");
     * 	player.start();
     * });
     * 	@example
     * //To pass in additional parameters in the second parameter
     * var buffers = new Tone.Buffers(urls, {
     * 	"onload" : callback,
     * 	"baseUrl" : "../path/to/audio/"
     * })
     */
    class Buffers extends Tone {
        constructor(urls: any | Array, callback?: (...params: any[]) => any);
        /**
         *  A path which is prefixed before every url.
         *  @type  {String}
         */
        baseUrl: string;
        /**
         *  Defaults
         *  @type  {Object}
         */
        static defaults: any;
        /**
         *  True if the buffers object has a buffer by that name.
         *  @param  {String|Number}  name  The key or index of the
         *                                 buffer.
         *  @return  {Boolean}
         */
        has(name: string | number): boolean;
        /**
         *  Get a buffer by name. If an array was loaded,
         *  then use the array index.
         *  @param  {String|Number}  name  The key or index of the
         *                                 buffer.
         *  @return  {Tone.Buffer}
         */
        get(name: string | number): Tone.Buffer;
        /**
         * If the buffers are loaded or not
         * @memberOf Tone.Buffers#
         * @type {Boolean}
         * @name loaded
         * @readOnly
         */
        readonly loaded: boolean;
        /**
         *  Add a buffer by name and url to the Buffers
         *  @param  {String}    name      A unique name to give
         *                                the buffer
         *  @param  {String|Tone.Buffer|Audiobuffer}  url  Either the url of the bufer,
         *                                                 or a buffer which will be added
         *                                                 with the given name.
         *  @param  {Function=}  callback  The callback to invoke
         *                                 when the url is loaded.
         */
        add(name: string, url: string | Tone.Buffer | Audiobuffer, callback?: (...params: any[]) => any): void;
        /**
         *  Clean up.
         *  @return  {Tone.Buffers} this
         */
        dispose(): Tone.Buffers;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  A sample accurate clock which provides a callback at the given rate.
     *          While the callback is not sample-accurate (it is still susceptible to
     *          loose JS timing), the time passed in as the argument to the callback
     *          is precise. For most applications, it is better to use Tone.Transport
     *          instead of the Clock by itself since you can synchronize multiple callbacks.
     *
     * 	@constructor
     *  @extends {Tone.Emitter}
     * 	@param {function} callback The callback to be invoked with the time of the audio event
     * 	@param {Frequency} frequency The rate of the callback
     * 	@example
     * //the callback will be invoked approximately once a second
     * //and will print the time exactly once a second apart.
     * var clock = new Tone.Clock(function(time){
     * 	console.log(time);
     * }, 1);
     */
    class Clock extends Tone.Emitter {
        constructor(callback: (...params: any[]) => any, frequency: Frequency);
        /**
         *  The callback function to invoke at the scheduled tick.
         *  @type  {Function}
         */
        callback: (...params: any[]) => any;
        /**
         *  The rate the callback function should be invoked.
         *  @type  {BPM}
         *  @signal
         */
        frequency: BPM;
        /**
         *  The defaults
         *  @const
         *  @type  {Object}
         */
        static readonly defaults: any;
        /**
         *  Returns the playback state of the source, either "started", "stopped" or "paused".
         *  @type {Tone.State}
         *  @readOnly
         *  @memberOf Tone.Clock#
         *  @name state
         */
        readonly state: Tone.State;
        /**
         *  Start the clock at the given time. Optionally pass in an offset
         *  of where to start the tick counter from.
         *  @param  {Time=}  time    The time the clock should start
         *  @param  {Ticks=}  offset  Where the tick counter starts counting from.
         *  @return  {Tone.Clock}  this
         */
        start(time?: Time, offset?: Ticks): Tone.Clock;
        /**
         *  Stop the clock. Stopping the clock resets the tick counter to 0.
         *  @param {Time} [time=now] The time when the clock should stop.
         *  @returns {Tone.Clock} this
         *  @example
         * clock.stop();
         */
        stop(time?: Time): Tone.Clock;
        /**
         *  Pause the clock. Pausing does not reset the tick counter.
         *  @param {Time} [time=now] The time when the clock should stop.
         *  @returns {Tone.Clock} this
         */
        pause(time?: Time): Tone.Clock;
        /**
         *  Return the elapsed seconds at the given time.
         *  @param  {Time}  time  When to get the elapsed seconds
         *  @return  {Seconds}  The number of elapsed seconds
         */
        getSecondsAtTime(time: Time): Seconds;
        /**
         * Set the clock's ticks at the given time.
         * @param  {Ticks} ticks The tick value to set
         * @param  {Time} time  When to set the tick value
         * @return {Tone.Clock}       this
         */
        setTicksAtTime(ticks: Ticks, time: Time): Tone.Clock;
        /**
         * Get the clock's ticks at the given time.
         * @param  {Time} time  When to get the tick value
         * @return {Ticks}       The tick value at the given time.
         */
        getTicksAtTime(time: Time): Ticks;
        /**
         * Get the time of the next tick
         * @param  {Ticks} ticks The tick number.
         * @param  {Time} before
         * @return {Tone.Clock}       this
         */
        nextTickTime(ticks: Ticks, before: Time): Tone.Clock;
        /**
         *  Returns the scheduled state at the given time.
         *  @param  {Time}  time  The time to query.
         *  @return  {String}  The name of the state input in setStateAtTime.
         *  @example
         * clock.start("+0.1");
         * clock.getStateAtTime("+0.1"); //returns "started"
         */
        getStateAtTime(time: Time): string;
        /**
         *  Clean up
         *  @returns {Tone.Clock} this
         */
        dispose(): Tone.Clock;
        /**
         *  Bind a callback to a specific event.
         *  @param  {String}    event     The name of the event to listen for.
         *  @param  {Function}  callback  The callback to invoke when the
         *                                event is emitted
         *  @return  {Tone.Emitter}    this
         */
        on(event: string, callback: (...params: any[]) => any): Tone.Emitter;
        /**
         *  Bind a callback which is only invoked once
         *  @param  {String}    event     The name of the event to listen for.
         *  @param  {Function}  callback  The callback to invoke when the
         *                                event is emitted
         *  @return  {Tone.Emitter}    this
         */
        once(event: string, callback: (...params: any[]) => any): Tone.Emitter;
        /**
         *  Remove the event listener.
         *  @param  {String}    event     The event to stop listening to.
         *  @param  {Function=}  callback  The callback which was bound to
         *                                the event with Tone.Emitter.on.
         *                                If no callback is given, all callbacks
         *                                events are removed.
         *  @return  {Tone.Emitter}    this
         */
        off(event: string, callback?: (...params: any[]) => any): Tone.Emitter;
        /**
         *  Invoke all of the callbacks bound to the event
         *  with any arguments passed in.
         *  @param  {String}  event  The name of the event.
         *  @param {...*} args The arguments to pass to the functions listening.
         *  @return  {Tone.Emitter}  this
         */
        emit(event: string, ...args: any[]): Tone.Emitter;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Wrapper around the native AudioContext.
     *  @extends {Tone.Emitter}
     *  @param {AudioContext=} context optionally pass in a context
     */
    class Context extends Tone.Emitter {
        constructor(context?: AudioContext);
        /**
         *  The amount of time events are scheduled
         *  into the future
         *  @type  {Number}
         */
        lookAhead: number;
        /**
         * defaults
         * @static
         * @type {Object}
         */
        static defaults: any;
        /**
         * Is an instanceof Tone.Context
         * @type {Boolean}
         */
        isContext: boolean;
        /**
         *  The current audio context time
         *  @return  {Number}
         */
        now(): number;
        /**
         *  Starts the audio context from a suspended state. This is required
         *  to initially start the AudioContext.
         *  @return  {Promise}
         */
        resume(): Promise;
        /**
         *  Promise which is invoked when the context is running.
         *  Tries to resume the context if it's not started.
         *  @return  {Promise}
         */
        close(): Promise;
        /**
         *  Generate a looped buffer at some constant value.
         *  @param  {Number}  val
         *  @return  {BufferSourceNode}
         */
        getConstant(val: number): BufferSourceNode;
        /**
         *  A setTimeout which is gaurenteed by the clock source.
         *  Also runs in the offline context.
         *  @param  {Function}  fn       The callback to invoke
         *  @param  {Seconds}    timeout  The timeout in seconds
         *  @returns {Number} ID to use when invoking Tone.Context.clearTimeout
         */
        setTimeout(fn: (...params: any[]) => any, timeout: Seconds): number;
        /**
         *  Clears a previously scheduled timeout with Tone.context.setTimeout
         *  @param  {Number}  id  The ID returned from setTimeout
         *  @return  {Tone.Context}  this
         */
        clearTimeout(id: number): Tone.Context;
        /**
         *  How often the Web Worker callback is invoked.
         *  This number corresponds to how responsive the scheduling
         *  can be. Context.updateInterval + Context.lookAhead gives you the
         *  total latency between scheduling an event and hearing it.
         *  @type {Number}
         *  @memberOf Tone.Context#
         *  @name updateInterval
         */
        updateInterval: number;
        /**
         *  The unwrapped AudioContext.
         *  @type {AudioContext}
         *  @memberOf Tone.Context#
         *  @name rawContext
         *  @readOnly
         */
        readonly rawContext: AudioContext;
        /**
         *  What the source of the clock is, either "worker" (Web Worker [default]),
         *  "timeout" (setTimeout), or "offline" (none).
         *  @type {String}
         *  @memberOf Tone.Context#
         *  @name clockSource
         */
        clockSource: string;
        /**
         *  The type of playback, which affects tradeoffs between audio
         *  output latency and responsiveness.
         *
         *  In addition to setting the value in seconds, the latencyHint also
         *  accepts the strings "interactive" (prioritizes low latency),
         *  "playback" (prioritizes sustained playback), "balanced" (balances
         *  latency and performance), and "fastest" (lowest latency, might glitch more often).
         *  @type {String|Seconds}
         *  @memberOf Tone.Context#
         *  @name latencyHint
         *  @example
         * //set the lookAhead to 0.3 seconds
         * Tone.context.latencyHint = 0.3;
         */
        latencyHint: string | Seconds;
        /**
         *  Unlike other dispose methods, this returns a Promise
         *  which executes when the context is closed and disposed
         *  @returns {Promise} this
         */
        dispose(): Promise;
        /**
         *  Bind a callback to a specific event.
         *  @param  {String}    event     The name of the event to listen for.
         *  @param  {Function}  callback  The callback to invoke when the
         *                                event is emitted
         *  @return  {Tone.Emitter}    this
         */
        on(event: string, callback: (...params: any[]) => any): Tone.Emitter;
        /**
         *  Bind a callback which is only invoked once
         *  @param  {String}    event     The name of the event to listen for.
         *  @param  {Function}  callback  The callback to invoke when the
         *                                event is emitted
         *  @return  {Tone.Emitter}    this
         */
        once(event: string, callback: (...params: any[]) => any): Tone.Emitter;
        /**
         *  Remove the event listener.
         *  @param  {String}    event     The event to stop listening to.
         *  @param  {Function=}  callback  The callback which was bound to
         *                                the event with Tone.Emitter.on.
         *                                If no callback is given, all callbacks
         *                                events are removed.
         *  @return  {Tone.Emitter}    this
         */
        off(event: string, callback?: (...params: any[]) => any): Tone.Emitter;
        /**
         *  Invoke all of the callbacks bound to the event
         *  with any arguments passed in.
         *  @param  {String}  event  The name of the event.
         *  @param {...*} args The arguments to pass to the functions listening.
         *  @return  {Tone.Emitter}  this
         */
        emit(event: string, ...args: any[]): Tone.Emitter;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Wrapper around Web Audio's native [DelayNode](http://webaudio.github.io/web-audio-api/#the-delaynode-interface).
     *  @extends {Tone}
     *  @param {Time=} delayTime The delay applied to the incoming signal.
     *  @param {Time=} maxDelay The maximum delay time.
     */
    class Delay extends Tone {
        constructor(delayTime?: Time, maxDelay?: Time);
        /**
         *  The amount of time the incoming signal is
         *  delayed.
         *  @type {Time}
         *  @signal
         */
        delayTime: Time;
        /**
         *  The defaults
         *  @const
         *  @type  {Object}
         */
        static readonly defaults: any;
        /**
         * The maximum delay time. This cannot be changed. The value is passed into the constructor.
         * @memberof Tone.Delay#
         * @type {Time}
         * @name maxDelay
         * @readOnly
         */
        readonly maxDelay: Time;
        /**
         *  Clean up.
         *  @return  {Tone.Delay}  this
         */
        dispose(): Tone.Delay;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.Draw is useful for synchronizing visuals and audio events.
     *         Callbacks from Tone.Transport or any of the Tone.Event classes
     *         always happen _before_ the scheduled time and are not synchronized
     *         to the animation frame so they are not good for triggering tightly
     *         synchronized visuals and sound. Tone.Draw makes it easy to schedule
     *         callbacks using the AudioContext time and uses requestAnimationFrame.
     *
     *  @singleton
     *  @extends {Tone}
     *  @example
     * Tone.Transport.schedule(function(time){
     * 	//use the time argument to schedule a callback with Tone.Draw
     * 	Tone.Draw.schedule(function(){
     * 		//do drawing or DOM manipulation here
     * 	}, time)
     * }, "+0.5")
     */
    class Draw extends Tone {
        /**
         *  The duration after which events are not invoked.
         *  @type  {Number}
         *  @default 0.25
         */
        expiration: number;
        /**
         *  The amount of time before the scheduled time
         *  that the callback can be invoked. Default is
         *  half the time of an animation frame (0.008 seconds).
         *  @type  {Number}
         *  @default 0.008
         */
        anticipation: number;
        /**
         *  Schedule a function at the given time to be invoked
         *  on the nearest animation frame.
         *  @param  {Function}  callback  Callback is invoked at the given time.
         *  @param  {Time}    time      The time relative to the AudioContext time
         *                              to invoke the callback.
         *  @return  {Tone.Draw}    this
         */
        schedule(callback: (...params: any[]) => any, time: Time): Tone.Draw;
        /**
         *  Cancel events scheduled after the given time
         *  @param  {Time=}  after  Time after which scheduled events will
         *                          be removed from the scheduling timeline.
         *  @return  {Tone.Draw}  this
         */
        cancel(after?: Time): Tone.Draw;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.Emitter gives classes which extend it
     *         the ability to listen for and emit events.
     *         Inspiration and reference from Jerome Etienne's [MicroEvent](https://github.com/jeromeetienne/microevent.js).
     *         MIT (c) 2011 Jerome Etienne.
     *
     *  @extends {Tone}
     */
    class Emitter extends Tone {
        /**
         *  Bind a callback to a specific event.
         *  @param  {String}    event     The name of the event to listen for.
         *  @param  {Function}  callback  The callback to invoke when the
         *                                event is emitted
         *  @return  {Tone.Emitter}    this
         */
        on(event: string, callback: (...params: any[]) => any): Tone.Emitter;
        /**
         *  Bind a callback which is only invoked once
         *  @param  {String}    event     The name of the event to listen for.
         *  @param  {Function}  callback  The callback to invoke when the
         *                                event is emitted
         *  @return  {Tone.Emitter}    this
         */
        once(event: string, callback: (...params: any[]) => any): Tone.Emitter;
        /**
         *  Remove the event listener.
         *  @param  {String}    event     The event to stop listening to.
         *  @param  {Function=}  callback  The callback which was bound to
         *                                the event with Tone.Emitter.on.
         *                                If no callback is given, all callbacks
         *                                events are removed.
         *  @return  {Tone.Emitter}    this
         */
        off(event: string, callback?: (...params: any[]) => any): Tone.Emitter;
        /**
         *  Invoke all of the callbacks bound to the event
         *  with any arguments passed in.
         *  @param  {String}  event  The name of the event.
         *  @param {...*} args The arguments to pass to the functions listening.
         *  @return  {Tone.Emitter}  this
         */
        emit(event: string, ...args: any[]): Tone.Emitter;
        /**
         *  Add Emitter functions (on/off/emit) to the object
         *  @param  {Object|Function}  object  The object or class to extend.
         *  @returns {Tone.Emitter}
         */
        static mixin(object: any | ((...params: any[]) => any)): Tone.Emitter;
        /**
         *  Clean up
         *  @return  {Tone.Emitter}  this
         */
        dispose(): Tone.Emitter;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class A thin wrapper around the Native Web Audio GainNode.
     *         The GainNode is a basic building block of the Web Audio
     *         API and is useful for routing audio and adjusting gains.
     *  @extends {Tone.AudioNode}
     *  @param  {Number=}  gain  The initial gain of the GainNode
     *  @param {Tone.Type=} units The units of the gain parameter.
     */
    class Gain extends Tone.AudioNode {
        constructor(gain?: number, units?: Tone.Type);
        /**
         *  The gain parameter of the gain node.
         *  @type {Gain}
         *  @signal
         */
        gain: Gain;
        /**
         *  The defaults
         *  @const
         *  @type  {Object}
         */
        static readonly defaults: any;
        /**
         *  Clean up.
         *  @return  {Tone.Gain}  this
         */
        dispose(): Tone.Gain;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Similar to Tone.Timeline, but all events represent
     *         intervals with both "time" and "duration" times. The
     *         events are placed in a tree structure optimized
     *         for querying an intersection point with the timeline
     *         events. Internally uses an [Interval Tree](https://en.wikipedia.org/wiki/Interval_tree)
     *         to represent the data.
     *  @extends {Tone}
     */
    class IntervalTimeline extends Tone {
        /**
         *  The event to add to the timeline. All events must
         *  have a time and duration value
         *  @param  {Object}  event  The event to add to the timeline
         *  @return  {Tone.IntervalTimeline}  this
         */
        add(event: any): Tone.IntervalTimeline;
        /**
         *  Remove an event from the timeline.
         *  @param  {Object}  event  The event to remove from the timeline
         *  @return  {Tone.IntervalTimeline}  this
         */
        remove(event: any): Tone.IntervalTimeline;
        /**
         *  The number of items in the timeline.
         *  @type {Number}
         *  @memberOf Tone.IntervalTimeline#
         *  @name length
         *  @readOnly
         */
        readonly length: number;
        /**
         *  Remove events whose time time is after the given time
         *  @param  {Number}  time  The time to query.
         *  @returns {Tone.IntervalTimeline} this
         */
        cancel(time: number): Tone.IntervalTimeline;
        /**
         *  Get an event whose time and duration span the give time. Will
         *  return the match whose "time" value is closest to the given time.
         *  @param  {Object}  event  The event to add to the timeline
         *  @return  {Object}  The event which spans the desired time
         */
        get(event: any): any;
        /**
         *  Iterate over everything in the timeline.
         *  @param  {Function}  callback The callback to invoke with every item
         *  @returns {Tone.IntervalTimeline} this
         */
        forEach(callback: (...params: any[]) => any): Tone.IntervalTimeline;
        /**
         *  Iterate over everything in the array in which the given time
         *  overlaps with the time and duration time of the event.
         *  @param  {Number}  time The time to check if items are overlapping
         *  @param  {Function}  callback The callback to invoke with every item
         *  @returns {Tone.IntervalTimeline} this
         */
        forEachAtTime(time: number, callback: (...params: any[]) => any): Tone.IntervalTimeline;
        /**
         *  Iterate over everything in the array in which the time is greater
         *  than or equal to the given time.
         *  @param  {Number}  time The time to check if items are before
         *  @param  {Function}  callback The callback to invoke with every item
         *  @returns {Tone.IntervalTimeline} this
         */
        forEachFrom(time: number, callback: (...params: any[]) => any): Tone.IntervalTimeline;
        /**
         *  Clean up
         *  @return  {Tone.IntervalTimeline}  this
         */
        dispose(): Tone.IntervalTimeline;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Both Tone.Panner3D and Tone.Listener have a position in 3D space
     *          using a right-handed cartesian coordinate system.
     *          The units used in the coordinate system are not defined;
     *          these coordinates are independent/invariant of any particular
     *          units such as meters or feet. Tone.Panner3D objects have an forward
     *          vector representing the direction the sound is projecting. Additionally,
     *          they have a sound cone representing how directional the sound is.
     *          For example, the sound could be omnidirectional, in which case it would
     *          be heard anywhere regardless of its forward, or it can be more directional
     *          and heard only if it is facing the listener. Tone.Listener objects
     *          (representing a person's ears) have an forward and up vector
     *          representing in which direction the person is facing. Because both the
     *          source stream and the listener can be moving, they both have a velocity
     *          vector representing both the speed and direction of movement. Taken together,
     *          these two velocities can be used to generate a doppler shift effect which changes the pitch.
     *          <br><br>
     *          Note: the position of the Listener will have no effect on nodes not connected to a Tone.Panner3D
     *
     *  @constructor
     *  @extends {Tone}
     *  @singleton
     */
    class Listener extends Tone {
        /**
         *  Defaults according to the specification
         *  @static
         *  @const
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         * Is an instanceof Tone.Listener
         * @type {Boolean}
         */
        isListener: boolean;
        /**
         *  Sets the position of the listener in 3d space.
         *  @param  {Number}  x
         *  @param  {Number}  y
         *  @param  {Number}  z
         *  @return {Tone.Listener} this
         */
        setPosition(x: number, y: number, z: number): Tone.Listener;
        /**
         *  Sets the orientation of the listener using two vectors, the forward
         *  vector (which direction the listener is facing) and the up vector
         *  (which the up direction of the listener). An up vector
         *  of 0, 0, 1 is equivalent to the listener standing up in the Z direction.
         *  @param  {Number}  x
         *  @param  {Number}  y
         *  @param  {Number}  z
         *  @param  {Number}  upX
         *  @param  {Number}  upY
         *  @param  {Number}  upZ
         *  @return {Tone.Listener} this
         */
        setOrientation(x: number, y: number, z: number, upX: number, upY: number, upZ: number): Tone.Listener;
        /**
         *  The x position of the panner object.
         *  @type {Number}
         *  @memberOf Tone.Listener#
         *  @name positionX
         */
        positionX: number;
        /**
         *  The y position of the panner object.
         *  @type {Number}
         *  @memberOf Tone.Listener#
         *  @name positionY
         */
        positionY: number;
        /**
         *  The z position of the panner object.
         *  @type {Number}
         *  @memberOf Tone.Listener#
         *  @name positionZ
         */
        positionZ: number;
        /**
         *  The x coordinate of the listeners front direction. i.e.
         *  which way they are facing.
         *  @type {Number}
         *  @memberOf Tone.Listener#
         *  @name forwardX
         */
        forwardX: number;
        /**
         *  The y coordinate of the listeners front direction. i.e.
         *  which way they are facing.
         *  @type {Number}
         *  @memberOf Tone.Listener#
         *  @name forwardY
         */
        forwardY: number;
        /**
         *  The z coordinate of the listeners front direction. i.e.
         *  which way they are facing.
         *  @type {Number}
         *  @memberOf Tone.Listener#
         *  @name forwardZ
         */
        forwardZ: number;
        /**
         *  The x coordinate of the listener's up direction. i.e.
         *  the direction the listener is standing in.
         *  @type {Number}
         *  @memberOf Tone.Listener#
         *  @name upX
         */
        upX: number;
        /**
         *  The y coordinate of the listener's up direction. i.e.
         *  the direction the listener is standing in.
         *  @type {Number}
         *  @memberOf Tone.Listener#
         *  @name upY
         */
        upY: number;
        /**
         *  The z coordinate of the listener's up direction. i.e.
         *  the direction the listener is standing in.
         *  @type {Number}
         *  @memberOf Tone.Listener#
         *  @name upZ
         */
        upZ: number;
        /**
         *  Clean up.
         *  @returns {Tone.Listener} this
         */
        dispose(): Tone.Listener;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  A single master output which is connected to the
     *          AudioDestinationNode (aka your speakers).
     *          It provides useful conveniences such as the ability
     *          to set the volume and mute the entire application.
     *          It also gives you the ability to apply master effects to your application.
     *          <br><br>
     *          Like Tone.Transport, A single Tone.Master is created
     *          on initialization and you do not need to explicitly construct one.
     *
     *  @constructor
     *  @extends {Tone}
     *  @singleton
     *  @example
     * //the audio will go from the oscillator to the speakers
     * oscillator.connect(Tone.Master);
     * //a convenience for connecting to the master output is also provided:
     * oscillator.toMaster();
     * //the above two examples are equivalent.
     */
    class Master extends Tone {
        /**
         * The volume of the master output.
         * @type {Decibels}
         * @signal
         */
        volume: Decibels;
        /**
         *  @type {Object}
         *  @const
         */
        static readonly defaults: any;
        /**
         * Is an instanceof Tone.Master
         * @type {Boolean}
         */
        isMaster: boolean;
        /**
         * Mute the output.
         * @memberOf Tone.Master#
         * @type {boolean}
         * @name mute
         * @example
         * //mute the output
         * Tone.Master.mute = true;
         */
        mute: boolean;
        /**
         *  Add a master effects chain. NOTE: this will disconnect any nodes which were previously
         *  chained in the master effects chain.
         *  @param {...(AudioNode|Tone)} nodes All arguments will be connected in a row
         *                                  and the Master will be routed through it.
         *  @return  {Tone.Master}  this
         *  @example
         * //some overall compression to keep the levels in check
         * var masterCompressor = new Tone.Compressor({
         * 	"threshold" : -6,
         * 	"ratio" : 3,
         * 	"attack" : 0.5,
         * 	"release" : 0.1
         * });
         * //give a little boost to the lows
         * var lowBump = new Tone.Filter(200, "lowshelf");
         * //route everything through the filter
         * //and compressor before going to the speakers
         * Tone.Master.chain(lowBump, masterCompressor);
         */
        chain(...nodes: (AudioNode | Tone)[]): Tone.Master;
        /**
         *  Clean up
         *  @return  {Tone.Master}  this
         */
        dispose(): Tone.Master;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Wrapper around the OfflineAudioContext
     *  @extends {Tone.Context}
     *  @param  {Number}  channels  The number of channels to render
     *  @param  {Number}  duration  The duration to render in samples
     *  @param {Number} sampleRate the sample rate to render at
     */
    class OfflineContext extends Tone.Context {
        constructor(channels: number, duration: number, sampleRate: number);
        /**
         *  Override the now method to point to the internal clock time
         *  @return  {Number}
         */
        now(): number;
        /**
         *  Overwrite resume, should not do anything in the OfflineAudioContext.
         *  @return {Promise}
         */
        resume(): Promise;
        /**
         *  Render the output of the OfflineContext
         *  @return  {Promise}
         */
        render(): Promise;
        /**
         *  Close the context
         *  @return  {Promise}
         */
        close(): Promise;
        /**
         *  The amount of time events are scheduled
         *  into the future
         *  @type  {Number}
         */
        lookAhead: number;
        /**
         * Is an instanceof Tone.Context
         * @type {Boolean}
         */
        isContext: boolean;
        /**
         *  Generate a looped buffer at some constant value.
         *  @param  {Number}  val
         *  @return  {BufferSourceNode}
         */
        getConstant(val: number): BufferSourceNode;
        /**
         *  A setTimeout which is gaurenteed by the clock source.
         *  Also runs in the offline context.
         *  @param  {Function}  fn       The callback to invoke
         *  @param  {Seconds}    timeout  The timeout in seconds
         *  @returns {Number} ID to use when invoking Tone.Context.clearTimeout
         */
        setTimeout(fn: (...params: any[]) => any, timeout: Seconds): number;
        /**
         *  Clears a previously scheduled timeout with Tone.context.setTimeout
         *  @param  {Number}  id  The ID returned from setTimeout
         *  @return  {Tone.Context}  this
         */
        clearTimeout(id: number): Tone.Context;
        /**
         *  How often the Web Worker callback is invoked.
         *  This number corresponds to how responsive the scheduling
         *  can be. Context.updateInterval + Context.lookAhead gives you the
         *  total latency between scheduling an event and hearing it.
         *  @type {Number}
         *  @memberOf Tone.Context#
         *  @name updateInterval
         */
        updateInterval: number;
        /**
         *  The unwrapped AudioContext.
         *  @type {AudioContext}
         *  @memberOf Tone.Context#
         *  @name rawContext
         *  @readOnly
         */
        readonly rawContext: AudioContext;
        /**
         *  What the source of the clock is, either "worker" (Web Worker [default]),
         *  "timeout" (setTimeout), or "offline" (none).
         *  @type {String}
         *  @memberOf Tone.Context#
         *  @name clockSource
         */
        clockSource: string;
        /**
         *  The type of playback, which affects tradeoffs between audio
         *  output latency and responsiveness.
         *
         *  In addition to setting the value in seconds, the latencyHint also
         *  accepts the strings "interactive" (prioritizes low latency),
         *  "playback" (prioritizes sustained playback), "balanced" (balances
         *  latency and performance), and "fastest" (lowest latency, might glitch more often).
         *  @type {String|Seconds}
         *  @memberOf Tone.Context#
         *  @name latencyHint
         *  @example
         * //set the lookAhead to 0.3 seconds
         * Tone.context.latencyHint = 0.3;
         */
        latencyHint: string | Seconds;
        /**
         *  Unlike other dispose methods, this returns a Promise
         *  which executes when the context is closed and disposed
         *  @returns {Promise} this
         */
        dispose(): Promise;
        /**
         *  Bind a callback to a specific event.
         *  @param  {String}    event     The name of the event to listen for.
         *  @param  {Function}  callback  The callback to invoke when the
         *                                event is emitted
         *  @return  {Tone.Emitter}    this
         */
        on(event: string, callback: (...params: any[]) => any): Tone.Emitter;
        /**
         *  Bind a callback which is only invoked once
         *  @param  {String}    event     The name of the event to listen for.
         *  @param  {Function}  callback  The callback to invoke when the
         *                                event is emitted
         *  @return  {Tone.Emitter}    this
         */
        once(event: string, callback: (...params: any[]) => any): Tone.Emitter;
        /**
         *  Remove the event listener.
         *  @param  {String}    event     The event to stop listening to.
         *  @param  {Function=}  callback  The callback which was bound to
         *                                the event with Tone.Emitter.on.
         *                                If no callback is given, all callbacks
         *                                events are removed.
         *  @return  {Tone.Emitter}    this
         */
        off(event: string, callback?: (...params: any[]) => any): Tone.Emitter;
        /**
         *  Invoke all of the callbacks bound to the event
         *  with any arguments passed in.
         *  @param  {String}  event  The name of the event.
         *  @param {...*} args The arguments to pass to the functions listening.
         *  @return  {Tone.Emitter}  this
         */
        emit(event: string, ...args: any[]): Tone.Emitter;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.Param wraps the native Web Audio's AudioParam to provide
     *         additional unit conversion functionality. It also
     *         serves as a base-class for classes which have a single,
     *         automatable parameter.
     *  @extends {Tone.AudioNode}
     *  @param  {AudioParam}  param  The parameter to wrap.
     *  @param  {Tone.Type} units The units of the audio param.
     *  @param  {Boolean} convert If the param should be converted.
     */
    class Param extends Tone.AudioNode {
        constructor(param: AudioParam, units: Tone.Type, convert: boolean);
        /**
         *  The units of the parameter
         *  @type {Tone.Type}
         */
        units: Tone.Type;
        /**
         *  If the value should be converted or not
         *  @type {Boolean}
         */
        convert: boolean;
        /**
         *  Defaults
         *  @type  {Object}
         *  @const
         */
        static readonly defaults: any;
        /**
         * The current value of the parameter.
         * @memberOf Tone.Param#
         * @type {Number}
         * @name value
         */
        value: number;
        /**
         * The current value of the parameter.
         * @memberOf Tone.Param#
         * @type {Number}
         * @name value
         */
        value: number;
        /**
         * The current value of the parameter.
         * @memberOf Tone.Param#
         * @type {Number}
         * @name value
         */
        value: number;
        /**
         *  Schedules a parameter value change at the given time.
         *  @param {*}	value The value to set the signal.
         *  @param {Time}  time The time when the change should occur.
         *  @returns {Tone.Param} this
         *  @example
         * //set the frequency to "G4" in exactly 1 second from now.
         * freq.setValueAtTime("G4", "+1");
         */
        setValueAtTime(value: any, time: Time): Tone.Param;
        /**
         *  Get the signals value at the given time. Subsequent scheduling
         *  may invalidate the returned value.
         *  @param {Time} time When to get the value
         *  @returns {Number} The value at the given time
         */
        getValueAtTime(time: Time): number;
        /**
         *  Creates a schedule point with the current value at the current time.
         *  This is useful for creating an automation anchor point in order to
         *  schedule changes from the current value.
         *
         *  @param {number=} now (Optionally) pass the now value in.
         *  @returns {Tone.Param} this
         */
        setRampPoint(now?: number): Tone.Param;
        /**
         *  Schedules a linear continuous change in parameter value from the
         *  previous scheduled parameter value to the given value.
         *
         *  @param  {number} value
         *  @param  {Time} endTime
         *  @returns {Tone.Param} this
         */
        linearRampToValueAtTime(value: number, endTime: Time): Tone.Param;
        /**
         *  Schedules an exponential continuous change in parameter value from
         *  the previous scheduled parameter value to the given value.
         *
         *  @param  {number} value
         *  @param  {Time} endTime
         *  @returns {Tone.Param} this
         */
        exponentialRampToValueAtTime(value: number, endTime: Time): Tone.Param;
        /**
         *  Schedules an exponential continuous change in parameter value from
         *  the current time and current value to the given value over the
         *  duration of the rampTime.
         *
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        exponentialRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Schedules an linear continuous change in parameter value from
         *  the current time and current value to the given value over the
         *  duration of the rampTime.
         *
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //linearly ramp to the value 4 over 3 seconds.
         * signal.linearRampTo(4, 3);
         */
        linearRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time. Since it
         *  is an exponential approach it will continue approaching after the ramp duration. The
         *  rampTime is the time that it takes to reach over 99% of the way towards the value.
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        targetRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time. Since it
         *  is an exponential approach it will continue approaching after the ramp duration. The
         *  rampTime is the time that it takes to reach over 99% of the way towards the value. This methods
         *  is similar to setTargetAtTime except the third argument is a time instead of a 'timeConstant'
         *  @param  {number} value   The value to ramp to.
         *  @param {Time}	time 	When the ramp should start.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        exponentialApproachValueAtTime(value: number, time: Time, rampTime: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time with
         *  a rate having the given time constant.
         *  @param {number} value
         *  @param {Time} startTime
         *  @param {number} timeConstant
         *  @returns {Tone.Param} this
         */
        setTargetAtTime(value: number, startTime: Time, timeConstant: number): Tone.Param;
        /**
         *  Sets an array of arbitrary parameter values starting at the given time
         *  for the given duration.
         *
         *  @param {Array} values
         *  @param {Time} startTime
         *  @param {Time} duration
         *  @param {NormalRange} [scaling=1] If the values in the curve should be scaled by some value
         *  @returns {Tone.Param} this
         */
        setValueCurveAtTime(values: Array, startTime: Time, duration: Time, scaling?: NormalRange): Tone.Param;
        /**
         *  Cancels all scheduled parameter changes with times greater than or
         *  equal to startTime.
         *
         *  @param  {Time} time
         *  @returns {Tone.Param} this
         */
        cancelScheduledValues(time: Time): Tone.Param;
        /**
         *  This is similar to [cancelScheduledValues](#cancelScheduledValues) except
         *  it holds the automated value at time until the next automated event.
         *  @param  {Time} time
         *  @returns {Tone.Param} this
         */
        cancelAndHoldAtTime(time: Time): Tone.Param;
        /**
         *  Ramps to the given value over the duration of the rampTime.
         *  Automatically selects the best ramp type (exponential or linear)
         *  depending on the `units` of the signal
         *
         *  @param  {number} value
         *  @param  {Time} rampTime 	The time that it takes the
         *                              value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //ramp to the value either linearly or exponentially
         * //depending on the "units" value of the signal
         * signal.rampTo(0, 10);
         *  @example
         * //schedule it to ramp starting at a specific time
         * signal.rampTo(0, 10, 5)
         */
        rampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Clean up
         *  @returns {Tone.Param} this
         */
        dispose(): Tone.Param;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class A Timeline class for scheduling and maintaining state
     *         along a timeline. All events must have a "time" property.
     *         Internally, events are stored in time order for fast
     *         retrieval.
     *  @extends {Tone}
     *  @param {Positive} [memory=Infinity] The number of previous events that are retained.
     */
    class Timeline extends Tone {
        constructor(memory?: Positive);
        /**
         *  The memory of the timeline, i.e.
         *  how many events in the past it will retain
         *  @type {Positive}
         */
        memory: Positive;
        /**
         *  the default parameters
         *  @static
         *  @const
         */
        static readonly defaults: any;
        /**
         *  The number of items in the timeline.
         *  @type {Number}
         *  @memberOf Tone.Timeline#
         *  @name length
         *  @readOnly
         */
        readonly length: number;
        /**
         *  Insert an event object onto the timeline. Events must have a "time" attribute.
         *  @param  {Object}  event  The event object to insert into the
         *                           timeline.
         *  @returns {Tone.Timeline} this
         */
        add(event: any): Tone.Timeline;
        /**
         *  Remove an event from the timeline.
         *  @param  {Object}  event  The event object to remove from the list.
         *  @returns {Tone.Timeline} this
         */
        remove(event: any): Tone.Timeline;
        /**
         *  Get the nearest event whose time is less than or equal to the given time.
         *  @param  {Number}  time  The time to query.
         *  @param  {String}  comparator Which value in the object to compare
         *  @returns {Object} The event object set after that time.
         */
        get(time: number, comparator: string): any;
        /**
         *  Return the first event in the timeline without removing it
         *  @returns {Object} The first event object
         */
        peek(): any;
        /**
         *  Return the first event in the timeline and remove it
         *  @returns {Object} The first event object
         */
        shift(): any;
        /**
         *  Get the event which is scheduled after the given time.
         *  @param  {Number}  time  The time to query.
         *  @param  {String}  comparator Which value in the object to compare
         *  @returns {Object} The event object after the given time
         */
        getAfter(time: number, comparator: string): any;
        /**
         *  Get the event before the event at the given time.
         *  @param  {Number}  time  The time to query.
         *  @param  {String}  comparator Which value in the object to compare
         *  @returns {Object} The event object before the given time
         */
        getBefore(time: number, comparator: string): any;
        /**
         *  Cancel events after the given time
         *  @param  {Number}  time  The time to query.
         *  @returns {Tone.Timeline} this
         */
        cancel(time: number): Tone.Timeline;
        /**
         *  Cancel events before or equal to the given time.
         *  @param  {Number}  time  The time to cancel before.
         *  @returns {Tone.Timeline} this
         */
        cancelBefore(time: number): Tone.Timeline;
        /**
         * Returns the previous event if there is one. null otherwise
         * @param  {Object} event The event to find the previous one of
         * @return {Object}       The event right before the given event
         */
        previousEvent(event: any): any;
        /**
         *  Iterate over everything in the array
         *  @param  {Function}  callback The callback to invoke with every item
         *  @returns {Tone.Timeline} this
         */
        forEach(callback: (...params: any[]) => any): Tone.Timeline;
        /**
         *  Iterate over everything in the array at or before the given time.
         *  @param  {Number}  time The time to check if items are before
         *  @param  {Function}  callback The callback to invoke with every item
         *  @returns {Tone.Timeline} this
         */
        forEachBefore(time: number, callback: (...params: any[]) => any): Tone.Timeline;
        /**
         *  Iterate over everything in the array after the given time.
         *  @param  {Number}  time The time to check if items are before
         *  @param  {Function}  callback The callback to invoke with every item
         *  @returns {Tone.Timeline} this
         */
        forEachAfter(time: number, callback: (...params: any[]) => any): Tone.Timeline;
        /**
         *  Iterate over everything in the array between the startTime and endTime.
         *  The timerange is inclusive of the startTime, but exclusive of the endTime.
         *  range = [startTime, endTime).
         *  @param  {Number}  startTime The time to check if items are before
         *  @param  {Number}  endTime The end of the test interval.
         *  @param  {Function}  callback The callback to invoke with every item
         *  @returns {Tone.Timeline} this
         */
        forEachBetween(startTime: number, endTime: number, callback: (...params: any[]) => any): Tone.Timeline;
        /**
         *  Iterate over everything in the array at or after the given time. Similar to
         *  forEachAfter, but includes the item(s) at the given time.
         *  @param  {Number}  time The time to check if items are before
         *  @param  {Function}  callback The callback to invoke with every item
         *  @returns {Tone.Timeline} this
         */
        forEachFrom(time: number, callback: (...params: any[]) => any): Tone.Timeline;
        /**
         *  Iterate over everything in the array at the given time
         *  @param  {Number}  time The time to check if items are before
         *  @param  {Function}  callback The callback to invoke with every item
         *  @returns {Tone.Timeline} this
         */
        forEachAtTime(time: number, callback: (...params: any[]) => any): Tone.Timeline;
        /**
         *  Clean up.
         *  @return  {Tone.Timeline}  this
         */
        dispose(): Tone.Timeline;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  A Timeline State. Provides the methods: <code>setStateAtTime("state", time)</code>
     *          and <code>getValueAtTime(time)</code>.
     *
     *  @extends {Tone.Timeline}
     *  @param {String} initial The initial state of the TimelineState.
     *                          Defaults to <code>undefined</code>
     */
    class TimelineState extends Tone.Timeline {
        constructor(initial: string);
        /**
         *  Returns the scheduled state scheduled before or at
         *  the given time.
         *  @param  {Number}  time  The time to query.
         *  @return  {String}  The name of the state input in setStateAtTime.
         */
        getValueAtTime(time: number): string;
        /**
         *  Add a state to the timeline.
         *  @param  {String}  state The name of the state to set.
         *  @param  {Number}  time  The time to query.
         *  @returns {Tone.TimelineState} this
         */
        setStateAtTime(state: string, time: number): Tone.TimelineState;
        /**
         *  Return the event before the time with the given state
         *  @param {Tone.State} state The state to look for
         *  @param  {Time}  time  When to check before
         *  @return  {Object}  The event with the given state before the time
         */
        getLastState(state: Tone.State, time: Time): any;
        /**
         *  Return the event after the time with the given state
         *  @param {Tone.State} state The state to look for
         *  @param  {Time}  time  When to check from
         *  @return  {Object}  The event with the given state after the time
         */
        getNextState(state: Tone.State, time: Time): any;
        /**
         *  The memory of the timeline, i.e.
         *  how many events in the past it will retain
         *  @type {Positive}
         */
        memory: Positive;
        /**
         *  The number of items in the timeline.
         *  @type {Number}
         *  @memberOf Tone.Timeline#
         *  @name length
         *  @readOnly
         */
        readonly length: number;
        /**
         *  Insert an event object onto the timeline. Events must have a "time" attribute.
         *  @param  {Object}  event  The event object to insert into the
         *                           timeline.
         *  @returns {Tone.Timeline} this
         */
        add(event: any): Tone.Timeline;
        /**
         *  Remove an event from the timeline.
         *  @param  {Object}  event  The event object to remove from the list.
         *  @returns {Tone.Timeline} this
         */
        remove(event: any): Tone.Timeline;
        /**
         *  Get the nearest event whose time is less than or equal to the given time.
         *  @param  {Number}  time  The time to query.
         *  @param  {String}  comparator Which value in the object to compare
         *  @returns {Object} The event object set after that time.
         */
        get(time: number, comparator: string): any;
        /**
         *  Return the first event in the timeline without removing it
         *  @returns {Object} The first event object
         */
        peek(): any;
        /**
         *  Return the first event in the timeline and remove it
         *  @returns {Object} The first event object
         */
        shift(): any;
        /**
         *  Get the event which is scheduled after the given time.
         *  @param  {Number}  time  The time to query.
         *  @param  {String}  comparator Which value in the object to compare
         *  @returns {Object} The event object after the given time
         */
        getAfter(time: number, comparator: string): any;
        /**
         *  Get the event before the event at the given time.
         *  @param  {Number}  time  The time to query.
         *  @param  {String}  comparator Which value in the object to compare
         *  @returns {Object} The event object before the given time
         */
        getBefore(time: number, comparator: string): any;
        /**
         *  Cancel events after the given time
         *  @param  {Number}  time  The time to query.
         *  @returns {Tone.Timeline} this
         */
        cancel(time: number): Tone.Timeline;
        /**
         *  Cancel events before or equal to the given time.
         *  @param  {Number}  time  The time to cancel before.
         *  @returns {Tone.Timeline} this
         */
        cancelBefore(time: number): Tone.Timeline;
        /**
         * Returns the previous event if there is one. null otherwise
         * @param  {Object} event The event to find the previous one of
         * @return {Object}       The event right before the given event
         */
        previousEvent(event: any): any;
        /**
         *  Iterate over everything in the array
         *  @param  {Function}  callback The callback to invoke with every item
         *  @returns {Tone.Timeline} this
         */
        forEach(callback: (...params: any[]) => any): Tone.Timeline;
        /**
         *  Iterate over everything in the array at or before the given time.
         *  @param  {Number}  time The time to check if items are before
         *  @param  {Function}  callback The callback to invoke with every item
         *  @returns {Tone.Timeline} this
         */
        forEachBefore(time: number, callback: (...params: any[]) => any): Tone.Timeline;
        /**
         *  Iterate over everything in the array after the given time.
         *  @param  {Number}  time The time to check if items are before
         *  @param  {Function}  callback The callback to invoke with every item
         *  @returns {Tone.Timeline} this
         */
        forEachAfter(time: number, callback: (...params: any[]) => any): Tone.Timeline;
        /**
         *  Iterate over everything in the array between the startTime and endTime.
         *  The timerange is inclusive of the startTime, but exclusive of the endTime.
         *  range = [startTime, endTime).
         *  @param  {Number}  startTime The time to check if items are before
         *  @param  {Number}  endTime The end of the test interval.
         *  @param  {Function}  callback The callback to invoke with every item
         *  @returns {Tone.Timeline} this
         */
        forEachBetween(startTime: number, endTime: number, callback: (...params: any[]) => any): Tone.Timeline;
        /**
         *  Iterate over everything in the array at or after the given time. Similar to
         *  forEachAfter, but includes the item(s) at the given time.
         *  @param  {Number}  time The time to check if items are before
         *  @param  {Function}  callback The callback to invoke with every item
         *  @returns {Tone.Timeline} this
         */
        forEachFrom(time: number, callback: (...params: any[]) => any): Tone.Timeline;
        /**
         *  Iterate over everything in the array at the given time
         *  @param  {Number}  time The time to check if items are before
         *  @param  {Function}  callback The callback to invoke with every item
         *  @returns {Tone.Timeline} this
         */
        forEachAtTime(time: number, callback: (...params: any[]) => any): Tone.Timeline;
        /**
         *  Clean up.
         *  @return  {Tone.Timeline}  this
         */
        dispose(): Tone.Timeline;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     * Possible play states.
     * @enum {String}
     */
    enum State {
        Started,
        Stopped,
        Paused
    }
    /**
     *  @class  Transport for timing musical events.
     *          Supports tempo curves and time changes. Unlike browser-based timing (setInterval, requestAnimationFrame)
     *          Tone.Transport timing events pass in the exact time of the scheduled event
     *          in the argument of the callback function. Pass that time value to the object
     *          you're scheduling. <br><br>
     *          A single transport is created for you when the library is initialized.
     *          <br><br>
     *          The transport emits the events: "start", "stop", "pause", and "loop" which are
     *          called with the time of that event as the argument.
     *
     *  @extends {Tone.Emitter}
     *  @singleton
     *  @example
     * //repeated event every 8th note
     * Tone.Transport.scheduleRepeat(function(time){
     * 	//do something with the time
     * }, "8n");
     *  @example
     * //schedule an event on the 16th measure
     * Tone.Transport.schedule(function(time){
     * 	//do something with the time
     * }, "16:0:0");
     */
    class Transport extends Tone.Emitter {
        /**
         * 	If the transport loops or not.
         *  @type {boolean}
         */
        loop: boolean;
        /**
         *  The Beats Per Minute of the Transport.
         *  @type {BPM}
         *  @signal
         *  @example
         * Tone.Transport.bpm.value = 80;
         * //ramp the bpm to 120 over 10 seconds
         * Tone.Transport.bpm.rampTo(120, 10);
         */
        bpm: BPM;
        /**
         *  the defaults
         *  @type {Object}
         *  @const
         *  @static
         */
        static readonly defaults: any;
        /**
         * Is an instanceof Tone.Transport
         * @type {Boolean}
         */
        isTransport: boolean;
        /**
         *  Schedule an event along the timeline.
         *  @param {Function} callback The callback to be invoked at the time.
         *  @param {TransportTime}  time The time to invoke the callback at.
         *  @return {Number} The id of the event which can be used for canceling the event.
         *  @example
         * //trigger the callback when the Transport reaches the desired time
         * Tone.Transport.schedule(function(time){
         * 	envelope.triggerAttack(time);
         * }, "128i");
         */
        schedule(callback: (...params: any[]) => any, time: TransportTime): number;
        /**
         *  Schedule a repeated event along the timeline. The event will fire
         *  at the `interval` starting at the `startTime` and for the specified
         *  `duration`.
         *  @param  {Function}  callback   The callback to invoke.
         *  @param  {Time}    interval   The duration between successive
         *                               callbacks. Must be a positive number.
         *  @param  {TransportTime=}    startTime  When along the timeline the events should
         *                               start being invoked.
         *  @param {Time} [duration=Infinity] How long the event should repeat.
         *  @return  {Number}    The ID of the scheduled event. Use this to cancel
         *                           the event.
         *  @example
         * //a callback invoked every eighth note after the first measure
         * Tone.Transport.scheduleRepeat(callback, "8n", "1m");
         */
        scheduleRepeat(callback: (...params: any[]) => any, interval: Time, startTime?: TransportTime, duration?: Time): number;
        /**
         *  Schedule an event that will be removed after it is invoked.
         *  @param {Function} callback The callback to invoke once.
         *  @param {TransportTime} time The time the callback should be invoked.
         *  @returns {Number} The ID of the scheduled event.
         */
        scheduleOnce(callback: (...params: any[]) => any, time: TransportTime): number;
        /**
         *  Clear the passed in event id from the timeline
         *  @param {Number} eventId The id of the event.
         *  @returns {Tone.Transport} this
         */
        clear(eventId: number): Tone.Transport;
        /**
         *  Remove scheduled events from the timeline after
         *  the given time. Repeated events will be removed
         *  if their startTime is after the given time
         *  @param {TransportTime} [after=0] Clear all events after
         *                          this time.
         *  @returns {Tone.Transport} this
         */
        cancel(after?: TransportTime): Tone.Transport;
        /**
         *  Returns the playback state of the source, either "started", "stopped", or "paused"
         *  @type {Tone.State}
         *  @readOnly
         *  @memberOf Tone.Transport#
         *  @name state
         */
        readonly state: Tone.State;
        /**
         *  Start the transport and all sources synced to the transport.
         *  @param  {Time} [time=now] The time when the transport should start.
         *  @param  {TransportTime=} offset The timeline offset to start the transport.
         *  @returns {Tone.Transport} this
         *  @example
         * //start the transport in one second starting at beginning of the 5th measure.
         * Tone.Transport.start("+1", "4:0:0");
         */
        start(time?: Time, offset?: TransportTime): Tone.Transport;
        /**
         *  Stop the transport and all sources synced to the transport.
         *  @param  {Time} [time=now] The time when the transport should stop.
         *  @returns {Tone.Transport} this
         *  @example
         * Tone.Transport.stop();
         */
        stop(time?: Time): Tone.Transport;
        /**
         *  Pause the transport and all sources synced to the transport.
         *  @param  {Time} [time=now]
         *  @returns {Tone.Transport} this
         */
        pause(time?: Time): Tone.Transport;
        /**
         * Toggle the current state of the transport. If it is
         * started, it will stop it, otherwise it will start the Transport.
         * @param  {Time=} time The time of the event
         * @return {Tone.Transport}      this
         */
        toggle(time?: Time): Tone.Transport;
        /**
         *  The time signature as just the numerator over 4.
         *  For example 4/4 would be just 4 and 6/8 would be 3.
         *  @memberOf Tone.Transport#
         *  @type {Number|Array}
         *  @name timeSignature
         *  @example
         * //common time
         * Tone.Transport.timeSignature = 4;
         * // 7/8
         * Tone.Transport.timeSignature = [7, 8];
         * //this will be reduced to a single number
         * Tone.Transport.timeSignature; //returns 3.5
         */
        timeSignature: number | Array;
        /**
         * When the Tone.Transport.loop = true, this is the starting position of the loop.
         * @memberOf Tone.Transport#
         * @type {Time}
         * @name loopStart
         */
        loopStart: Time;
        /**
         * When the Tone.Transport.loop = true, this is the ending position of the loop.
         * @memberOf Tone.Transport#
         * @type {Time}
         * @name loopEnd
         */
        loopEnd: Time;
        /**
         *  Set the loop start and stop at the same time.
         *  @param {TransportTime} startPosition
         *  @param {TransportTime} endPosition
         *  @returns {Tone.Transport} this
         *  @example
         * //loop over the first measure
         * Tone.Transport.setLoopPoints(0, "1m");
         * Tone.Transport.loop = true;
         */
        setLoopPoints(startPosition: TransportTime, endPosition: TransportTime): Tone.Transport;
        /**
         *  The swing value. Between 0-1 where 1 equal to
         *  the note + half the subdivision.
         *  @memberOf Tone.Transport#
         *  @type {NormalRange}
         *  @name swing
         */
        swing: NormalRange;
        /**
         *  Set the subdivision which the swing will be applied to.
         *  The default value is an 8th note. Value must be less
         *  than a quarter note.
         *
         *  @memberOf Tone.Transport#
         *  @type {Time}
         *  @name swingSubdivision
         */
        swingSubdivision: Time;
        /**
         *  The Transport's position in Bars:Beats:Sixteenths.
         *  Setting the value will jump to that position right away.
         *  @memberOf Tone.Transport#
         *  @type {BarsBeatsSixteenths}
         *  @name position
         */
        position: BarsBeatsSixteenths;
        /**
         *  The Transport's position in seconds
         *  Setting the value will jump to that position right away.
         *  @memberOf Tone.Transport#
         *  @type {Seconds}
         *  @name seconds
         */
        seconds: Seconds;
        /**
         *  The Transport's loop position as a normalized value. Always
         *  returns 0 if the transport if loop is not true.
         *  @memberOf Tone.Transport#
         *  @name progress
         *  @type {NormalRange}
         */
        progress: NormalRange;
        /**
         *  The transports current tick position.
         *
         *  @memberOf Tone.Transport#
         *  @type {Ticks}
         *  @name ticks
         */
        ticks: Ticks;
        /**
         * Get the clock's ticks at the given time.
         * @param  {Time} time  When to get the tick value
         * @return {Ticks}       The tick value at the given time.
         */
        getTicksAtTime(time: Time): Ticks;
        /**
         *  Return the elapsed seconds at the given time.
         *  @param  {Time}  time  When to get the elapsed seconds
         *  @return  {Seconds}  The number of elapsed seconds
         */
        getSecondsAtTime(time: Time): Seconds;
        /**
         *  Pulses Per Quarter note. This is the smallest resolution
         *  the Transport timing supports. This should be set once
         *  on initialization and not set again. Changing this value
         *  after other objects have been created can cause problems.
         *
         *  @memberOf Tone.Transport#
         *  @type {Number}
         *  @name PPQ
         */
        PPQ: number;
        /**
         *  Returns the time aligned to the next subdivision
         *  of the Transport. If the Transport is not started,
         *  it will return 0.
         *  Note: this will not work precisely during tempo ramps.
         *  @param  {Time}  subdivision  The subdivision to quantize to
         *  @return  {Number}  The context time of the next subdivision.
         *  @example
         * Tone.Transport.start(); //the transport must be started
         * Tone.Transport.nextSubdivision("4n");
         */
        nextSubdivision(subdivision: Time): number;
        /**
         *  Attaches the signal to the tempo control signal so that
         *  any changes in the tempo will change the signal in the same
         *  ratio.
         *
         *  @param  {Tone.Signal} signal
         *  @param {number=} ratio Optionally pass in the ratio between
         *                         the two signals. Otherwise it will be computed
         *                         based on their current values.
         *  @returns {Tone.Transport} this
         */
        syncSignal(signal: Tone.Signal, ratio?: number): Tone.Transport;
        /**
         *  Unsyncs a previously synced signal from the transport's control.
         *  See Tone.Transport.syncSignal.
         *  @param  {Tone.Signal} signal
         *  @returns {Tone.Transport} this
         */
        unsyncSignal(signal: Tone.Signal): Tone.Transport;
        /**
         *  Bind a callback to a specific event.
         *  @param  {String}    event     The name of the event to listen for.
         *  @param  {Function}  callback  The callback to invoke when the
         *                                event is emitted
         *  @return  {Tone.Emitter}    this
         */
        on(event: string, callback: (...params: any[]) => any): Tone.Emitter;
        /**
         *  Bind a callback which is only invoked once
         *  @param  {String}    event     The name of the event to listen for.
         *  @param  {Function}  callback  The callback to invoke when the
         *                                event is emitted
         *  @return  {Tone.Emitter}    this
         */
        once(event: string, callback: (...params: any[]) => any): Tone.Emitter;
        /**
         *  Remove the event listener.
         *  @param  {String}    event     The event to stop listening to.
         *  @param  {Function=}  callback  The callback which was bound to
         *                                the event with Tone.Emitter.on.
         *                                If no callback is given, all callbacks
         *                                events are removed.
         *  @return  {Tone.Emitter}    this
         */
        off(event: string, callback?: (...params: any[]) => any): Tone.Emitter;
        /**
         *  Invoke all of the callbacks bound to the event
         *  with any arguments passed in.
         *  @param  {String}  event  The name of the event.
         *  @param {...*} args The arguments to pass to the functions listening.
         *  @return  {Tone.Emitter}  this
         */
        emit(event: string, ...args: any[]): Tone.Emitter;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.TransportEvent is an internal class used by (Tone.Transport)[Transport]
     *         to schedule events. Do no invoke this class directly, it is
     *         handled from within Tone.Transport.
     *  @extends {Tone}
     *  @param {Object} options
     */
    class TransportEvent extends Tone {
        constructor(options: any);
        /**
         * Reference to the Transport that created it
         * @type {Tone.Transport}
         */
        Transport: Tone.Transport;
        /**
         * The unique id of the event
         * @type {Number}
         */
        id: number;
        /**
         * The time the event starts
         * @type {Ticks}
         */
        time: Ticks;
        /**
         * The callback to invoke
         * @type {Function}
         */
        callback: (...params: any[]) => any;
        /**
         * The defaults
         * @static
         * @type {Object}
         */
        static defaults: any;
        /**
         * Invoke the event callback.
         * @param  {Time} time  The AudioContext time in seconds of the event
         */
        invoke(time: Time): void;
        /**
         * Clean up
         * @return {Tone.TransportEvent} this
         */
        dispose(): Tone.TransportEvent;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.TransportRepeatEvent is an internal class used by Tone.Transport
     *         to schedule repeat events. This class should not be instantiated directly.
     *  @extends {Tone.TransportEvent}
     *  @param {Object} options
     */
    class TransportRepeatEvent extends Tone.TransportEvent {
        constructor(options: any);
        /**
         * The defaults
         * @static
         * @type {Object}
         */
        static defaults: any;
        /**
         * Invoke the callback. Returns the tick time which
         * the next event should be scheduled at.
         * @param  {Number} time  The AudioContext time in seconds of the event
         */
        invoke(time: number): void;
        /**
         * Clean up
         * @return {Tone.TransportRepeatEvent} this
         */
        dispose(): Tone.TransportRepeatEvent;
        /**
         * Reference to the Transport that created it
         * @type {Tone.Transport}
         */
        Transport: Tone.Transport;
        /**
         * The unique id of the event
         * @type {Number}
         */
        id: number;
        /**
         * The time the event starts
         * @type {Ticks}
         */
        time: Ticks;
        /**
         * The callback to invoke
         * @type {Function}
         */
        callback: (...params: any[]) => any;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.AutoFilter is a Tone.Filter with a Tone.LFO connected to the filter cutoff frequency.
     *         Setting the LFO rate and depth allows for control over the filter modulation rate
     *         and depth.
     *
     *  @constructor
     *  @extends {Tone.Effect}
     *  @param {Time|Object} [frequency] The rate of the LFO.
     *  @param {Frequency=} baseFrequency The lower value of the LFOs oscillation
     *  @param {Frequency=} octaves The number of octaves above the baseFrequency
     *  @example
     * //create an autofilter and start it's LFO
     * var autoFilter = new Tone.AutoFilter("4n").toMaster().start();
     * //route an oscillator through the filter and start it
     * var oscillator = new Tone.Oscillator().connect(autoFilter).start();
     */
    class AutoFilter extends Tone.Effect {
        constructor(frequency?: Time | any, baseFrequency?: Frequency, octaves?: Frequency);
        /**
         * The range of the filter modulating between the min and max frequency.
         * 0 = no modulation. 1 = full modulation.
         * @type {NormalRange}
         * @signal
         */
        depth: NormalRange;
        /**
         * How fast the filter modulates between min and max.
         * @type {Frequency}
         * @signal
         */
        frequency: Frequency;
        /**
         *  The filter node
         *  @type {Tone.Filter}
         */
        filter: Tone.Filter;
        /**
         *  defaults
         *  @static
         *  @type {Object}
         */
        static defaults: any;
        /**
         * Start the effect.
         * @param {Time} [time=now] When the LFO will start.
         * @returns {Tone.AutoFilter} this
         */
        start(time?: Time): Tone.AutoFilter;
        /**
         * Stop the effect.
         * @param {Time} [time=now] When the LFO will stop.
         * @returns {Tone.AutoFilter} this
         */
        stop(time?: Time): Tone.AutoFilter;
        /**
         * Sync the filter to the transport.
         * @param {Time} [delay=0] Delay time before starting the effect after the
         *                               Transport has started.
         * @returns {Tone.AutoFilter} this
         */
        sync(delay?: Time): Tone.AutoFilter;
        /**
         * Unsync the filter from the transport.
         * @returns {Tone.AutoFilter} this
         */
        unsync(): Tone.AutoFilter;
        /**
         * Type of oscillator attached to the AutoFilter.
         * Possible values: "sine", "square", "triangle", "sawtooth".
         * @memberOf Tone.AutoFilter#
         * @type {string}
         * @name type
         */
        type: string;
        /**
         * The minimum value of the filter's cutoff frequency.
         * @memberOf Tone.AutoFilter#
         * @type {Frequency}
         * @name baseFrequency
         */
        baseFrequency: Frequency;
        /**
         * The maximum value of the filter's cutoff frequency.
         * @memberOf Tone.AutoFilter#
         * @type {Positive}
         * @name octaves
         */
        octaves: Positive;
        /**
         *  Clean up.
         *  @returns {Tone.AutoFilter} this
         */
        dispose(): Tone.AutoFilter;
        /**
         * Type of oscillator attached to the AutoFilter.
         * Possible values: "sine", "square", "triangle", "sawtooth".
         * @memberOf Tone.AutoFilter#
         * @type {string}
         * @name type
         */
        type: string;
        /**
         *  The wet control is how much of the effected
         *  will pass through to the output. 1 = 100% effected
         *  signal, 0 = 100% dry signal.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.AutoPanner is a Tone.Panner with an LFO connected to the pan amount.
     *         More on using autopanners [here](https://www.ableton.com/en/blog/autopan-chopper-effect-and-more-liveschool/).
     *
     *  @constructor
     *  @extends {Tone.Effect}
     *  @param {Frequency|Object} [frequency] Rate of left-right oscillation.
     *  @example
     * //create an autopanner and start it's LFO
     * var autoPanner = new Tone.AutoPanner("4n").toMaster().start();
     * //route an oscillator through the panner and start it
     * var oscillator = new Tone.Oscillator().connect(autoPanner).start();
     */
    class AutoPanner extends Tone.Effect {
        constructor(frequency?: Frequency | any);
        /**
         * The amount of panning between left and right.
         * 0 = always center. 1 = full range between left and right.
         * @type {NormalRange}
         * @signal
         */
        depth: NormalRange;
        /**
         * How fast the panner modulates between left and right.
         * @type {Frequency}
         * @signal
         */
        frequency: Frequency;
        /**
         *  defaults
         *  @static
         *  @type {Object}
         */
        static defaults: any;
        /**
         * Start the effect.
         * @param {Time} [time=now] When the LFO will start.
         * @returns {Tone.AutoPanner} this
         */
        start(time?: Time): Tone.AutoPanner;
        /**
         * Stop the effect.
         * @param {Time} [time=now] When the LFO will stop.
         * @returns {Tone.AutoPanner} this
         */
        stop(time?: Time): Tone.AutoPanner;
        /**
         * Sync the panner to the transport.
         * @param {Time} [delay=0] Delay time before starting the effect after the
         *                               Transport has started.
         * @returns {Tone.AutoPanner} this
         */
        sync(delay?: Time): Tone.AutoPanner;
        /**
         * Unsync the panner from the transport
         * @returns {Tone.AutoPanner} this
         */
        unsync(): Tone.AutoPanner;
        /**
         *  clean up
         *  @returns {Tone.AutoPanner} this
         */
        dispose(): Tone.AutoPanner;
        /**
         *  The wet control is how much of the effected
         *  will pass through to the output. 1 = 100% effected
         *  signal, 0 = 100% dry signal.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Tone.AutoWah connects a Tone.Follower to a bandpass filter (Tone.Filter).
     *          The frequency of the filter is adjusted proportionally to the
     *          incoming signal's amplitude. Inspiration from [Tuna.js](https://github.com/Dinahmoe/tuna).
     *
     *  @constructor
     *  @extends {Tone.Effect}
     *  @param {Frequency|Object} [baseFrequency] The frequency the filter is set
     *                                            to at the low point of the wah
     *  @param {Positive} [octaves] The number of octaves above the baseFrequency
     *                                the filter will sweep to when fully open
     *  @param {Decibels} [sensitivity] The decibel threshold sensitivity for
     *                                   the incoming signal. Normal range of -40 to 0.
     *  @example
     * var autoWah = new Tone.AutoWah(50, 6, -30).toMaster();
     * //initialize the synth and connect to autowah
     * var synth = new Synth.connect(autoWah);
     * //Q value influences the effect of the wah - default is 2
     * autoWah.Q.value = 6;
     * //more audible on higher notes
     * synth.triggerAttackRelease("C4", "8n")
     */
    class AutoWah extends Tone.Effect {
        constructor(baseFrequency?: Frequency | any, octaves?: Positive, sensitivity?: Decibels);
        /**
         * The gain of the filter.
         * @type {Number}
         * @signal
         */
        gain: number;
        /**
         * The quality of the filter.
         * @type {Positive}
         * @signal
         */
        Q: Positive;
        /**
         *  @static
         *  @type {Object}
         */
        static defaults: any;
        /**
         * The number of octaves that the filter will sweep above the
         * baseFrequency.
         * @memberOf Tone.AutoWah#
         * @type {Number}
         * @name octaves
         */
        octaves: number;
        /**
         * The base frequency from which the sweep will start from.
         * @memberOf Tone.AutoWah#
         * @type {Frequency}
         * @name baseFrequency
         */
        baseFrequency: Frequency;
        /**
         * The sensitivity to control how responsive to the input signal the filter is.
         * @memberOf Tone.AutoWah#
         * @type {Decibels}
         * @name sensitivity
         */
        sensitivity: Decibels;
        /**
         *  Clean up.
         *  @returns {Tone.AutoWah} this
         */
        dispose(): Tone.AutoWah;
        /**
         *  The wet control is how much of the effected
         *  will pass through to the output. 1 = 100% effected
         *  signal, 0 = 100% dry signal.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.Bitcrusher downsamples the incoming signal to a different bitdepth.
     *         Lowering the bitdepth of the signal creates distortion. Read more about Bitcrushing
     *         on [Wikipedia](https://en.wikipedia.org/wiki/Bitcrusher).
     *
     *  @constructor
     *  @extends {Tone.Effect}
     *  @param {Number} bits The number of bits to downsample the signal. Nominal range
     *                       of 1 to 8.
     *  @example
     * //initialize crusher and route a synth through it
     * var crusher = new Tone.BitCrusher(4).toMaster();
     * var synth = new Tone.MonoSynth().connect(crusher);
     */
    class BitCrusher extends Tone.Effect {
        constructor(bits: number);
        /**
         *  the default values
         *  @static
         *  @type {Object}
         */
        static defaults: any;
        /**
         * The bit depth of the effect. Nominal range of 1-8.
         * @memberOf Tone.BitCrusher#
         * @type {number}
         * @name bits
         */
        bits: number;
        /**
         *  Clean up.
         *  @returns {Tone.BitCrusher} this
         */
        dispose(): Tone.BitCrusher;
        /**
         *  The wet control is how much of the effected
         *  will pass through to the output. 1 = 100% effected
         *  signal, 0 = 100% dry signal.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.ChebyShev is a Chebyshev waveshaper, an effect which is good
     *         for making different types of distortion sounds.
     *         Note that odd orders sound very different from even ones,
     *         and order = 1 is no change.
     *         Read more at [music.columbia.edu](http://music.columbia.edu/cmc/musicandcomputers/chapter4/04_06.php).
     *
     *  @extends {Tone.Effect}
     *  @constructor
     *  @param {Positive|Object} [order] The order of the chebyshev polynomial. Normal range between 1-100.
     *  @example
     * //create a new cheby
     * var cheby = new Tone.Chebyshev(50);
     * //create a monosynth connected to our cheby
     * synth = new Tone.MonoSynth().connect(cheby);
     */
    class Chebyshev extends Tone.Effect {
        constructor(order?: Positive | any);
        /**
         *  @static
         *  @const
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         * The order of the Chebyshev polynomial which creates
         * the equation which is applied to the incoming
         * signal through a Tone.WaveShaper. The equations
         * are in the form:<br>
         * order 2: 2x^2 + 1<br>
         * order 3: 4x^3 + 3x <br>
         * @memberOf Tone.Chebyshev#
         * @type {Positive}
         * @name order
         */
        order: Positive;
        /**
         * The oversampling of the effect. Can either be "none", "2x" or "4x".
         * @memberOf Tone.Chebyshev#
         * @type {string}
         * @name oversample
         */
        oversample: string;
        /**
         *  Clean up.
         *  @returns {Tone.Chebyshev} this
         */
        dispose(): Tone.Chebyshev;
        /**
         *  The wet control is how much of the effected
         *  will pass through to the output. 1 = 100% effected
         *  signal, 0 = 100% dry signal.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.Chorus is a stereo chorus effect composed of
     *         a left and right delay with a Tone.LFO applied to the delayTime of each channel.
     *         Inspiration from [Tuna.js](https://github.com/Dinahmoe/tuna/blob/master/tuna.js).
     *         Read more on the chorus effect on [SoundOnSound](http://www.soundonsound.com/sos/jun04/articles/synthsecrets.htm).
     *
     *	@constructor
     *	@extends {Tone.StereoEffect}
     *	@param {Frequency|Object} [frequency] The frequency of the LFO.
     *	@param {Milliseconds} [delayTime] The delay of the chorus effect in ms.
     *	@param {NormalRange} [depth] The depth of the chorus.
     *	@example
     * var chorus = new Tone.Chorus(4, 2.5, 0.5);
     * var synth = new Tone.PolySynth(4, Tone.MonoSynth).connect(chorus);
     * synth.triggerAttackRelease(["C3","E3","G3"], "8n");
     */
    class Chorus extends Tone.StereoEffect {
        constructor(frequency?: Frequency | any, delayTime?: Milliseconds, depth?: NormalRange);
        /**
         * The frequency of the LFO which modulates the delayTime.
         * @type {Frequency}
         * @signal
         */
        frequency: Frequency;
        /**
         *  @static
         *  @type {Object}
         */
        static defaults: any;
        /**
         * The depth of the effect. A depth of 1 makes the delayTime
         * modulate between 0 and 2*delayTime (centered around the delayTime).
         * @memberOf Tone.Chorus#
         * @type {NormalRange}
         * @name depth
         */
        depth: NormalRange;
        /**
         * The delayTime in milliseconds of the chorus. A larger delayTime
         * will give a more pronounced effect. Nominal range a delayTime
         * is between 2 and 20ms.
         * @memberOf Tone.Chorus#
         * @type {Milliseconds}
         * @name delayTime
         */
        delayTime: Milliseconds;
        /**
         * The oscillator type of the LFO.
         * @memberOf Tone.Chorus#
         * @type {string}
         * @name type
         */
        type: string;
        /**
         * Amount of stereo spread. When set to 0, both LFO's will be panned centrally.
         * When set to 180, LFO's will be panned hard left and right respectively.
         * @memberOf Tone.Chorus#
         * @type {Degrees}
         * @name spread
         */
        spread: Degrees;
        /**
         *  Clean up.
         *  @returns {Tone.Chorus} this
         */
        dispose(): Tone.Chorus;
        /**
         *  The wet control, i.e. how much of the effected
         *  will pass through to the output.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Tone.Convolver is a wrapper around the Native Web Audio
     *          [ConvolverNode](http://webaudio.github.io/web-audio-api/#the-convolvernode-interface).
     *          Convolution is useful for reverb and filter emulation. Read more about convolution reverb on
     *          [Wikipedia](https://en.wikipedia.org/wiki/Convolution_reverb).
     *
     *  @constructor
     *  @extends {Tone.Effect}
     *  @param {string|Tone.Buffer|Object} [url] The URL of the impulse response or the Tone.Buffer
     *                                           contianing the impulse response.
     *  @param {Function=} onload The callback to invoke when the url is loaded.
     *  @example
     * //initializing the convolver with an impulse response
     * var convolver = new Tone.Convolver("./path/to/ir.wav").toMaster();
     */
    class Convolver extends Tone.Effect {
        constructor(url?: string | Tone.Buffer | any, onload?: (...params: any[]) => any);
        /**
         *  @static
         *  @const
         *  @type  {Object}
         */
        static readonly defaults: any;
        /**
         *  The convolver's buffer
         *  @memberOf Tone.Convolver#
         *  @type {AudioBuffer}
         *  @name buffer
         */
        buffer: AudioBuffer;
        /**
         *  The normalize property of the ConvolverNode interface is a boolean that controls whether the impulse response from the buffer will be scaled by an equal-power normalization when the buffer attribute is set, or not.
         *  @memberOf Tone.Convolver#
         *  @type {Boolean}
         *  @name normalize
         */
        normalize: boolean;
        /**
         *  Load an impulse response url as an audio buffer.
         *  Decodes the audio asynchronously and invokes
         *  the callback once the audio buffer loads.
         *  @param {string} url The url of the buffer to load.
         *                      filetype support depends on the
         *                      browser.
         *  @param  {function=} callback
         *  @returns {Promise}
         */
        load(url: string, callback?: (...params: any[]) => any): Promise;
        /**
         *  Clean up.
         *  @returns {Tone.Convolver} this
         */
        dispose(): Tone.Convolver;
        /**
         *  The wet control is how much of the effected
         *  will pass through to the output. 1 = 100% effected
         *  signal, 0 = 100% dry signal.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.Distortion is a simple distortion effect using Tone.WaveShaper.
     *         Algorithm from [a stackoverflow answer](http://stackoverflow.com/a/22313408).
     *
     *  @extends {Tone.Effect}
     *  @constructor
     *  @param {Number|Object} [distortion] The amount of distortion (nominal range of 0-1)
     *  @example
     * var dist = new Tone.Distortion(0.8).toMaster();
     * var fm = new Tone.SimpleFM().connect(dist);
     * //this sounds good on bass notes
     * fm.triggerAttackRelease("A1", "8n");
     */
    class Distortion extends Tone.Effect {
        constructor(distortion?: number | any);
        /**
         *  @static
         *  @const
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         * The amount of distortion.
         * @memberOf Tone.Distortion#
         * @type {NormalRange}
         * @name distortion
         */
        distortion: NormalRange;
        /**
         * The oversampling of the effect. Can either be "none", "2x" or "4x".
         * @memberOf Tone.Distortion#
         * @type {string}
         * @name oversample
         */
        oversample: string;
        /**
         *  Clean up.
         *  @returns {Tone.Distortion} this
         */
        dispose(): Tone.Distortion;
        /**
         *  The wet control is how much of the effected
         *  will pass through to the output. 1 = 100% effected
         *  signal, 0 = 100% dry signal.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     * 	@class  Tone.Effect is the base class for effects. Connect the effect between
     * 	        the effectSend and effectReturn GainNodes, then control the amount of
     * 	        effect which goes to the output using the wet control.
     *
     *  @constructor
     *  @extends {Tone.AudioNode}
     *  @param {NormalRange|Object} [wet] The starting wet value.
     */
    class Effect extends Tone.AudioNode {
        constructor(wet?: NormalRange | any);
        /**
         *  The wet control is how much of the effected
         *  will pass through to the output. 1 = 100% effected
         *  signal, 0 = 100% dry signal.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         *  @static
         *  @type {Object}
         */
        static defaults: any;
        /**
         *  Clean up.
         *  @returns {Tone.Effect} this
         */
        dispose(): Tone.Effect;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Tone.FeedbackDelay is a DelayNode in which part of output
     *          signal is fed back into the delay.
     *
     *  @constructor
     *  @extends {Tone.FeedbackEffect}
     *  @param {Time|Object} [delayTime] The delay applied to the incoming signal.
     *  @param {NormalRange=} feedback The amount of the effected signal which
     *                            is fed back through the delay.
     *  @example
     * var feedbackDelay = new Tone.FeedbackDelay("8n", 0.5).toMaster();
     * var tom = new Tone.MembraneSynth({
     * 	"octaves" : 4,
     * 	"pitchDecay" : 0.1
     * }).connect(feedbackDelay);
     * tom.triggerAttackRelease("A2","32n");
     */
    class FeedbackDelay extends Tone.FeedbackEffect {
        constructor(delayTime?: Time | any, feedback?: NormalRange);
        /**
         *  The delayTime of the DelayNode.
         *  @type {Time}
         *  @signal
         */
        delayTime: Time;
        /**
         *  The default values.
         *  @const
         *  @static
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         *  clean up
         *  @returns {Tone.FeedbackDelay} this
         */
        dispose(): Tone.FeedbackDelay;
        /**
         *  The amount of signal which is fed back into the effect input.
         *  @type {NormalRange}
         *  @signal
         */
        feedback: NormalRange;
        /**
         *  The wet control is how much of the effected
         *  will pass through to the output. 1 = 100% effected
         *  signal, 0 = 100% dry signal.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     * 	@class  Tone.FeedbackEffect provides a loop between an
     * 	        audio source and its own output. This is a base-class
     * 	        for feedback effects.
     *
     *  @constructor
     *  @extends {Tone.Effect}
     *  @param {NormalRange|Object} [feedback] The initial feedback value.
     */
    class FeedbackEffect extends Tone.Effect {
        constructor(feedback?: NormalRange | any);
        /**
         *  The amount of signal which is fed back into the effect input.
         *  @type {NormalRange}
         *  @signal
         */
        feedback: NormalRange;
        /**
         *  @static
         *  @type {Object}
         */
        static defaults: any;
        /**
         *  Clean up.
         *  @returns {Tone.FeedbackEffect} this
         */
        dispose(): Tone.FeedbackEffect;
        /**
         *  The wet control is how much of the effected
         *  will pass through to the output. 1 = 100% effected
         *  signal, 0 = 100% dry signal.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.Freeverb is a reverb based on [Freeverb](https://ccrma.stanford.edu/~jos/pasp/Freeverb.html).
     *         Read more on reverb on [Sound On Sound](https://web.archive.org/web/20160404083902/http://www.soundonsound.com:80/sos/feb01/articles/synthsecrets.asp).
     *
     *  @extends {Tone.Effect}
     *  @constructor
     *  @param {NormalRange|Object} [roomSize] Correlated to the decay time.
     *  @param {Frequency} [dampening] The cutoff frequency of a lowpass filter as part
     *                                 of the reverb.
     *  @example
     * var freeverb = new Tone.Freeverb().toMaster();
     * freeverb.dampening.value = 1000;
     * //routing synth through the reverb
     * var synth = new Tone.AMSynth().connect(freeverb);
     */
    class Freeverb extends Tone.Effect {
        constructor(roomSize?: NormalRange | any, dampening?: Frequency);
        /**
         *  The roomSize value between. A larger roomSize
         *  will result in a longer decay.
         *  @type {NormalRange}
         *  @signal
         */
        roomSize: NormalRange;
        /**
         *  The amount of dampening of the reverberant signal.
         *  @type {Frequency}
         *  @signal
         */
        dampening: Frequency;
        /**
         *  @static
         *  @type {Object}
         */
        static defaults: any;
        /**
         *  Clean up.
         *  @returns {Tone.Freeverb} this
         */
        dispose(): Tone.Freeverb;
        /**
         *  The wet control is how much of the effected
         *  will pass through to the output. 1 = 100% effected
         *  signal, 0 = 100% dry signal.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.JCReverb is a simple [Schroeder Reverberator](https://ccrma.stanford.edu/~jos/pasp/Schroeder_Reverberators.html)
     *         tuned by John Chowning in 1970.
     *         It is made up of three allpass filters and four Tone.FeedbackCombFilter.
     *
     *
     *  @extends {Tone.Effect}
     *  @constructor
     *  @param {NormalRange|Object} [roomSize] Coorelates to the decay time.
     *  @example
     * var reverb = new Tone.JCReverb(0.4).connect(Tone.Master);
     * var delay = new Tone.FeedbackDelay(0.5);
     * //connecting the synth to reverb through delay
     * var synth = new Tone.DuoSynth().chain(delay, reverb);
     * synth.triggerAttackRelease("A4","8n");
     */
    class JCReverb extends Tone.Effect {
        constructor(roomSize?: NormalRange | any);
        /**
         *  room size control values between [0,1]
         *  @type {NormalRange}
         *  @signal
         */
        roomSize: NormalRange;
        /**
         *  the default values
         *  @static
         *  @const
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         *  Clean up.
         *  @returns {Tone.JCReverb} this
         */
        dispose(): Tone.JCReverb;
        /**
         *  The wet control is how much of the effected
         *  will pass through to the output. 1 = 100% effected
         *  signal, 0 = 100% dry signal.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Mid/Side processing separates the the 'mid' signal
     *         (which comes out of both the left and the right channel)
     *         and the 'side' (which only comes out of the the side channels)
     *         and effects them separately before being recombined.
     *         Applies a Mid/Side seperation and recombination.
     *         Algorithm found in [kvraudio forums](http://www.kvraudio.com/forum/viewtopic.php?t=212587).
     *         <br><br>
     *         This is a base-class for Mid/Side Effects.
     *
     *  @extends {Tone.Effect}
     *  @constructor
     */
    class MidSideEffect extends Tone.Effect {
        /**
         *  Clean up.
         *  @returns {Tone.MidSideEffect} this
         */
        dispose(): Tone.MidSideEffect;
        /**
         *  The wet control is how much of the effected
         *  will pass through to the output. 1 = 100% effected
         *  signal, 0 = 100% dry signal.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.Phaser is a phaser effect. Phasers work by changing the phase
     *         of different frequency components of an incoming signal. Read more on
     *         [Wikipedia](https://en.wikipedia.org/wiki/Phaser_(effect)).
     *         Inspiration for this phaser comes from [Tuna.js](https://github.com/Dinahmoe/tuna/).
     *
     *	@extends {Tone.StereoEffect}
     *	@constructor
     *	@param {Frequency|Object} [frequency] The speed of the phasing.
     *	@param {number} [octaves] The octaves of the effect.
     *	@param {Frequency} [baseFrequency] The base frequency of the filters.
     *	@example
     * var phaser = new Tone.Phaser({
     * 	"frequency" : 15,
     * 	"octaves" : 5,
     * 	"baseFrequency" : 1000
     * }).toMaster();
     * var synth = new Tone.FMSynth().connect(phaser);
     * synth.triggerAttackRelease("E3", "2n");
     */
    class Phaser extends Tone.StereoEffect {
        constructor(frequency?: Frequency | any, octaves?: number, baseFrequency?: Frequency);
        /**
         *  The quality factor of the filters
         *  @type {Positive}
         *  @signal
         */
        Q: Positive;
        /**
         * the frequency of the effect
         * @type {Tone.Signal}
         */
        frequency: Tone.Signal;
        /**
         *  defaults
         *  @static
         *  @type {object}
         */
        static defaults: any;
        /**
         * The number of octaves the phase goes above
         * the baseFrequency
         * @memberOf Tone.Phaser#
         * @type {Positive}
         * @name octaves
         */
        octaves: Positive;
        /**
         * The the base frequency of the filters.
         * @memberOf Tone.Phaser#
         * @type {number}
         * @name baseFrequency
         */
        baseFrequency: number;
        /**
         *  clean up
         *  @returns {Tone.Phaser} this
         */
        dispose(): Tone.Phaser;
        /**
         *  The wet control, i.e. how much of the effected
         *  will pass through to the output.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Tone.PingPongDelay is a feedback delay effect where the echo is heard
     *          first in one channel and next in the opposite channel. In a stereo
     *          system these are the right and left channels.
     *          PingPongDelay in more simplified terms is two Tone.FeedbackDelays
     *          with independent delay values. Each delay is routed to one channel
     *          (left or right), and the channel triggered second will always
     *          trigger at the same interval after the first.
     *
     * 	@constructor
     * 	@extends {Tone.StereoXFeedbackEffect}
     *  @param {Time|Object} [delayTime] The delayTime between consecutive echos.
     *  @param {NormalRange=} feedback The amount of the effected signal which
     *                                 is fed back through the delay.
     *  @example
     * var pingPong = new Tone.PingPongDelay("4n", 0.2).toMaster();
     * var drum = new Tone.MembraneSynth().connect(pingPong);
     * drum.triggerAttackRelease("C4", "32n");
     */
    class PingPongDelay extends Tone.StereoXFeedbackEffect {
        constructor(delayTime?: Time | any, feedback?: NormalRange);
        /**
         *  the delay time signal
         *  @type {Time}
         *  @signal
         */
        delayTime: Time;
        /**
         *  @static
         *  @type {Object}
         */
        static defaults: any;
        /**
         *  Clean up.
         *  @returns {Tone.PingPongDelay} this
         */
        dispose(): Tone.PingPongDelay;
        /**
         *  The amount of feedback from the output
         *  back into the input of the effect (routed
         *  across left and right channels).
         *  @type {NormalRange}
         *  @signal
         */
        feedback: NormalRange;
        /**
         *  The wet control, i.e. how much of the effected
         *  will pass through to the output.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.PitchShift does near-realtime pitch shifting to the incoming signal.
     *         The effect is achieved by speeding up or slowing down the delayTime
     *         of a DelayNode using a sawtooth wave.
     *         Algorithm found in [this pdf](http://dsp-book.narod.ru/soundproc.pdf).
     *         Additional reference by [Miller Pucket](http://msp.ucsd.edu/techniques/v0.11/book-html/node115.html).
     *
     *  @extends {Tone.FeedbackEffect}
     *  @param {Interval=} pitch The interval to transpose the incoming signal by.
     */
    class PitchShift extends Tone.FeedbackEffect {
        constructor(pitch?: Interval);
        /**
         *  The amount of delay on the input signal
         *  @type {Time}
         *  @signal
         */
        delayTime: Time;
        /**
         *  default values
         *  @static
         *  @type {Object}
         *  @const
         */
        static readonly defaults: any;
        /**
         * Repitch the incoming signal by some interval (measured
         * in semi-tones).
         * @memberOf Tone.PitchShift#
         * @type {Interval}
         * @name pitch
         * @example
         * pitchShift.pitch = -12; //down one octave
         * pitchShift.pitch = 7; //up a fifth
         */
        pitch: Interval;
        /**
         * The window size corresponds roughly to the sample length in a looping sampler.
         * Smaller values are desirable for a less noticeable delay time of the pitch shifted
         * signal, but larger values will result in smoother pitch shifting for larger intervals.
         * A nominal range of 0.03 to 0.1 is recommended.
         * @memberOf Tone.PitchShift#
         * @type {Time}
         * @name windowSize
         * @example
         * pitchShift.windowSize = 0.1;
         */
        windowSize: Time;
        /**
         *  Clean up.
         *  @return  {Tone.PitchShift}  this
         */
        dispose(): Tone.PitchShift;
        /**
         *  The amount of signal which is fed back into the effect input.
         *  @type {NormalRange}
         *  @signal
         */
        feedback: NormalRange;
        /**
         *  The wet control is how much of the effected
         *  will pass through to the output. 1 = 100% effected
         *  signal, 0 = 100% dry signal.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Simple convolution created with decaying noise.
     *  		Generates an Impulse Response Buffer
     * 			with Tone.Offline then feeds the IR into ConvolverNode.
     * 			Note: the Reverb will not make any sound until [generate](#generate)
     * 			has been invoked and resolved.
     *
     * 			Inspiration from [ReverbGen](https://github.com/adelespinasse/reverbGen).
     * 			Copyright (c) 2014 Alan deLespinasse Apache 2.0 License.
     *
     *  @extends {Tone.Convolver}
     *  @param {Time=} decay The amount of time it will reverberate for.
     */
    class Reverb extends Tone.Convolver {
        constructor(decay?: Time);
        /**
         * The duration of the reverb
         * @type {Time}
         */
        decay: Time;
        /**
         * The amount of time before the reverb is fully
         * ramped in.
         * @type {Time}
         */
        preDelay: Time;
        /**
         * The defaults
         * @type {Object}
         * @static
         */
        static defaults: any;
        /**
         * Generate the Impulse Response. Returns a promise while the IR is being
         * generated.
         * @return {Promise<Tone.Reverb>} Promise which returns this object.
         */
        generate(): Promise<Tone.Reverb>;
        /**
         *  Clean up.
         *  @return  {Tone.Reverb}  this
         */
        dispose(): Tone.Reverb;
        /**
         *  The convolver's buffer
         *  @memberOf Tone.Convolver#
         *  @type {AudioBuffer}
         *  @name buffer
         */
        buffer: AudioBuffer;
        /**
         *  The normalize property of the ConvolverNode interface is a boolean that controls whether the impulse response from the buffer will be scaled by an equal-power normalization when the buffer attribute is set, or not.
         *  @memberOf Tone.Convolver#
         *  @type {Boolean}
         *  @name normalize
         */
        normalize: boolean;
        /**
         *  Load an impulse response url as an audio buffer.
         *  Decodes the audio asynchronously and invokes
         *  the callback once the audio buffer loads.
         *  @param {string} url The url of the buffer to load.
         *                      filetype support depends on the
         *                      browser.
         *  @param  {function=} callback
         *  @returns {Promise}
         */
        load(url: string, callback?: (...params: any[]) => any): Promise;
        /**
         *  The wet control is how much of the effected
         *  will pass through to the output. 1 = 100% effected
         *  signal, 0 = 100% dry signal.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Base class for Stereo effects. Provides effectSendL/R and effectReturnL/R.
     *
     *	@constructor
     *	@extends {Tone.Effect}
     */
    class StereoEffect extends Tone.Effect {
        /**
         *  The wet control, i.e. how much of the effected
         *  will pass through to the output.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         *  Clean up.
         *  @returns {Tone.StereoEffect} this
         */
        dispose(): Tone.StereoEffect;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Base class for stereo feedback effects where the effectReturn
     *         is fed back into the same channel.
     *
     *	@constructor
     *	@extends {Tone.StereoEffect}
     */
    class StereoFeedbackEffect extends Tone.StereoEffect {
        /**
         *  controls the amount of feedback
         *  @type {NormalRange}
         *  @signal
         */
        feedback: NormalRange;
        /**
         *  clean up
         *  @returns {Tone.StereoFeedbackEffect} this
         */
        dispose(): Tone.StereoFeedbackEffect;
        /**
         *  The wet control, i.e. how much of the effected
         *  will pass through to the output.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Applies a width factor to the mid/side seperation.
     *         0 is all mid and 1 is all side.
     *         Algorithm found in [kvraudio forums](http://www.kvraudio.com/forum/viewtopic.php?t=212587).
     *         <br><br>
     *         <code>
     *         Mid *= 2*(1-width)<br>
     *         Side *= 2*width
     *         </code>
     *
     *  @extends {Tone.MidSideEffect}
     *  @constructor
     *  @param {NormalRange|Object} [width] The stereo width. A width of 0 is mono and 1 is stereo. 0.5 is no change.
     */
    class StereoWidener extends Tone.MidSideEffect {
        constructor(width?: NormalRange | any);
        /**
         *  The width control. 0 = 100% mid. 1 = 100% side. 0.5 = no change.
         *  @type {NormalRange}
         *  @signal
         */
        width: NormalRange;
        /**
         *  the default values
         *  @static
         *  @type {Object}
         */
        static defaults: any;
        /**
         *  Clean up.
         *  @returns {Tone.StereoWidener} this
         */
        dispose(): Tone.StereoWidener;
        /**
         *  The wet control is how much of the effected
         *  will pass through to the output. 1 = 100% effected
         *  signal, 0 = 100% dry signal.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Just like a stereo feedback effect, but the feedback is routed from left to right
     *         and right to left instead of on the same channel.
     *
     *	@constructor
     *	@extends {Tone.StereoEffect}
     */
    class StereoXFeedbackEffect extends Tone.StereoEffect {
        /**
         *  The amount of feedback from the output
         *  back into the input of the effect (routed
         *  across left and right channels).
         *  @type {NormalRange}
         *  @signal
         */
        feedback: NormalRange;
        /**
         *  clean up
         *  @returns {Tone.StereoXFeedbackEffect} this
         */
        dispose(): Tone.StereoXFeedbackEffect;
        /**
         *  The wet control, i.e. how much of the effected
         *  will pass through to the output.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.Tremolo modulates the amplitude of an incoming signal using a Tone.LFO.
     *         The type, frequency, and depth of the LFO is controllable.
     *
     *  @extends {Tone.StereoEffect}
     *  @constructor
     *  @param {Frequency} [frequency] The rate of the effect.
     *  @param {NormalRange} [depth] The depth of the effect.
     *  @example
     * //create a tremolo and start it's LFO
     * var tremolo = new Tone.Tremolo(9, 0.75).toMaster().start();
     * //route an oscillator through the tremolo and start it
     * var oscillator = new Tone.Oscillator().connect(tremolo).start();
     */
    class Tremolo extends Tone.StereoEffect {
        constructor(frequency?: Frequency, depth?: NormalRange);
        /**
         *  The frequency of the tremolo.
         *  @type  {Frequency}
         *  @signal
         */
        frequency: Frequency;
        /**
         *  The depth of the effect. A depth of 0, has no effect
         *  on the amplitude, and a depth of 1 makes the amplitude
         *  modulate fully between 0 and 1.
         *  @type  {NormalRange}
         *  @signal
         */
        depth: NormalRange;
        /**
         *  @static
         *  @const
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         * Start the tremolo.
         * @param {Time} [time=now] When the tremolo begins.
         * @returns {Tone.Tremolo} this
         */
        start(time?: Time): Tone.Tremolo;
        /**
         * Stop the tremolo.
         * @param {Time} [time=now] When the tremolo stops.
         * @returns {Tone.Tremolo} this
         */
        stop(time?: Time): Tone.Tremolo;
        /**
         * Sync the effect to the transport.
         * @param {Time} [delay=0] Delay time before starting the effect after the
         *                              Transport has started.
         * @returns {Tone.Tremolo} this
         */
        sync(delay?: Time): Tone.Tremolo;
        /**
         * Unsync the filter from the transport
         * @returns {Tone.Tremolo} this
         */
        unsync(): Tone.Tremolo;
        /**
         * The Tremolo's oscillator type.
         * @memberOf Tone.Tremolo#
         * @type {string}
         * @name type
         */
        type: string;
        /**
         * Amount of stereo spread. When set to 0, both LFO's will be panned centrally.
         * When set to 180, LFO's will be panned hard left and right respectively.
         * @memberOf Tone.Tremolo#
         * @type {Degrees}
         * @name spread
         */
        spread: Degrees;
        /**
         *  clean up
         *  @returns {Tone.Tremolo} this
         */
        dispose(): Tone.Tremolo;
        /**
         *  The wet control, i.e. how much of the effected
         *  will pass through to the output.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class A Vibrato effect composed of a Tone.Delay and a Tone.LFO. The LFO
     *         modulates the delayTime of the delay, causing the pitch to rise
     *         and fall.
     *  @extends {Tone.Effect}
     *  @param {Frequency} frequency The frequency of the vibrato.
     *  @param {NormalRange} depth The amount the pitch is modulated.
     */
    class Vibrato extends Tone.Effect {
        constructor(frequency: Frequency, depth: NormalRange);
        /**
         *  The frequency of the vibrato
         *  @type {Frequency}
         *  @signal
         */
        frequency: Frequency;
        /**
         *  The depth of the vibrato.
         *  @type {NormalRange}
         *  @signal
         */
        depth: NormalRange;
        /**
         *  The defaults
         *  @type  {Object}
         *  @const
         */
        static readonly defaults: any;
        /**
         * Type of oscillator attached to the Vibrato.
         * @memberOf Tone.Vibrato#
         * @type {string}
         * @name type
         */
        type: string;
        /**
         *  Clean up.
         *  @returns {Tone.Vibrato} this
         */
        dispose(): Tone.Vibrato;
        /**
         *  The wet control is how much of the effected
         *  will pass through to the output. 1 = 100% effected
         *  signal, 0 = 100% dry signal.
         *  @type {NormalRange}
         *  @signal
         */
        wet: NormalRange;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Tone.Event abstracts away Tone.Transport.schedule and provides a schedulable
     *          callback for a single or repeatable events along the timeline.
     *
     *  @extends {Tone}
     *  @param {function} callback The callback to invoke at the time.
     *  @param {*} value The value or values which should be passed to
     *                      the callback function on invocation.
     *  @example
     * var chord = new Tone.Event(function(time, chord){
     * 	//the chord as well as the exact time of the event
     * 	//are passed in as arguments to the callback function
     * }, ["D4", "E4", "F4"]);
     * //start the chord at the beginning of the transport timeline
     * chord.start();
     * //loop it every measure for 8 measures
     * chord.loop = 8;
     * chord.loopEnd = "1m";
     */
    class Event extends Tone {
        constructor(callback: (...params: any[]) => any, value: any);
        /**
         *  The callback to invoke.
         *  @type  {Function}
         */
        callback: (...params: any[]) => any;
        /**
         *  If mute is true, the callback won't be
         *  invoked.
         *  @type {Boolean}
         */
        mute: boolean;
        /**
         *  The default values
         *  @type  {Object}
         *  @const
         */
        static readonly defaults: any;
        /**
         *  Returns the playback state of the note, either "started" or "stopped".
         *  @type {String}
         *  @readOnly
         *  @memberOf Tone.Event#
         *  @name state
         */
        readonly state: string;
        /**
         *  The probability of the notes being triggered.
         *  @memberOf Tone.Event#
         *  @type {NormalRange}
         *  @name probability
         */
        probability: NormalRange;
        /**
         *  If set to true, will apply small random variation
         *  to the callback time. If the value is given as a time, it will randomize
         *  by that amount.
         *  @type {Boolean|Time}
         *  @name humanize
         *  @memberof Tone.Event#
         *  @example
         * event.humanize = true;
         */
        humanize: boolean | Time;
        /**
         *  Start the note at the given time.
         *  @param  {TimelinePosition}  time  When the note should start.
         *  @return  {Tone.Event}  this
         */
        start(time: TimelinePosition): Tone.Event;
        /**
         *  Stop the Event at the given time.
         *  @param  {TimelinePosition}  time  When the note should stop.
         *  @return  {Tone.Event}  this
         */
        stop(time: TimelinePosition): Tone.Event;
        /**
         *  Cancel all scheduled events greater than or equal to the given time
         *  @param  {TimelinePosition}  [time=0]  The time after which events will be cancel.
         *  @return  {Tone.Event}  this
         */
        cancel(time?: TimelinePosition): Tone.Event;
        /**
         *  If the note should loop or not
         *  between Tone.Event.loopStart and
         *  Tone.Event.loopEnd. If set to true,
         *  the event will loop indefinitely,
         *  if set to a number greater than 1
         *  it will play a specific number of
         *  times, if set to false, 0 or 1, the
         *  part will only play once.
         *  @memberOf Tone.Event#
         *  @type {Boolean|Positive}
         *  @name loop
         */
        loop: boolean | Positive;
        /**
         * 	The playback rate of the note. Defaults to 1.
         *  @memberOf Tone.Event#
         *  @type {Positive}
         *  @name playbackRate
         *  @example
         * note.loop = true;
         * //repeat the note twice as fast
         * note.playbackRate = 2;
         */
        playbackRate: Positive;
        /**
         *  The loopEnd point is the time the event will loop
         *  if Tone.Event.loop is true.
         *  @memberOf Tone.Event#
         *  @type {Time}
         *  @name loopEnd
         */
        loopEnd: Time;
        /**
         *  The time when the loop should start.
         *  @memberOf Tone.Event#
         *  @type {Time}
         *  @name loopStart
         */
        loopStart: Time;
        /**
         *  The current progress of the loop interval.
         *  Returns 0 if the event is not started yet or
         *  it is not set to loop.
         *  @memberOf Tone.Event#
         *  @type {NormalRange}
         *  @name progress
         *  @readOnly
         */
        readonly progress: NormalRange;
        /**
         *  Clean up
         *  @return  {Tone.Event}  this
         */
        dispose(): Tone.Event;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.Loop creates a looped callback at the
     *         specified interval. The callback can be
     *         started, stopped and scheduled along
     *         the Transport's timeline.
     *  @example
     * var loop = new Tone.Loop(function(time){
     * 	//triggered every eighth note.
     * 	console.log(time);
     * }, "8n").start(0);
     * Tone.Transport.start();
     *  @extends {Tone}
     *  @param {Function} callback The callback to invoke with the event.
     *  @param {Time} interval The time between successive callback calls.
     */
    class Loop extends Tone {
        constructor(callback: (...params: any[]) => any, interval: Time);
        /**
         *  The event which produces the callbacks
         */
        _event: any;
        /**
         *  The callback to invoke with the next event in the pattern
         *  @type {Function}
         */
        callback: (...params: any[]) => any;
        /**
         *  The defaults
         *  @const
         *  @type  {Object}
         */
        static readonly defaults: any;
        /**
         *  Start the loop at the specified time along the Transport's
         *  timeline.
         *  @param  {TimelinePosition=}  time  When to start the Loop.
         *  @return  {Tone.Loop}  this
         */
        start(time?: TimelinePosition): Tone.Loop;
        /**
         *  Stop the loop at the given time.
         *  @param  {TimelinePosition=}  time  When to stop the Loop.
         *  @return  {Tone.Loop}  this
         */
        stop(time?: TimelinePosition): Tone.Loop;
        /**
         *  Cancel all scheduled events greater than or equal to the given time
         *  @param  {TimelinePosition}  [time=0]  The time after which events will be cancel.
         *  @return  {Tone.Loop}  this
         */
        cancel(time?: TimelinePosition): Tone.Loop;
        /**
         *  The state of the Loop, either started or stopped.
         *  @memberOf Tone.Loop#
         *  @type {String}
         *  @name state
         *  @readOnly
         */
        readonly state: string;
        /**
         *  The progress of the loop as a value between 0-1. 0, when
         *  the loop is stopped or done iterating.
         *  @memberOf Tone.Loop#
         *  @type {NormalRange}
         *  @name progress
         *  @readOnly
         */
        readonly progress: NormalRange;
        /**
         *  The time between successive callbacks.
         *  @example
         * loop.interval = "8n"; //loop every 8n
         *  @memberOf Tone.Loop#
         *  @type {Time}
         *  @name interval
         */
        interval: Time;
        /**
         *  The playback rate of the loop. The normal playback rate is 1 (no change).
         *  A `playbackRate` of 2 would be twice as fast.
         *  @memberOf Tone.Loop#
         *  @type {Time}
         *  @name playbackRate
         */
        playbackRate: Time;
        /**
         *  Random variation +/-0.01s to the scheduled time.
         *  Or give it a time value which it will randomize by.
         *  @type {Boolean|Time}
         *  @memberOf Tone.Loop#
         *  @name humanize
         */
        humanize: boolean | Time;
        /**
         *  The probably of the callback being invoked.
         *  @memberOf Tone.Loop#
         *  @type {NormalRange}
         *  @name probability
         */
        probability: NormalRange;
        /**
         *  Muting the Loop means that no callbacks are invoked.
         *  @memberOf Tone.Loop#
         *  @type {Boolean}
         *  @name mute
         */
        mute: boolean;
        /**
         *  The number of iterations of the loop. The default
         *  value is Infinity (loop forever).
         *  @memberOf Tone.Loop#
         *  @type {Positive}
         *  @name iterations
         */
        iterations: Positive;
        /**
         *  Clean up
         *  @return  {Tone.Loop}  this
         */
        dispose(): Tone.Loop;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.Part is a collection Tone.Events which can be
     *         started/stopped and looped as a single unit.
     *
     *  @extends {Tone.Event}
     *  @param {Function} callback The callback to invoke on each event
     *  @param {Array} events the array of events
     *  @example
     * var part = new Tone.Part(function(time, note){
     * 	//the notes given as the second element in the array
     * 	//will be passed in as the second argument
     * 	synth.triggerAttackRelease(note, "8n", time);
     * }, [[0, "C2"], ["0:2", "C3"], ["0:3:2", "G2"]]);
     *  @example
     * //use an array of objects as long as the object has a "time" attribute
     * var part = new Tone.Part(function(time, value){
     * 	//the value is an object which contains both the note and the velocity
     * 	synth.triggerAttackRelease(value.note, "8n", time, value.velocity);
     * }, [{"time" : 0, "note" : "C3", "velocity": 0.9},
     * 	   {"time" : "0:2", "note" : "C4", "velocity": 0.5}
     * ]).start(0);
     */
    class Part extends Tone.Event {
        constructor(callback: (...params: any[]) => any, events: Array);
        /**
         *  The default values
         *  @type  {Object}
         *  @const
         */
        static readonly defaults: any;
        /**
         *  Start the part at the given time.
         *  @param  {TransportTime}  time    When to start the part.
         *  @param  {Time=}  offset  The offset from the start of the part
         *                           to begin playing at.
         *  @return  {Tone.Part}  this
         */
        start(time: TransportTime, offset?: Time): Tone.Part;
        /**
         *  Stop the part at the given time.
         *  @param  {TimelinePosition}  time  When to stop the part.
         *  @return  {Tone.Part}  this
         */
        stop(time: TimelinePosition): Tone.Part;
        /**
         *  Get/Set an Event's value at the given time.
         *  If a value is passed in and no event exists at
         *  the given time, one will be created with that value.
         *  If two events are at the same time, the first one will
         *  be returned.
         *  @example
         * part.at("1m"); //returns the part at the first measure
         *
         * part.at("2m", "C2"); //set the value at "2m" to C2.
         * //if an event didn't exist at that time, it will be created.
         *  @param {TransportTime} time The time of the event to get or set.
         *  @param {*=} value If a value is passed in, the value of the
         *                    event at the given time will be set to it.
         *  @return {Tone.Event} the event at the time
         */
        at(time: TransportTime, value?: any): Tone.Event;
        /**
         *  Add a an event to the part.
         *  @param {Time} time The time the note should start.
         *                            If an object is passed in, it should
         *                            have a 'time' attribute and the rest
         *                            of the object will be used as the 'value'.
         *  @param  {Tone.Event|*}  value
         *  @returns {Tone.Part} this
         *  @example
         * part.add("1m", "C#+11");
         */
        add(time: Time, value: Tone.Event | any): Tone.Part;
        /**
         *  Remove an event from the part. If the event at that time is a Tone.Part,
         *  it will remove the entire part.
         *  @param {Time} time The time of the event
         *  @param {*} value Optionally select only a specific event value
         *  @return  {Tone.Part}  this
         */
        remove(time: Time, value: any): Tone.Part;
        /**
         *  Remove all of the notes from the group.
         *  @return  {Tone.Part}  this
         */
        removeAll(): Tone.Part;
        /**
         *  Cancel scheduled state change events: i.e. "start" and "stop".
         *  @param {TimelinePosition} after The time after which to cancel the scheduled events.
         *  @return  {Tone.Part}  this
         */
        cancel(after: TimelinePosition): Tone.Part;
        /**
         *  The probability of the notes being triggered.
         *  @memberOf Tone.Part#
         *  @type {NormalRange}
         *  @name probability
         */
        probability: NormalRange;
        /**
         *  If set to true, will apply small random variation
         *  to the callback time. If the value is given as a time, it will randomize
         *  by that amount.
         *  @example
         * event.humanize = true;
         *  @type {Boolean|Time}
         *  @name humanize
         *  @memberof Tone.Part#
         */
        humanize: boolean | Time;
        /**
         *  If the part should loop or not
         *  between Tone.Part.loopStart and
         *  Tone.Part.loopEnd. If set to true,
         *  the part will loop indefinitely,
         *  if set to a number greater than 1
         *  it will play a specific number of
         *  times, if set to false, 0 or 1, the
         *  part will only play once.
         *  @memberOf Tone.Part#
         *  @type {Boolean|Positive}
         *  @name loop
         *  @example
         * //loop the part 8 times
         * part.loop = 8;
         */
        loop: boolean | Positive;
        /**
         *  The loopEnd point determines when it will
         *  loop if Tone.Part.loop is true.
         *  @memberOf Tone.Part#
         *  @type {Time}
         *  @name loopEnd
         */
        loopEnd: Time;
        /**
         *  The loopStart point determines when it will
         *  loop if Tone.Part.loop is true.
         *  @memberOf Tone.Part#
         *  @type {Time}
         *  @name loopStart
         */
        loopStart: Time;
        /**
         * 	The playback rate of the part
         *  @memberOf Tone.Part#
         *  @type {Positive}
         *  @name playbackRate
         */
        playbackRate: Positive;
        /**
         * 	The number of scheduled notes in the part.
         *  @memberOf Tone.Part#
         *  @type {Positive}
         *  @name length
         *  @readOnly
         */
        readonly length: Positive;
        /**
         *  Clean up
         *  @return  {Tone.Part}  this
         */
        dispose(): Tone.Part;
        /**
         *  The callback to invoke.
         *  @type  {Function}
         */
        callback: (...params: any[]) => any;
        /**
         *  If mute is true, the callback won't be
         *  invoked.
         *  @type {Boolean}
         */
        mute: boolean;
        /**
         *  Returns the playback state of the note, either "started" or "stopped".
         *  @type {String}
         *  @readOnly
         *  @memberOf Tone.Event#
         *  @name state
         */
        readonly state: string;
        /**
         *  The current progress of the loop interval.
         *  Returns 0 if the event is not started yet or
         *  it is not set to loop.
         *  @memberOf Tone.Event#
         *  @type {NormalRange}
         *  @name progress
         *  @readOnly
         */
        readonly progress: NormalRange;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.Pattern arpeggiates between the given notes
     *         in a number of patterns. See Tone.CtrlPattern for
     *         a full list of patterns.
     *  @example
     * var pattern = new Tone.Pattern(function(time, note){
     *   //the order of the notes passed in depends on the pattern
     * }, ["C2", "D4", "E5", "A6"], "upDown");
     *  @extends {Tone.Loop}
     *  @param {Function} callback The callback to invoke with the event.
     *  @param {Array} values The values to arpeggiate over.
     */
    class Pattern extends Tone.Loop {
        constructor(callback: (...params: any[]) => any, values: Array);
        /**
         *  The defaults
         *  @const
         *  @type  {Object}
         */
        static readonly defaults: any;
        /**
         *  The current index in the values array.
         *  @memberOf Tone.Pattern#
         *  @type {Positive}
         *  @name index
         */
        index: Positive;
        /**
         *  The array of events.
         *  @memberOf Tone.Pattern#
         *  @type {Array}
         *  @name values
         */
        values: Array;
        /**
         *  The current value of the pattern.
         *  @memberOf Tone.Pattern#
         *  @type {*}
         *  @name value
         *  @readOnly
         */
        readonly value: any;
        /**
         *  The pattern type. See Tone.CtrlPattern for the full list of patterns.
         *  @memberOf Tone.Pattern#
         *  @type {String}
         *  @name pattern
         */
        pattern: string;
        /**
         *  Clean up
         *  @return  {Tone.Pattern}  this
         */
        dispose(): Tone.Pattern;
        /**
         *  The event which produces the callbacks
         */
        _event: any;
        /**
         *  The callback to invoke with the next event in the pattern
         *  @type {Function}
         */
        callback: (...params: any[]) => any;
        /**
         *  Start the loop at the specified time along the Transport's
         *  timeline.
         *  @param  {TimelinePosition=}  time  When to start the Loop.
         *  @return  {Tone.Loop}  this
         */
        start(time?: TimelinePosition): Tone.Loop;
        /**
         *  Stop the loop at the given time.
         *  @param  {TimelinePosition=}  time  When to stop the Loop.
         *  @return  {Tone.Loop}  this
         */
        stop(time?: TimelinePosition): Tone.Loop;
        /**
         *  Cancel all scheduled events greater than or equal to the given time
         *  @param  {TimelinePosition}  [time=0]  The time after which events will be cancel.
         *  @return  {Tone.Loop}  this
         */
        cancel(time?: TimelinePosition): Tone.Loop;
        /**
         *  The state of the Loop, either started or stopped.
         *  @memberOf Tone.Loop#
         *  @type {String}
         *  @name state
         *  @readOnly
         */
        readonly state: string;
        /**
         *  The progress of the loop as a value between 0-1. 0, when
         *  the loop is stopped or done iterating.
         *  @memberOf Tone.Loop#
         *  @type {NormalRange}
         *  @name progress
         *  @readOnly
         */
        readonly progress: NormalRange;
        /**
         *  The time between successive callbacks.
         *  @example
         * loop.interval = "8n"; //loop every 8n
         *  @memberOf Tone.Loop#
         *  @type {Time}
         *  @name interval
         */
        interval: Time;
        /**
         *  The playback rate of the loop. The normal playback rate is 1 (no change).
         *  A `playbackRate` of 2 would be twice as fast.
         *  @memberOf Tone.Loop#
         *  @type {Time}
         *  @name playbackRate
         */
        playbackRate: Time;
        /**
         *  Random variation +/-0.01s to the scheduled time.
         *  Or give it a time value which it will randomize by.
         *  @type {Boolean|Time}
         *  @memberOf Tone.Loop#
         *  @name humanize
         */
        humanize: boolean | Time;
        /**
         *  The probably of the callback being invoked.
         *  @memberOf Tone.Loop#
         *  @type {NormalRange}
         *  @name probability
         */
        probability: NormalRange;
        /**
         *  Muting the Loop means that no callbacks are invoked.
         *  @memberOf Tone.Loop#
         *  @type {Boolean}
         *  @name mute
         */
        mute: boolean;
        /**
         *  The number of iterations of the loop. The default
         *  value is Infinity (loop forever).
         *  @memberOf Tone.Loop#
         *  @type {Positive}
         *  @name iterations
         */
        iterations: Positive;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class A sequence is an alternate notation of a part. Instead
     *         of passing in an array of [time, event] pairs, pass
     *         in an array of events which will be spaced at the
     *         given subdivision. Sub-arrays will subdivide that beat
     *         by the number of items are in the array.
     *         Sequence notation inspiration from [Tidal](http://yaxu.org/tidal/)
     *  @param  {Function}  callback  The callback to invoke with every note
     *  @param  {Array}    events  The sequence
     *  @param  {Time} subdivision  The subdivision between which events are placed.
     *  @extends {Tone.Part}
     *  @example
     * var seq = new Tone.Sequence(function(time, note){
     * 	console.log(note);
     * //straight quater notes
     * }, ["C4", "E4", "G4", "A4"], "4n");
     *  @example
     * var seq = new Tone.Sequence(function(time, note){
     * 	console.log(note);
     * //subdivisions are given as subarrays
     * }, ["C4", ["E4", "D4", "E4"], "G4", ["A4", "G4"]]);
     */
    class Sequence extends Tone.Part {
        constructor(callback: (...params: any[]) => any, events: Array, subdivision: Time);
        /**
         *  The default values.
         *  @type  {Object}
         */
        static defaults: any;
        /**
         *  The subdivision of the sequence. This can only be
         *  set in the constructor. The subdivision is the
         *  interval between successive steps.
         *  @type {Time}
         *  @memberOf Tone.Sequence#
         *  @name subdivision
         *  @readOnly
         */
        readonly subdivision: Time;
        /**
         *  Get/Set an index of the sequence. If the index contains a subarray,
         *  a Tone.Sequence representing that sub-array will be returned.
         *  @example
         * var sequence = new Tone.Sequence(playNote, ["E4", "C4", "F#4", ["A4", "Bb3"]])
         * sequence.at(0)// => returns "E4"
         * //set a value
         * sequence.at(0, "G3");
         * //get a nested sequence
         * sequence.at(3).at(1)// => returns "Bb3"
         * @param {Positive} index The index to get or set
         * @param {*} value Optionally pass in the value to set at the given index.
         */
        at(index: Positive, value: any): void;
        /**
         *  Add an event at an index, if there's already something
         *  at that index, overwrite it. If `value` is an array,
         *  it will be parsed as a subsequence.
         *  @param {Number} index The index to add the event to
         *  @param {*} value The value to add at that index
         *  @returns {Tone.Sequence} this
         */
        add(index: number, value: any): Tone.Sequence;
        /**
         *  Remove a value from the sequence by index
         *  @param {Number} index The index of the event to remove
         *  @returns {Tone.Sequence} this
         */
        remove(index: number): Tone.Sequence;
        /**
         *  Clean up.
         *  @return {Tone.Sequence} this
         */
        dispose(): Tone.Sequence;
        /**
         *  Start the part at the given time.
         *  @param  {TransportTime}  time    When to start the part.
         *  @param  {Time=}  offset  The offset from the start of the part
         *                           to begin playing at.
         *  @return  {Tone.Part}  this
         */
        start(time: TransportTime, offset?: Time): Tone.Part;
        /**
         *  Stop the part at the given time.
         *  @param  {TimelinePosition}  time  When to stop the part.
         *  @return  {Tone.Part}  this
         */
        stop(time: TimelinePosition): Tone.Part;
        /**
         *  Remove all of the notes from the group.
         *  @return  {Tone.Part}  this
         */
        removeAll(): Tone.Part;
        /**
         *  Cancel scheduled state change events: i.e. "start" and "stop".
         *  @param {TimelinePosition} after The time after which to cancel the scheduled events.
         *  @return  {Tone.Part}  this
         */
        cancel(after: TimelinePosition): Tone.Part;
        /**
         *  The probability of the notes being triggered.
         *  @memberOf Tone.Part#
         *  @type {NormalRange}
         *  @name probability
         */
        probability: NormalRange;
        /**
         *  If set to true, will apply small random variation
         *  to the callback time. If the value is given as a time, it will randomize
         *  by that amount.
         *  @example
         * event.humanize = true;
         *  @type {Boolean|Time}
         *  @name humanize
         *  @memberof Tone.Part#
         */
        humanize: boolean | Time;
        /**
         *  If the part should loop or not
         *  between Tone.Part.loopStart and
         *  Tone.Part.loopEnd. If set to true,
         *  the part will loop indefinitely,
         *  if set to a number greater than 1
         *  it will play a specific number of
         *  times, if set to false, 0 or 1, the
         *  part will only play once.
         *  @memberOf Tone.Part#
         *  @type {Boolean|Positive}
         *  @name loop
         *  @example
         * //loop the part 8 times
         * part.loop = 8;
         */
        loop: boolean | Positive;
        /**
         *  The loopEnd point determines when it will
         *  loop if Tone.Part.loop is true.
         *  @memberOf Tone.Part#
         *  @type {Time}
         *  @name loopEnd
         */
        loopEnd: Time;
        /**
         *  The loopStart point determines when it will
         *  loop if Tone.Part.loop is true.
         *  @memberOf Tone.Part#
         *  @type {Time}
         *  @name loopStart
         */
        loopStart: Time;
        /**
         * 	The playback rate of the part
         *  @memberOf Tone.Part#
         *  @type {Positive}
         *  @name playbackRate
         */
        playbackRate: Positive;
        /**
         * 	The number of scheduled notes in the part.
         *  @memberOf Tone.Part#
         *  @type {Positive}
         *  @name length
         *  @readOnly
         */
        readonly length: Positive;
        /**
         *  The callback to invoke.
         *  @type  {Function}
         */
        callback: (...params: any[]) => any;
        /**
         *  If mute is true, the callback won't be
         *  invoked.
         *  @type {Boolean}
         */
        mute: boolean;
        /**
         *  Returns the playback state of the note, either "started" or "stopped".
         *  @type {String}
         *  @readOnly
         *  @memberOf Tone.Event#
         *  @name state
         */
        readonly state: string;
        /**
         *  The current progress of the loop interval.
         *  Returns 0 if the event is not started yet or
         *  it is not set to loop.
         *  @memberOf Tone.Event#
         *  @type {NormalRange}
         *  @name progress
         *  @readOnly
         */
        readonly progress: NormalRange;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  AMSynth uses the output of one Tone.Synth to modulate the
     *          amplitude of another Tone.Synth. The harmonicity (the ratio between
     *          the two signals) affects the timbre of the output signal greatly.
     *          Read more about Amplitude Modulation Synthesis on
     *          [SoundOnSound](https://web.archive.org/web/20160404103653/http://www.soundonsound.com:80/sos/mar00/articles/synthsecrets.htm).
     *          <img src="https://docs.google.com/drawings/d/1TQu8Ed4iFr1YTLKpB3U1_hur-UwBrh5gdBXc8BxfGKw/pub?w=1009&h=457">
     *
     *  @constructor
     *  @extends {Tone.Monophonic}
     *  @param {Object} [options] the options available for the synth
     *                            see defaults below
     *  @example
     * var synth = new Tone.AMSynth().toMaster();
     * synth.triggerAttackRelease("C4", "4n");
     */
    class AMSynth extends Tone.Monophonic {
        constructor(options?: any);
        /**
         *  The carrier's oscillator
         *  @type {Tone.Oscillator}
         */
        oscillator: Tone.Oscillator;
        /**
         *  The carrier's envelope
         *  @type {Tone.AmplitudeEnvelope}
         */
        envelope: Tone.AmplitudeEnvelope;
        /**
         *  The modulator's oscillator which is applied
         *  to the amplitude of the oscillator
         *  @type {Tone.Oscillator}
         */
        modulation: Tone.Oscillator;
        /**
         *  The modulator's envelope
         *  @type {Tone.AmplitudeEnvelope}
         */
        modulationEnvelope: Tone.AmplitudeEnvelope;
        /**
         *  The frequency.
         *  @type {Frequency}
         *  @signal
         */
        frequency: Frequency;
        /**
         *  The detune in cents
         *  @type {Cents}
         *  @signal
         */
        detune: Cents;
        /**
         *  Harmonicity is the ratio between the two voices. A harmonicity of
         *  1 is no change. Harmonicity = 2 means a change of an octave.
         *  @type {Positive}
         *  @signal
         *  @example
         * //pitch voice1 an octave below voice0
         * synth.harmonicity.value = 0.5;
         */
        harmonicity: Positive;
        /**
         *  @static
         *  @type {Object}
         */
        static defaults: any;
        /**
         *  clean up
         *  @returns {Tone.AMSynth} this
         */
        dispose(): Tone.AMSynth;
        /**
         *  The glide time between notes.
         *  @type {Time}
         */
        portamento: Time;
        /**
         *  Trigger the attack of the note optionally with a given velocity.
         *
         *
         *  @param  {Frequency} note     The note to trigger.
         *  @param  {Time} [time=now]     When the note should start.
         *  @param  {number} [velocity=1] velocity The velocity scaler
         *                                determines how "loud" the note
         *                                will be triggered.
         *  @returns {Tone.Monophonic} this
         *  @example
         * synth.triggerAttack("C4");
         *  @example
         * //trigger the note a half second from now at half velocity
         * synth.triggerAttack("C4", "+0.5", 0.5);
         */
        triggerAttack(note: Frequency, time?: Time, velocity?: number): Tone.Monophonic;
        /**
         *  Trigger the release portion of the envelope
         *  @param  {Time} [time=now] If no time is given, the release happens immediatly
         *  @returns {Tone.Monophonic} this
         *  @example
         * synth.triggerRelease();
         */
        triggerRelease(time?: Time): Tone.Monophonic;
        /**
         *  Get the level of the output at the given time. Measures
         *  the envelope(s) value at the time.
         *  @param {Time} time The time to query the envelope value
         *  @return {NormalRange} The output level between 0-1
         */
        getLevelAtTime(time: Time): NormalRange;
        /**
         *  Set the note at the given time. If no time is given, the note
         *  will set immediately.
         *  @param {Frequency} note The note to change to.
         *  @param  {Time} [time=now] The time when the note should be set.
         *  @returns {Tone.Monophonic} this
         * @example
         * //change to F#6 in one quarter note from now.
         * synth.setNote("F#6", "+4n");
         * @example
         * //change to Bb4 right now
         * synth.setNote("Bb4");
         */
        setNote(note: Frequency, time?: Time): Tone.Monophonic;
        /**
         * The volume of the output in decibels.
         * @type {Decibels}
         * @signal
         * @example
         * source.volume.value = -6;
         */
        volume: Decibels;
        /**
         * Sync the instrument to the Transport. All subsequent calls of
         * [triggerAttack](#triggerattack) and [triggerRelease](#triggerrelease)
         * will be scheduled along the transport.
         * @example
         * instrument.sync()
         * //schedule 3 notes when the transport first starts
         * instrument.triggerAttackRelease('C4', '8n', 0)
         * instrument.triggerAttackRelease('E4', '8n', '8n')
         * instrument.triggerAttackRelease('G4', '8n', '4n')
         * //start the transport to hear the notes
         * Transport.start()
         * @returns {Tone.Instrument} this
         */
        sync(): Tone.Instrument;
        /**
         * Unsync the instrument from the Transport
         * @returns {Tone.Instrument} this
         */
        unsync(): Tone.Instrument;
        /**
         *  Trigger the attack and then the release after the duration.
         *  @param  {Frequency} note     The note to trigger.
         *  @param  {Time} duration How long the note should be held for before
         *                          triggering the release. This value must be greater than 0.
         *  @param {Time} [time=now]  When the note should be triggered.
         *  @param  {NormalRange} [velocity=1] The velocity the note should be triggered at.
         *  @returns {Tone.Instrument} this
         *  @example
         * //trigger "C4" for the duration of an 8th note
         * synth.triggerAttackRelease("C4", "8n");
         */
        triggerAttackRelease(note: Frequency, duration: Time, time?: Time, velocity?: NormalRange): Tone.Instrument;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Tone.DuoSynth is a monophonic synth composed of two
     *          MonoSynths run in parallel with control over the
     *          frequency ratio between the two voices and vibrato effect.
     *          <img src="https://docs.google.com/drawings/d/1bL4GXvfRMMlqS7XyBm9CjL9KJPSUKbcdBNpqOlkFLxk/pub?w=1012&h=448">
     *
     *  @constructor
     *  @extends {Tone.Monophonic}
     *  @param {Object} [options] the options available for the synth
     *                          see defaults below
     *  @example
     * var duoSynth = new Tone.DuoSynth().toMaster();
     * duoSynth.triggerAttackRelease("C4", "2n");
     */
    class DuoSynth extends Tone.Monophonic {
        constructor(options?: any);
        /**
         *  the first voice
         *  @type {Tone.MonoSynth}
         */
        voice0: Tone.MonoSynth;
        /**
         *  the second voice
         *  @type {Tone.MonoSynth}
         */
        voice1: Tone.MonoSynth;
        /**
         * the vibrato frequency
         * @type {Frequency}
         * @signal
         */
        vibratoRate: Frequency;
        /**
         * The amount of vibrato
         * @type {Positive}
         * @signal
         */
        vibratoAmount: Positive;
        /**
         *  the frequency control
         *  @type {Frequency}
         *  @signal
         */
        frequency: Frequency;
        /**
         *  Harmonicity is the ratio between the two voices. A harmonicity of
         *  1 is no change. Harmonicity = 2 means a change of an octave.
         *  @type {Positive}
         *  @signal
         *  @example
         * //pitch voice1 an octave below voice0
         * duoSynth.harmonicity.value = 0.5;
         */
        harmonicity: Positive;
        /**
         *  @static
         *  @type {Object}
         */
        static defaults: any;
        /**
         *  Get the level of the output at the given time. Measures
         *  the envelope(s) value at the time.
         *  @param {Time} time The time to query the envelope value
         *  @return {NormalRange} The output level between 0-1
         */
        getLevelAtTime(time: Time): NormalRange;
        /**
         *  clean up
         *  @returns {Tone.DuoSynth} this
         */
        dispose(): Tone.DuoSynth;
        /**
         *  The glide time between notes.
         *  @type {Time}
         */
        portamento: Time;
        /**
         *  Trigger the attack of the note optionally with a given velocity.
         *
         *
         *  @param  {Frequency} note     The note to trigger.
         *  @param  {Time} [time=now]     When the note should start.
         *  @param  {number} [velocity=1] velocity The velocity scaler
         *                                determines how "loud" the note
         *                                will be triggered.
         *  @returns {Tone.Monophonic} this
         *  @example
         * synth.triggerAttack("C4");
         *  @example
         * //trigger the note a half second from now at half velocity
         * synth.triggerAttack("C4", "+0.5", 0.5);
         */
        triggerAttack(note: Frequency, time?: Time, velocity?: number): Tone.Monophonic;
        /**
         *  Trigger the release portion of the envelope
         *  @param  {Time} [time=now] If no time is given, the release happens immediatly
         *  @returns {Tone.Monophonic} this
         *  @example
         * synth.triggerRelease();
         */
        triggerRelease(time?: Time): Tone.Monophonic;
        /**
         *  Set the note at the given time. If no time is given, the note
         *  will set immediately.
         *  @param {Frequency} note The note to change to.
         *  @param  {Time} [time=now] The time when the note should be set.
         *  @returns {Tone.Monophonic} this
         * @example
         * //change to F#6 in one quarter note from now.
         * synth.setNote("F#6", "+4n");
         * @example
         * //change to Bb4 right now
         * synth.setNote("Bb4");
         */
        setNote(note: Frequency, time?: Time): Tone.Monophonic;
        /**
         * The volume of the output in decibels.
         * @type {Decibels}
         * @signal
         * @example
         * source.volume.value = -6;
         */
        volume: Decibels;
        /**
         * Sync the instrument to the Transport. All subsequent calls of
         * [triggerAttack](#triggerattack) and [triggerRelease](#triggerrelease)
         * will be scheduled along the transport.
         * @example
         * instrument.sync()
         * //schedule 3 notes when the transport first starts
         * instrument.triggerAttackRelease('C4', '8n', 0)
         * instrument.triggerAttackRelease('E4', '8n', '8n')
         * instrument.triggerAttackRelease('G4', '8n', '4n')
         * //start the transport to hear the notes
         * Transport.start()
         * @returns {Tone.Instrument} this
         */
        sync(): Tone.Instrument;
        /**
         * Unsync the instrument from the Transport
         * @returns {Tone.Instrument} this
         */
        unsync(): Tone.Instrument;
        /**
         *  Trigger the attack and then the release after the duration.
         *  @param  {Frequency} note     The note to trigger.
         *  @param  {Time} duration How long the note should be held for before
         *                          triggering the release. This value must be greater than 0.
         *  @param {Time} [time=now]  When the note should be triggered.
         *  @param  {NormalRange} [velocity=1] The velocity the note should be triggered at.
         *  @returns {Tone.Instrument} this
         *  @example
         * //trigger "C4" for the duration of an 8th note
         * synth.triggerAttackRelease("C4", "8n");
         */
        triggerAttackRelease(note: Frequency, duration: Time, time?: Time, velocity?: NormalRange): Tone.Instrument;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  FMSynth is composed of two Tone.Synths where one Tone.Synth modulates
     *          the frequency of a second Tone.Synth. A lot of spectral content
     *          can be explored using the modulationIndex parameter. Read more about
     *          frequency modulation synthesis on Sound On Sound: [Part 1](https://web.archive.org/web/20160403123704/http://www.soundonsound.com/sos/apr00/articles/synthsecrets.htm), [Part 2](https://web.archive.org/web/20160403115835/http://www.soundonsound.com/sos/may00/articles/synth.htm).
     *          <img src="https://docs.google.com/drawings/d/1h0PUDZXPgi4Ikx6bVT6oncrYPLluFKy7lj53puxj-DM/pub?w=902&h=462">
     *
     *  @constructor
     *  @extends {Tone.Monophonic}
     *  @param {Object} [options] the options available for the synth
     *                          see defaults below
     *  @example
     * var fmSynth = new Tone.FMSynth().toMaster();
     * fmSynth.triggerAttackRelease("C5", "4n");
     */
    class FMSynth extends Tone.Monophonic {
        constructor(options?: any);
        /**
         *  The carrier's oscillator
         *  @type {Tone.Oscillator}
         */
        oscillator: Tone.Oscillator;
        /**
         *  The carrier's envelope
         *  @type {Tone.Oscillator}
         */
        envelope: Tone.Oscillator;
        /**
         *  The modulator's oscillator which is applied
         *  to the amplitude of the oscillator
         *  @type {Tone.Oscillator}
         */
        modulation: Tone.Oscillator;
        /**
         *  The modulator's envelope
         *  @type {Tone.Oscillator}
         */
        modulationEnvelope: Tone.Oscillator;
        /**
         *  The frequency control.
         *  @type {Frequency}
         *  @signal
         */
        frequency: Frequency;
        /**
         *  The detune in cents
         *  @type {Cents}
         *  @signal
         */
        detune: Cents;
        /**
         *  Harmonicity is the ratio between the two voices. A harmonicity of
         *  1 is no change. Harmonicity = 2 means a change of an octave.
         *  @type {Positive}
         *  @signal
         *  @example
         * //pitch voice1 an octave below voice0
         * synth.harmonicity.value = 0.5;
         */
        harmonicity: Positive;
        /**
         *  The modulation index which essentially the depth or amount of the modulation. It is the
         *  ratio of the frequency of the modulating signal (mf) to the amplitude of the
         *  modulating signal (ma) -- as in ma/mf.
         *	@type {Positive}
         *	@signal
         */
        modulationIndex: Positive;
        /**
         *  @static
         *  @type {Object}
         */
        static defaults: any;
        /**
         *  clean up
         *  @returns {Tone.FMSynth} this
         */
        dispose(): Tone.FMSynth;
        /**
         *  The glide time between notes.
         *  @type {Time}
         */
        portamento: Time;
        /**
         *  Trigger the attack of the note optionally with a given velocity.
         *
         *
         *  @param  {Frequency} note     The note to trigger.
         *  @param  {Time} [time=now]     When the note should start.
         *  @param  {number} [velocity=1] velocity The velocity scaler
         *                                determines how "loud" the note
         *                                will be triggered.
         *  @returns {Tone.Monophonic} this
         *  @example
         * synth.triggerAttack("C4");
         *  @example
         * //trigger the note a half second from now at half velocity
         * synth.triggerAttack("C4", "+0.5", 0.5);
         */
        triggerAttack(note: Frequency, time?: Time, velocity?: number): Tone.Monophonic;
        /**
         *  Trigger the release portion of the envelope
         *  @param  {Time} [time=now] If no time is given, the release happens immediatly
         *  @returns {Tone.Monophonic} this
         *  @example
         * synth.triggerRelease();
         */
        triggerRelease(time?: Time): Tone.Monophonic;
        /**
         *  Get the level of the output at the given time. Measures
         *  the envelope(s) value at the time.
         *  @param {Time} time The time to query the envelope value
         *  @return {NormalRange} The output level between 0-1
         */
        getLevelAtTime(time: Time): NormalRange;
        /**
         *  Set the note at the given time. If no time is given, the note
         *  will set immediately.
         *  @param {Frequency} note The note to change to.
         *  @param  {Time} [time=now] The time when the note should be set.
         *  @returns {Tone.Monophonic} this
         * @example
         * //change to F#6 in one quarter note from now.
         * synth.setNote("F#6", "+4n");
         * @example
         * //change to Bb4 right now
         * synth.setNote("Bb4");
         */
        setNote(note: Frequency, time?: Time): Tone.Monophonic;
        /**
         * The volume of the output in decibels.
         * @type {Decibels}
         * @signal
         * @example
         * source.volume.value = -6;
         */
        volume: Decibels;
        /**
         * Sync the instrument to the Transport. All subsequent calls of
         * [triggerAttack](#triggerattack) and [triggerRelease](#triggerrelease)
         * will be scheduled along the transport.
         * @example
         * instrument.sync()
         * //schedule 3 notes when the transport first starts
         * instrument.triggerAttackRelease('C4', '8n', 0)
         * instrument.triggerAttackRelease('E4', '8n', '8n')
         * instrument.triggerAttackRelease('G4', '8n', '4n')
         * //start the transport to hear the notes
         * Transport.start()
         * @returns {Tone.Instrument} this
         */
        sync(): Tone.Instrument;
        /**
         * Unsync the instrument from the Transport
         * @returns {Tone.Instrument} this
         */
        unsync(): Tone.Instrument;
        /**
         *  Trigger the attack and then the release after the duration.
         *  @param  {Frequency} note     The note to trigger.
         *  @param  {Time} duration How long the note should be held for before
         *                          triggering the release. This value must be greater than 0.
         *  @param {Time} [time=now]  When the note should be triggered.
         *  @param  {NormalRange} [velocity=1] The velocity the note should be triggered at.
         *  @returns {Tone.Instrument} this
         *  @example
         * //trigger "C4" for the duration of an 8th note
         * synth.triggerAttackRelease("C4", "8n");
         */
        triggerAttackRelease(note: Frequency, duration: Time, time?: Time, velocity?: NormalRange): Tone.Instrument;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Base-class for all instruments
     *
     *  @constructor
     *  @extends {Tone.AudioNode}
     */
    class Instrument extends Tone.AudioNode {
        /**
         * The volume of the output in decibels.
         * @type {Decibels}
         * @signal
         * @example
         * source.volume.value = -6;
         */
        volume: Decibels;
        /**
         *  the default attributes
         *  @type {object}
         */
        static defaults: any;
        /**
         *  @abstract
         *  @param {string|number} note the note to trigger
         *  @param {Time} [time=now] the time to trigger the ntoe
         *  @param {number} [velocity=1] the velocity to trigger the note
         */
        triggerAttack: any;
        /**
         *  @abstract
         *  @param {Time} [time=now] when to trigger the release
         */
        triggerRelease: any;
        /**
         * Sync the instrument to the Transport. All subsequent calls of
         * [triggerAttack](#triggerattack) and [triggerRelease](#triggerrelease)
         * will be scheduled along the transport.
         * @example
         * instrument.sync()
         * //schedule 3 notes when the transport first starts
         * instrument.triggerAttackRelease('C4', '8n', 0)
         * instrument.triggerAttackRelease('E4', '8n', '8n')
         * instrument.triggerAttackRelease('G4', '8n', '4n')
         * //start the transport to hear the notes
         * Transport.start()
         * @returns {Tone.Instrument} this
         */
        sync(): Tone.Instrument;
        /**
         * Unsync the instrument from the Transport
         * @returns {Tone.Instrument} this
         */
        unsync(): Tone.Instrument;
        /**
         *  Trigger the attack and then the release after the duration.
         *  @param  {Frequency} note     The note to trigger.
         *  @param  {Time} duration How long the note should be held for before
         *                          triggering the release. This value must be greater than 0.
         *  @param {Time} [time=now]  When the note should be triggered.
         *  @param  {NormalRange} [velocity=1] The velocity the note should be triggered at.
         *  @returns {Tone.Instrument} this
         *  @example
         * //trigger "C4" for the duration of an 8th note
         * synth.triggerAttackRelease("C4", "8n");
         */
        triggerAttackRelease(note: Frequency, duration: Time, time?: Time, velocity?: NormalRange): Tone.Instrument;
        /**
         *  clean up
         *  @returns {Tone.Instrument} this
         */
        dispose(): Tone.Instrument;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Tone.MembraneSynth makes kick and tom sounds using a single oscillator
     *          with an amplitude envelope and frequency ramp. A Tone.OmniOscillator
     *          is routed through a Tone.AmplitudeEnvelope to the output. The drum
     *          quality of the sound comes from the frequency envelope applied
     *          during Tone.MembraneSynth.triggerAttack(note). The frequency envelope
     *          starts at <code>note * .octaves</code> and ramps to <code>note</code>
     *          over the duration of <code>.pitchDecay</code>.
     *
     *  @constructor
     *  @extends {Tone.Instrument}
     *  @param {Object} [options] the options available for the synth
     *                          see defaults below
     *  @example
     * var synth = new Tone.MembraneSynth().toMaster();
     * synth.triggerAttackRelease("C2", "8n");
     */
    class MembraneSynth extends Tone.Instrument {
        constructor(options?: any);
        /**
         *  The oscillator.
         *  @type {Tone.OmniOscillator}
         */
        oscillator: Tone.OmniOscillator;
        /**
         *  The amplitude envelope.
         *  @type {Tone.AmplitudeEnvelope}
         */
        envelope: Tone.AmplitudeEnvelope;
        /**
         *  The number of octaves the pitch envelope ramps.
         *  @type {Positive}
         */
        octaves: Positive;
        /**
         *  The amount of time the frequency envelope takes.
         *  @type {Time}
         */
        pitchDecay: Time;
        /**
         *  @static
         *  @type {Object}
         */
        static defaults: any;
        /**
         *  Trigger the note at the given time with the given velocity.
         *
         *  @param  {Frequency} note     the note
         *  @param  {Time} [time=now]     the time, if not given is now
         *  @param  {number} [velocity=1] velocity defaults to 1
         *  @returns {Tone.MembraneSynth} this
         *  @example
         *  kick.triggerAttack(60);
         */
        triggerAttack(note: Frequency, time?: Time, velocity?: number): Tone.MembraneSynth;
        /**
         *  Trigger the release portion of the note.
         *
         *  @param  {Time} [time=now] the time the note will release
         *  @returns {Tone.MembraneSynth} this
         */
        triggerRelease(time?: Time): Tone.MembraneSynth;
        /**
         *  Clean up.
         *  @returns {Tone.MembraneSynth} this
         */
        dispose(): Tone.MembraneSynth;
        /**
         * The volume of the output in decibels.
         * @type {Decibels}
         * @signal
         * @example
         * source.volume.value = -6;
         */
        volume: Decibels;
        /**
         * Sync the instrument to the Transport. All subsequent calls of
         * [triggerAttack](#triggerattack) and [triggerRelease](#triggerrelease)
         * will be scheduled along the transport.
         * @example
         * instrument.sync()
         * //schedule 3 notes when the transport first starts
         * instrument.triggerAttackRelease('C4', '8n', 0)
         * instrument.triggerAttackRelease('E4', '8n', '8n')
         * instrument.triggerAttackRelease('G4', '8n', '4n')
         * //start the transport to hear the notes
         * Transport.start()
         * @returns {Tone.Instrument} this
         */
        sync(): Tone.Instrument;
        /**
         * Unsync the instrument from the Transport
         * @returns {Tone.Instrument} this
         */
        unsync(): Tone.Instrument;
        /**
         *  Trigger the attack and then the release after the duration.
         *  @param  {Frequency} note     The note to trigger.
         *  @param  {Time} duration How long the note should be held for before
         *                          triggering the release. This value must be greater than 0.
         *  @param {Time} [time=now]  When the note should be triggered.
         *  @param  {NormalRange} [velocity=1] The velocity the note should be triggered at.
         *  @returns {Tone.Instrument} this
         *  @example
         * //trigger "C4" for the duration of an 8th note
         * synth.triggerAttackRelease("C4", "8n");
         */
        triggerAttackRelease(note: Frequency, duration: Time, time?: Time, velocity?: NormalRange): Tone.Instrument;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  A highly inharmonic and spectrally complex source with a highpass filter
     *          and amplitude envelope which is good for making metalophone sounds. Based
     *          on CymbalSynth by [@polyrhythmatic](https://github.com/polyrhythmatic).
     *          Inspiration from [Sound on Sound](https://web.archive.org/web/20160610143924/https://www.soundonsound.com/sos/jul02/articles/synthsecrets0702.asp).
     *
     *  @constructor
     *  @extends {Tone.Instrument}
     *  @param {Object} [options] The options availble for the synth
     *                             see defaults below
     */
    class MetalSynth extends Tone.Instrument {
        constructor(options?: any);
        /**
         *  The frequency of the cymbal
         *  @type  {Frequency}
         *  @signal
         */
        frequency: Frequency;
        /**
         *  The envelope which is connected both to the
         *  amplitude and highpass filter's cutoff frequency
         *  @type  {Tone.Envelope}
         */
        envelope: Tone.Envelope;
        /**
         *  default values
         *  @static
         *  @const
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         *  Trigger the attack.
         *  @param  {Time}  time      When the attack should be triggered.
         *  @param  {NormalRange}  [velocity=1]  The velocity that the envelope should be triggered at.
         *  @return  {Tone.MetalSynth}  this
         */
        triggerAttack(time: Time, velocity?: NormalRange): Tone.MetalSynth;
        /**
         *  Trigger the release of the envelope.
         *  @param  {Time}  time      When the release should be triggered.
         *  @return  {Tone.MetalSynth}  this
         */
        triggerRelease(time: Time): Tone.MetalSynth;
        /**
         * Sync the instrument to the Transport. All subsequent calls of
         * [triggerAttack](#triggerattack) and [triggerRelease](#triggerrelease)
         * will be scheduled along the transport.
         * @example
         * synth.sync()
         * //schedule 3 notes when the transport first starts
         * synth.triggerAttackRelease('8n', 0)
         * synth.triggerAttackRelease('8n', '8n')
         * synth.triggerAttackRelease('8n', '4n')
         * //start the transport to hear the notes
         * Transport.start()
         * @returns {Tone.Instrument} this
         */
        sync(): Tone.Instrument;
        /**
         *  Trigger the attack and release of the envelope after the given
         *  duration.
         *  @param  {Time}  duration  The duration before triggering the release
         *  @param  {Time}  time      When the attack should be triggered.
         *  @param  {NormalRange}  [velocity=1]  The velocity that the envelope should be triggered at.
         *  @return  {Tone.MetalSynth}  this
         */
        triggerAttackRelease(duration: Time, time: Time, velocity?: NormalRange): Tone.MetalSynth;
        /**
         *  The modulationIndex of the oscillators which make up the source.
         *  see Tone.FMOscillator.modulationIndex
         *  @memberOf Tone.MetalSynth#
         *  @type {Positive}
         *  @name  modulationIndex
         */
        modulationIndex: Positive;
        /**
         *  The harmonicity of the oscillators which make up the source.
         *  see Tone.FMOscillator.harmonicity
         *  @memberOf Tone.MetalSynth#
         *  @type {Positive}
         *  @name  harmonicity
         */
        harmonicity: Positive;
        /**
         *  The frequency of the highpass filter attached to the envelope
         *  @memberOf Tone.MetalSynth#
         *  @type {Frequency}
         *  @name  resonance
         */
        resonance: Frequency;
        /**
         *  The number of octaves above the "resonance" frequency
         *  that the filter ramps during the attack/decay envelope
         *  @memberOf Tone.MetalSynth#
         *  @type {Number}
         *  @name  octaves
         */
        octaves: number;
        /**
         *  Clean up
         *  @returns {Tone.MetalSynth} this
         */
        dispose(): Tone.MetalSynth;
        /**
         * The volume of the output in decibels.
         * @type {Decibels}
         * @signal
         * @example
         * source.volume.value = -6;
         */
        volume: Decibels;
        /**
         * Unsync the instrument from the Transport
         * @returns {Tone.Instrument} this
         */
        unsync(): Tone.Instrument;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Tone.MonoSynth is composed of one oscillator, one filter, and two envelopes.
     *          The amplitude of the Tone.Oscillator and the cutoff frequency of the
     *          Tone.Filter are controlled by Tone.Envelopes.
     *          <img src="https://docs.google.com/drawings/d/1gaY1DF9_Hzkodqf8JI1Cg2VZfwSElpFQfI94IQwad38/pub?w=924&h=240">
     *
     *  @constructor
     *  @extends {Tone.Monophonic}
     *  @param {Object} [options] the options available for the synth
     *                          see defaults below
     *  @example
     * var synth = new Tone.MonoSynth({
     * 	"oscillator" : {
     * 		"type" : "square"
     *  },
     *  "envelope" : {
     *  	"attack" : 0.1
     *  }
     * }).toMaster();
     * synth.triggerAttackRelease("C4", "8n");
     */
    class MonoSynth extends Tone.Monophonic {
        constructor(options?: any);
        /**
         *  The oscillator.
         *  @type {Tone.OmniOscillator}
         */
        oscillator: Tone.OmniOscillator;
        /**
         *  The frequency control.
         *  @type {Frequency}
         *  @signal
         */
        frequency: Frequency;
        /**
         *  The detune control.
         *  @type {Cents}
         *  @signal
         */
        detune: Cents;
        /**
         *  The filter.
         *  @type {Tone.Filter}
         */
        filter: Tone.Filter;
        /**
         *  The filter envelope.
         *  @type {Tone.FrequencyEnvelope}
         */
        filterEnvelope: Tone.FrequencyEnvelope;
        /**
         *  The amplitude envelope.
         *  @type {Tone.AmplitudeEnvelope}
         */
        envelope: Tone.AmplitudeEnvelope;
        /**
         *  @const
         *  @static
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         *  clean up
         *  @returns {Tone.MonoSynth} this
         */
        dispose(): Tone.MonoSynth;
        /**
         *  The glide time between notes.
         *  @type {Time}
         */
        portamento: Time;
        /**
         *  Trigger the attack of the note optionally with a given velocity.
         *
         *
         *  @param  {Frequency} note     The note to trigger.
         *  @param  {Time} [time=now]     When the note should start.
         *  @param  {number} [velocity=1] velocity The velocity scaler
         *                                determines how "loud" the note
         *                                will be triggered.
         *  @returns {Tone.Monophonic} this
         *  @example
         * synth.triggerAttack("C4");
         *  @example
         * //trigger the note a half second from now at half velocity
         * synth.triggerAttack("C4", "+0.5", 0.5);
         */
        triggerAttack(note: Frequency, time?: Time, velocity?: number): Tone.Monophonic;
        /**
         *  Trigger the release portion of the envelope
         *  @param  {Time} [time=now] If no time is given, the release happens immediatly
         *  @returns {Tone.Monophonic} this
         *  @example
         * synth.triggerRelease();
         */
        triggerRelease(time?: Time): Tone.Monophonic;
        /**
         *  Get the level of the output at the given time. Measures
         *  the envelope(s) value at the time.
         *  @param {Time} time The time to query the envelope value
         *  @return {NormalRange} The output level between 0-1
         */
        getLevelAtTime(time: Time): NormalRange;
        /**
         *  Set the note at the given time. If no time is given, the note
         *  will set immediately.
         *  @param {Frequency} note The note to change to.
         *  @param  {Time} [time=now] The time when the note should be set.
         *  @returns {Tone.Monophonic} this
         * @example
         * //change to F#6 in one quarter note from now.
         * synth.setNote("F#6", "+4n");
         * @example
         * //change to Bb4 right now
         * synth.setNote("Bb4");
         */
        setNote(note: Frequency, time?: Time): Tone.Monophonic;
        /**
         * The volume of the output in decibels.
         * @type {Decibels}
         * @signal
         * @example
         * source.volume.value = -6;
         */
        volume: Decibels;
        /**
         * Sync the instrument to the Transport. All subsequent calls of
         * [triggerAttack](#triggerattack) and [triggerRelease](#triggerrelease)
         * will be scheduled along the transport.
         * @example
         * instrument.sync()
         * //schedule 3 notes when the transport first starts
         * instrument.triggerAttackRelease('C4', '8n', 0)
         * instrument.triggerAttackRelease('E4', '8n', '8n')
         * instrument.triggerAttackRelease('G4', '8n', '4n')
         * //start the transport to hear the notes
         * Transport.start()
         * @returns {Tone.Instrument} this
         */
        sync(): Tone.Instrument;
        /**
         * Unsync the instrument from the Transport
         * @returns {Tone.Instrument} this
         */
        unsync(): Tone.Instrument;
        /**
         *  Trigger the attack and then the release after the duration.
         *  @param  {Frequency} note     The note to trigger.
         *  @param  {Time} duration How long the note should be held for before
         *                          triggering the release. This value must be greater than 0.
         *  @param {Time} [time=now]  When the note should be triggered.
         *  @param  {NormalRange} [velocity=1] The velocity the note should be triggered at.
         *  @returns {Tone.Instrument} this
         *  @example
         * //trigger "C4" for the duration of an 8th note
         * synth.triggerAttackRelease("C4", "8n");
         */
        triggerAttackRelease(note: Frequency, duration: Time, time?: Time, velocity?: NormalRange): Tone.Instrument;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  This is an abstract base class for other monophonic instruments to
     *          extend. IMPORTANT: It does not make any sound on its own and
     *          shouldn't be directly instantiated.
     *
     *  @constructor
     *  @abstract
     *  @extends {Tone.Instrument}
     */
    class Monophonic extends Tone.Instrument {
        /**
         *  The glide time between notes.
         *  @type {Time}
         */
        portamento: Time;
        /**
         *  @static
         *  @const
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         *  Trigger the attack of the note optionally with a given velocity.
         *
         *
         *  @param  {Frequency} note     The note to trigger.
         *  @param  {Time} [time=now]     When the note should start.
         *  @param  {number} [velocity=1] velocity The velocity scaler
         *                                determines how "loud" the note
         *                                will be triggered.
         *  @returns {Tone.Monophonic} this
         *  @example
         * synth.triggerAttack("C4");
         *  @example
         * //trigger the note a half second from now at half velocity
         * synth.triggerAttack("C4", "+0.5", 0.5);
         */
        triggerAttack(note: Frequency, time?: Time, velocity?: number): Tone.Monophonic;
        /**
         *  Trigger the release portion of the envelope
         *  @param  {Time} [time=now] If no time is given, the release happens immediatly
         *  @returns {Tone.Monophonic} this
         *  @example
         * synth.triggerRelease();
         */
        triggerRelease(time?: Time): Tone.Monophonic;
        /**
         *  Get the level of the output at the given time. Measures
         *  the envelope(s) value at the time.
         *  @param {Time} time The time to query the envelope value
         *  @return {NormalRange} The output level between 0-1
         */
        getLevelAtTime(time: Time): NormalRange;
        /**
         *  Set the note at the given time. If no time is given, the note
         *  will set immediately.
         *  @param {Frequency} note The note to change to.
         *  @param  {Time} [time=now] The time when the note should be set.
         *  @returns {Tone.Monophonic} this
         * @example
         * //change to F#6 in one quarter note from now.
         * synth.setNote("F#6", "+4n");
         * @example
         * //change to Bb4 right now
         * synth.setNote("Bb4");
         */
        setNote(note: Frequency, time?: Time): Tone.Monophonic;
        /**
         * The volume of the output in decibels.
         * @type {Decibels}
         * @signal
         * @example
         * source.volume.value = -6;
         */
        volume: Decibels;
        /**
         * Sync the instrument to the Transport. All subsequent calls of
         * [triggerAttack](#triggerattack) and [triggerRelease](#triggerrelease)
         * will be scheduled along the transport.
         * @example
         * instrument.sync()
         * //schedule 3 notes when the transport first starts
         * instrument.triggerAttackRelease('C4', '8n', 0)
         * instrument.triggerAttackRelease('E4', '8n', '8n')
         * instrument.triggerAttackRelease('G4', '8n', '4n')
         * //start the transport to hear the notes
         * Transport.start()
         * @returns {Tone.Instrument} this
         */
        sync(): Tone.Instrument;
        /**
         * Unsync the instrument from the Transport
         * @returns {Tone.Instrument} this
         */
        unsync(): Tone.Instrument;
        /**
         *  Trigger the attack and then the release after the duration.
         *  @param  {Frequency} note     The note to trigger.
         *  @param  {Time} duration How long the note should be held for before
         *                          triggering the release. This value must be greater than 0.
         *  @param {Time} [time=now]  When the note should be triggered.
         *  @param  {NormalRange} [velocity=1] The velocity the note should be triggered at.
         *  @returns {Tone.Instrument} this
         *  @example
         * //trigger "C4" for the duration of an 8th note
         * synth.triggerAttackRelease("C4", "8n");
         */
        triggerAttackRelease(note: Frequency, duration: Time, time?: Time, velocity?: NormalRange): Tone.Instrument;
        /**
         *  clean up
         *  @returns {Tone.Instrument} this
         */
        dispose(): Tone.Instrument;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Tone.NoiseSynth is composed of a noise generator (Tone.Noise), one filter (Tone.Filter),
     *          and two envelopes (Tone.Envelop). One envelope controls the amplitude
     *          of the noise and the other is controls the cutoff frequency of the filter.
     *          <img src="https://docs.google.com/drawings/d/1rqzuX9rBlhT50MRvD2TKml9bnZhcZmzXF1rf_o7vdnE/pub?w=918&h=242">
     *
     *  @constructor
     *  @extends {Tone.Instrument}
     *  @param {Object} [options] the options available for the synth
     *                          see defaults below
     * @example
     * var noiseSynth = new Tone.NoiseSynth().toMaster();
     * noiseSynth.triggerAttackRelease("8n");
     */
    class NoiseSynth extends Tone.Instrument {
        constructor(options?: any);
        /**
         *  The noise source.
         *  @type {Tone.Noise}
         *  @example
         * noiseSynth.set("noise.type", "brown");
         */
        noise: Tone.Noise;
        /**
         *  The amplitude envelope.
         *  @type {Tone.AmplitudeEnvelope}
         */
        envelope: Tone.AmplitudeEnvelope;
        /**
         *  @const
         *  @static
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         *  Start the attack portion of the envelopes. Unlike other
         *  instruments, Tone.NoiseSynth doesn't have a note.
         *  @param {Time} [time=now] the time the attack should start
         *  @param {number} [velocity=1] the velocity of the note (0-1)
         *  @returns {Tone.NoiseSynth} this
         *  @example
         * noiseSynth.triggerAttack();
         */
        triggerAttack(time?: Time, velocity?: number): Tone.NoiseSynth;
        /**
         *  Start the release portion of the envelopes.
         *  @param {Time} [time=now] the time the release should start
         *  @returns {Tone.NoiseSynth} this
         */
        triggerRelease(time?: Time): Tone.NoiseSynth;
        /**
         * Sync the instrument to the Transport. All subsequent calls of
         * [triggerAttack](#triggerattack) and [triggerRelease](#triggerrelease)
         * will be scheduled along the transport.
         * @example
         * synth.sync()
         * //schedule 3 notes when the transport first starts
         * synth.triggerAttackRelease('8n', 0)
         * synth.triggerAttackRelease('8n', '8n')
         * synth.triggerAttackRelease('8n', '4n')
         * //start the transport to hear the notes
         * Transport.start()
         * @returns {Tone.Instrument} this
         */
        sync(): Tone.Instrument;
        /**
         *  Trigger the attack and then the release.
         *  @param  {Time} duration the duration of the note
         *  @param  {Time} [time=now]     the time of the attack
         *  @param  {number} [velocity=1] the velocity
         *  @returns {Tone.NoiseSynth} this
         */
        triggerAttackRelease(duration: Time, time?: Time, velocity?: number): Tone.NoiseSynth;
        /**
         *  Clean up.
         *  @returns {Tone.NoiseSynth} this
         */
        dispose(): Tone.NoiseSynth;
        /**
         * The volume of the output in decibels.
         * @type {Decibels}
         * @signal
         * @example
         * source.volume.value = -6;
         */
        volume: Decibels;
        /**
         * Unsync the instrument from the Transport
         * @returns {Tone.Instrument} this
         */
        unsync(): Tone.Instrument;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Karplus-String string synthesis. Often out of tune.
     *         Will change when the AudioWorkerNode is available across
     *         browsers.
     *
     *  @constructor
     *  @extends {Tone.Instrument}
     *  @param {Object} [options] see the defaults
     *  @example
     * var plucky = new Tone.PluckSynth().toMaster();
     * plucky.triggerAttack("C4");
     */
    class PluckSynth extends Tone.Instrument {
        constructor(options?: any);
        /**
         *  The amount of noise at the attack.
         *  Nominal range of [0.1, 20]
         *  @type {number}
         */
        attackNoise: number;
        /**
         *  The resonance control.
         *  @type {NormalRange}
         *  @signal
         */
        resonance: NormalRange;
        /**
         *  The dampening control. i.e. the lowpass filter frequency of the comb filter
         *  @type {Frequency}
         *  @signal
         */
        dampening: Frequency;
        /**
         *  @static
         *  @const
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         *  Trigger the note.
         *  @param {Frequency} note The note to trigger.
         *  @param {Time} [time=now] When the note should be triggered.
         *  @returns {Tone.PluckSynth} this
         */
        triggerAttack(note: Frequency, time?: Time): Tone.PluckSynth;
        /**
         *  Clean up.
         *  @returns {Tone.PluckSynth} this
         */
        dispose(): Tone.PluckSynth;
        /**
         * The volume of the output in decibels.
         * @type {Decibels}
         * @signal
         * @example
         * source.volume.value = -6;
         */
        volume: Decibels;
        /**
         *  @abstract
         *  @param {Time} [time=now] when to trigger the release
         */
        triggerRelease: any;
        /**
         * Sync the instrument to the Transport. All subsequent calls of
         * [triggerAttack](#triggerattack) and [triggerRelease](#triggerrelease)
         * will be scheduled along the transport.
         * @example
         * instrument.sync()
         * //schedule 3 notes when the transport first starts
         * instrument.triggerAttackRelease('C4', '8n', 0)
         * instrument.triggerAttackRelease('E4', '8n', '8n')
         * instrument.triggerAttackRelease('G4', '8n', '4n')
         * //start the transport to hear the notes
         * Transport.start()
         * @returns {Tone.Instrument} this
         */
        sync(): Tone.Instrument;
        /**
         * Unsync the instrument from the Transport
         * @returns {Tone.Instrument} this
         */
        unsync(): Tone.Instrument;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Tone.PolySynth handles voice creation and allocation for any
     *          instruments passed in as the second paramter. PolySynth is
     *          not a synthesizer by itself, it merely manages voices of
     *          one of the other types of synths, allowing any of the
     *          monophonic synthesizers to be polyphonic.
     *
     *  @constructor
     *  @extends {Tone.Instrument}
     *  @param {number|Object} [polyphony=4] The number of voices to create
     *  @param {function} [voice=Tone.Synth] The constructor of the voices
     *                                            uses Tone.Synth by default.
     *  @param {...*}	voiceArgs	All additional arguments will be passed into the class constructor.
     *  @example
     * //a polysynth composed of 6 Voices of Synth
     * var synth = new Tone.PolySynth(6, Tone.Synth, {
     *   oscillator : {
     * 		type : "square"
     * 	}
     * }).toMaster();
     * //set the attributes using the set interface
     * synth.set("detune", -1200);
     * //play a chord
     * synth.triggerAttackRelease(["C4", "E4", "A4"], "4n");
     */
    class PolySynth extends Tone.Instrument {
        constructor(polyphony?: number | any, voice?: (...params: any[]) => any, ...voiceArgs: any[]);
        /**
         *  the array of voices
         *  @type {Array}
         */
        voices: Array;
        /**
         *  The detune in cents
         *  @type {Cents}
         *  @signal
         */
        detune: Cents;
        /**
         *  the defaults
         *  @const
         *  @static
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         *  Trigger the attack portion of the note
         *  @param  {Frequency|Array} notes The notes to play. Accepts a single
         *                                  Frequency or an array of frequencies.
         *  @param  {Time} [time=now]  The start time of the note.
         *  @param {number} [velocity=1] The velocity of the note.
         *  @returns {Tone.PolySynth} this
         *  @example
         * //trigger a chord immediately with a velocity of 0.2
         * poly.triggerAttack(["Ab3", "C4", "F5"], undefined, 0.2);
         */
        triggerAttack(notes: Frequency | Array, time?: Time, velocity?: number): Tone.PolySynth;
        /**
         *  Trigger the release of the note. Unlike monophonic instruments,
         *  a note (or array of notes) needs to be passed in as the first argument.
         *  @param  {Frequency|Array} notes The notes to play. Accepts a single
         *                                  Frequency or an array of frequencies.
         *  @param  {Time} [time=now]  When the release will be triggered.
         *  @returns {Tone.PolySynth} this
         *  @example
         * poly.triggerRelease(["Ab3", "C4", "F5"], "+2n");
         */
        triggerRelease(notes: Frequency | Array, time?: Time): Tone.PolySynth;
        /**
         *  Trigger the attack and release after the specified duration
         *
         *  @param  {Frequency|Array} notes The notes to play. Accepts a single
         *                                  Frequency or an array of frequencies.
         *  @param  {Time} duration the duration of the note
         *  @param  {Time} [time=now]     if no time is given, defaults to now
         *  @param  {number} [velocity=1] the velocity of the attack (0-1)
         *  @returns {Tone.PolySynth} this
         *  @example
         * //trigger a chord for a duration of a half note
         * poly.triggerAttackRelease(["Eb3", "G4", "C5"], "2n");
         *  @example
         * //can pass in an array of durations as well
         * poly.triggerAttackRelease(["Eb3", "G4", "C5"], ["2n", "4n", "4n"]);
         */
        triggerAttackRelease(notes: Frequency | Array, duration: Time, time?: Time, velocity?: number): Tone.PolySynth;
        /**
         * Sync the instrument to the Transport. All subsequent calls of
         * [triggerAttack](#triggerattack) and [triggerRelease](#triggerrelease)
         * will be scheduled along the transport.
         * @example
         * synth.sync()
         * //schedule 3 notes when the transport first starts
         * synth.triggerAttackRelease('8n', 0)
         * synth.triggerAttackRelease('8n', '8n')
         * synth.triggerAttackRelease('8n', '4n')
         * //start the transport to hear the notes
         * Transport.start()
         * @returns {Tone.Instrument} this
         */
        sync(): Tone.Instrument;
        /**
         *  Set a member/attribute of the voices.
         *  @param {Object|string} params
         *  @param {number=} value
         *  @param {Time=} rampTime
         *  @returns {Tone.PolySynth} this
         *  @example
         * poly.set({
         * 	"filter" : {
         * 		"type" : "highpass"
         * 	},
         * 	"envelope" : {
         * 		"attack" : 0.25
         * 	}
         * });
         */
        set(params: any | string, value?: number, rampTime?: Time): Tone.PolySynth;
        /**
         *  Get the synth's attributes. Given no arguments get
         *  will return all available object properties and their corresponding
         *  values. Pass in a single attribute to retrieve or an array
         *  of attributes. The attribute strings can also include a "."
         *  to access deeper properties.
         *  @param {Array=} params the parameters to get, otherwise will return
         *  					   all available.
         */
        get(params?: Array): void;
        /**
         *  Trigger the release portion of all the currently active voices.
         *  @param {Time} [time=now] When the notes should be released.
         *  @return {Tone.PolySynth} this
         */
        releaseAll(time?: Time): Tone.PolySynth;
        /**
         *  Clean up.
         *  @returns {Tone.PolySynth} this
         */
        dispose(): Tone.PolySynth;
        /**
         *  The maximum number of notes that can be allocated
         *  to a polysynth.
         *  @type  {Number}
         *  @static
         */
        static MAX_POLYPHONY: number;
        /**
         * The volume of the output in decibels.
         * @type {Decibels}
         * @signal
         * @example
         * source.volume.value = -6;
         */
        volume: Decibels;
        /**
         * Unsync the instrument from the Transport
         * @returns {Tone.Instrument} this
         */
        unsync(): Tone.Instrument;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     * @class Automatically interpolates between a set of pitched samples. Pass in an object which maps the note's pitch or midi value to the url, then you can trigger the attack and release of that note like other instruments. By automatically repitching the samples, it is possible to play pitches which were not explicitly included which can save loading time.
     *        For sample or buffer playback where repitching is not necessary, use [Tone.Player](https://tonejs.github.io/docs/Player).
     * @param {Object} samples An object of samples mapping either Midi
     *                         Note Numbers or Scientific Pitch Notation
     *                         to the url of that sample.
     * @param {Function=} onload The callback to invoke when all of the samples are loaded.
     * @param {String=} baseUrl The root URL of all of the samples, which is prepended to all the URLs.
     * @example
     * var sampler = new Tone.Sampler({
     * 	"C3" : "path/to/C3.mp3",
     * 	"D#3" : "path/to/Dsharp3.mp3",
     * 	"F#3" : "path/to/Fsharp3.mp3",
     * 	"A3" : "path/to/A3.mp3",
     * }, function(){
     * 	//sampler will repitch the closest sample
     * 	sampler.triggerAttack("D3")
     * })
     * @extends {Tone.Instrument}
     */
    class Sampler extends Tone.Instrument {
        constructor(samples: any, onload?: (...params: any[]) => any, baseUrl?: string);
        /**
         * The envelope applied to the beginning of the sample.
         * @type {Time}
         */
        attack: Time;
        /**
         * The envelope applied to the end of the envelope.
         * @type {Time}
         */
        release: Time;
        /**
         *  The shape of the attack/release curve.
         *  Either "linear" or "exponential"
         *  @type {String}
         */
        curve: string;
        /**
         * The defaults
         * @const
         * @type {Object}
         */
        static readonly defaults: any;
        /**
         * @param  {(Frequency|Frequency[])} notes	The note to play, or an array of notes.
         * @param  {Time=} time     When to play the note
         * @param  {NormalRange=} velocity The velocity to play the sample back.
         * @return {Tone.Sampler}          this
         */
        triggerAttack(notes: Frequency | Frequency[], time?: Time, velocity?: NormalRange): Tone.Sampler;
        /**
         * @param  {(Frequency|Frequency[])} notes	The note to release, or an array of notes.
         * @param  {Time=} time     	When to release the note.
         * @return {Tone.Sampler}	this
         */
        triggerRelease(notes: Frequency | Frequency[], time?: Time): Tone.Sampler;
        /**
         * Release all currently active notes.
         * @param  {Time=} time     	When to release the notes.
         * @return {Tone.Sampler}	this
         */
        releaseAll(time?: Time): Tone.Sampler;
        /**
         * Sync the instrument to the Transport. All subsequent calls of
         * [triggerAttack](#triggerattack) and [triggerRelease](#triggerrelease)
         * will be scheduled along the transport.
         * @example
         * synth.sync()
         * //schedule 3 notes when the transport first starts
         * synth.triggerAttackRelease('8n', 0)
         * synth.triggerAttackRelease('8n', '8n')
         * synth.triggerAttackRelease('8n', '4n')
         * //start the transport to hear the notes
         * Transport.start()
         * @returns {Tone.Instrument} this
         */
        sync(): Tone.Instrument;
        /**
         * Invoke the attack phase, then after the duration, invoke the release.
         * @param  {(Frequency|Frequency[])} notes	The note to play and release, or an array of notes.
         * @param  {(Time|Time[])} duration The time the note should be held
         * @param  {Time=} time     When to start the attack
         * @param  {NormalRange} [velocity=1] The velocity of the attack
         * @return {Tone.Sampler}          this
         */
        triggerAttackRelease(notes: Frequency | Frequency[], duration: Time | Time[], time?: Time, velocity?: NormalRange): Tone.Sampler;
        /**
         *  Add a note to the sampler.
         *  @param  {Note|Midi}   note      The buffer's pitch.
         *  @param  {String|Tone.Buffer|Audiobuffer}  url  Either the url of the bufer,
         *                                                 or a buffer which will be added
         *                                                 with the given name.
         *  @param  {Function=}  callback  The callback to invoke
         *                                 when the url is loaded.
         */
        add(note: Note | Midi, url: string | Tone.Buffer | Audiobuffer, callback?: (...params: any[]) => any): void;
        /**
         * If the buffers are loaded or not
         * @memberOf Tone.Sampler#
         * @type {Boolean}
         * @name loaded
         * @readOnly
         */
        readonly loaded: boolean;
        /**
         * Clean up
         * @return {Tone.Sampler} this
         */
        dispose(): Tone.Sampler;
        /**
         * The volume of the output in decibels.
         * @type {Decibels}
         * @signal
         * @example
         * source.volume.value = -6;
         */
        volume: Decibels;
        /**
         * Unsync the instrument from the Transport
         * @returns {Tone.Instrument} this
         */
        unsync(): Tone.Instrument;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Tone.Synth is composed simply of a Tone.OmniOscillator
     *          routed through a Tone.AmplitudeEnvelope.
     *          <img src="https://docs.google.com/drawings/d/1-1_0YW2Z1J2EPI36P8fNCMcZG7N1w1GZluPs4og4evo/pub?w=1163&h=231">
     *
     *  @constructor
     *  @extends {Tone.Monophonic}
     *  @param {Object} [options] the options available for the synth
     *                          see defaults below
     *  @example
     * var synth = new Tone.Synth().toMaster();
     * synth.triggerAttackRelease("C4", "8n");
     */
    class Synth extends Tone.Monophonic {
        constructor(options?: any);
        /**
         *  The oscillator.
         *  @type {Tone.OmniOscillator}
         */
        oscillator: Tone.OmniOscillator;
        /**
         *  The frequency control.
         *  @type {Frequency}
         *  @signal
         */
        frequency: Frequency;
        /**
         *  The detune control.
         *  @type {Cents}
         *  @signal
         */
        detune: Cents;
        /**
         *  The amplitude envelope.
         *  @type {Tone.AmplitudeEnvelope}
         */
        envelope: Tone.AmplitudeEnvelope;
        /**
         *  @const
         *  @static
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         *  clean up
         *  @returns {Tone.Synth} this
         */
        dispose(): Tone.Synth;
        /**
         *  The glide time between notes.
         *  @type {Time}
         */
        portamento: Time;
        /**
         *  Trigger the attack of the note optionally with a given velocity.
         *
         *
         *  @param  {Frequency} note     The note to trigger.
         *  @param  {Time} [time=now]     When the note should start.
         *  @param  {number} [velocity=1] velocity The velocity scaler
         *                                determines how "loud" the note
         *                                will be triggered.
         *  @returns {Tone.Monophonic} this
         *  @example
         * synth.triggerAttack("C4");
         *  @example
         * //trigger the note a half second from now at half velocity
         * synth.triggerAttack("C4", "+0.5", 0.5);
         */
        triggerAttack(note: Frequency, time?: Time, velocity?: number): Tone.Monophonic;
        /**
         *  Trigger the release portion of the envelope
         *  @param  {Time} [time=now] If no time is given, the release happens immediatly
         *  @returns {Tone.Monophonic} this
         *  @example
         * synth.triggerRelease();
         */
        triggerRelease(time?: Time): Tone.Monophonic;
        /**
         *  Get the level of the output at the given time. Measures
         *  the envelope(s) value at the time.
         *  @param {Time} time The time to query the envelope value
         *  @return {NormalRange} The output level between 0-1
         */
        getLevelAtTime(time: Time): NormalRange;
        /**
         *  Set the note at the given time. If no time is given, the note
         *  will set immediately.
         *  @param {Frequency} note The note to change to.
         *  @param  {Time} [time=now] The time when the note should be set.
         *  @returns {Tone.Monophonic} this
         * @example
         * //change to F#6 in one quarter note from now.
         * synth.setNote("F#6", "+4n");
         * @example
         * //change to Bb4 right now
         * synth.setNote("Bb4");
         */
        setNote(note: Frequency, time?: Time): Tone.Monophonic;
        /**
         * The volume of the output in decibels.
         * @type {Decibels}
         * @signal
         * @example
         * source.volume.value = -6;
         */
        volume: Decibels;
        /**
         * Sync the instrument to the Transport. All subsequent calls of
         * [triggerAttack](#triggerattack) and [triggerRelease](#triggerrelease)
         * will be scheduled along the transport.
         * @example
         * instrument.sync()
         * //schedule 3 notes when the transport first starts
         * instrument.triggerAttackRelease('C4', '8n', 0)
         * instrument.triggerAttackRelease('E4', '8n', '8n')
         * instrument.triggerAttackRelease('G4', '8n', '4n')
         * //start the transport to hear the notes
         * Transport.start()
         * @returns {Tone.Instrument} this
         */
        sync(): Tone.Instrument;
        /**
         * Unsync the instrument from the Transport
         * @returns {Tone.Instrument} this
         */
        unsync(): Tone.Instrument;
        /**
         *  Trigger the attack and then the release after the duration.
         *  @param  {Frequency} note     The note to trigger.
         *  @param  {Time} duration How long the note should be held for before
         *                          triggering the release. This value must be greater than 0.
         *  @param {Time} [time=now]  When the note should be triggered.
         *  @param  {NormalRange} [velocity=1] The velocity the note should be triggered at.
         *  @returns {Tone.Instrument} this
         *  @example
         * //trigger "C4" for the duration of an 8th note
         * synth.triggerAttackRelease("C4", "8n");
         */
        triggerAttackRelease(note: Frequency, duration: Time, time?: Time, velocity?: NormalRange): Tone.Instrument;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Return the absolute value of an incoming signal.
     *
     *  @constructor
     *  @extends {Tone.SignalBase}
     *  @example
     * var signal = new Tone.Signal(-1);
     * var abs = new Tone.Abs();
     * signal.connect(abs);
     * //the output of abs is 1.
     */
    class Abs extends Tone.SignalBase {
        /**
         *  dispose method
         *  @returns {Tone.Abs} this
         */
        dispose(): Tone.Abs;
        /**
         *  When signals connect to other signals or AudioParams,
         *  they take over the output value of that signal or AudioParam.
         *  For all other nodes, the behavior is the same as a default <code>connect</code>.
         *
         *  @override
         *  @param {AudioParam|AudioNode|Tone.Signal|Tone} node
         *  @param {number} [outputNumber=0] The output number to connect from.
         *  @param {number} [inputNumber=0] The input number to connect to.
         *  @returns {Tone.SignalBase} this
         */
        connect(node: AudioParam | AudioNode | Tone.Signal | Tone, outputNumber?: number, inputNumber?: number): Tone.SignalBase;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Add a signal and a number or two signals. When no value is
     *         passed into the constructor, Tone.Add will sum <code>input[0]</code>
     *         and <code>input[1]</code>. If a value is passed into the constructor,
     *         the it will be added to the input.
     *
     *  @constructor
     *  @extends {Tone.Signal}
     *  @param {number=} value If no value is provided, Tone.Add will sum the first
     *                         and second inputs.
     *  @example
     * var signal = new Tone.Signal(2);
     * var add = new Tone.Add(2);
     * signal.connect(add);
     * //the output of add equals 4
     *  @example
     * //if constructed with no arguments
     * //it will add the first and second inputs
     * var add = new Tone.Add();
     * var sig0 = new Tone.Signal(3).connect(add, 0, 0);
     * var sig1 = new Tone.Signal(4).connect(add, 0, 1);
     * //the output of add equals 7.
     */
    class Add extends Tone.Signal {
        constructor(value?: number);
        /**
         *  Clean up.
         *  @returns {Tone.Add} this
         */
        dispose(): Tone.Add;
        /**
         * Return the current signal value at the given time.
         * @param  {Time} time When to get the signal value
         * @return {Number}
         */
        getValueAtTime(time: Time): number;
        /**
         *  The units of the parameter
         *  @type {Tone.Type}
         */
        units: Tone.Type;
        /**
         *  If the value should be converted or not
         *  @type {Boolean}
         */
        convert: boolean;
        /**
         * The current value of the parameter.
         * @memberOf Tone.Param#
         * @type {Number}
         * @name value
         */
        value: number;
        /**
         *  Schedules a parameter value change at the given time.
         *  @param {*}	value The value to set the signal.
         *  @param {Time}  time The time when the change should occur.
         *  @returns {Tone.Param} this
         *  @example
         * //set the frequency to "G4" in exactly 1 second from now.
         * freq.setValueAtTime("G4", "+1");
         */
        setValueAtTime(value: any, time: Time): Tone.Param;
        /**
         *  Creates a schedule point with the current value at the current time.
         *  This is useful for creating an automation anchor point in order to
         *  schedule changes from the current value.
         *
         *  @param {number=} now (Optionally) pass the now value in.
         *  @returns {Tone.Param} this
         */
        setRampPoint(now?: number): Tone.Param;
        /**
         *  Schedules a linear continuous change in parameter value from the
         *  previous scheduled parameter value to the given value.
         *
         *  @param  {number} value
         *  @param  {Time} endTime
         *  @returns {Tone.Param} this
         */
        linearRampToValueAtTime(value: number, endTime: Time): Tone.Param;
        /**
         *  Schedules an exponential continuous change in parameter value from
         *  the previous scheduled parameter value to the given value.
         *
         *  @param  {number} value
         *  @param  {Time} endTime
         *  @returns {Tone.Param} this
         */
        exponentialRampToValueAtTime(value: number, endTime: Time): Tone.Param;
        /**
         *  Schedules an exponential continuous change in parameter value from
         *  the current time and current value to the given value over the
         *  duration of the rampTime.
         *
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        exponentialRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Schedules an linear continuous change in parameter value from
         *  the current time and current value to the given value over the
         *  duration of the rampTime.
         *
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //linearly ramp to the value 4 over 3 seconds.
         * signal.linearRampTo(4, 3);
         */
        linearRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time. Since it
         *  is an exponential approach it will continue approaching after the ramp duration. The
         *  rampTime is the time that it takes to reach over 99% of the way towards the value.
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        targetRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time. Since it
         *  is an exponential approach it will continue approaching after the ramp duration. The
         *  rampTime is the time that it takes to reach over 99% of the way towards the value. This methods
         *  is similar to setTargetAtTime except the third argument is a time instead of a 'timeConstant'
         *  @param  {number} value   The value to ramp to.
         *  @param {Time}	time 	When the ramp should start.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        exponentialApproachValueAtTime(value: number, time: Time, rampTime: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time with
         *  a rate having the given time constant.
         *  @param {number} value
         *  @param {Time} startTime
         *  @param {number} timeConstant
         *  @returns {Tone.Param} this
         */
        setTargetAtTime(value: number, startTime: Time, timeConstant: number): Tone.Param;
        /**
         *  Sets an array of arbitrary parameter values starting at the given time
         *  for the given duration.
         *
         *  @param {Array} values
         *  @param {Time} startTime
         *  @param {Time} duration
         *  @param {NormalRange} [scaling=1] If the values in the curve should be scaled by some value
         *  @returns {Tone.Param} this
         */
        setValueCurveAtTime(values: Array, startTime: Time, duration: Time, scaling?: NormalRange): Tone.Param;
        /**
         *  Cancels all scheduled parameter changes with times greater than or
         *  equal to startTime.
         *
         *  @param  {Time} time
         *  @returns {Tone.Param} this
         */
        cancelScheduledValues(time: Time): Tone.Param;
        /**
         *  This is similar to [cancelScheduledValues](#cancelScheduledValues) except
         *  it holds the automated value at time until the next automated event.
         *  @param  {Time} time
         *  @returns {Tone.Param} this
         */
        cancelAndHoldAtTime(time: Time): Tone.Param;
        /**
         *  Ramps to the given value over the duration of the rampTime.
         *  Automatically selects the best ramp type (exponential or linear)
         *  depending on the `units` of the signal
         *
         *  @param  {number} value
         *  @param  {Time} rampTime 	The time that it takes the
         *                              value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //ramp to the value either linearly or exponentially
         * //depending on the "units" value of the signal
         * signal.rampTo(0, 10);
         *  @example
         * //schedule it to ramp starting at a specific time
         * signal.rampTo(0, 10, 5)
         */
        rampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class AudioToGain converts an input in AudioRange [-1,1] to NormalRange [0,1].
     *         See Tone.GainToAudio.
     *
     *  @extends {Tone.SignalBase}
     *  @constructor
     *  @example
     *  var a2g = new Tone.AudioToGain();
     */
    class AudioToGain extends Tone.SignalBase {
        /**
         *  clean up
         *  @returns {Tone.AudioToGain} this
         */
        dispose(): Tone.AudioToGain;
        /**
         *  When signals connect to other signals or AudioParams,
         *  they take over the output value of that signal or AudioParam.
         *  For all other nodes, the behavior is the same as a default <code>connect</code>.
         *
         *  @override
         *  @param {AudioParam|AudioNode|Tone.Signal|Tone} node
         *  @param {number} [outputNumber=0] The output number to connect from.
         *  @param {number} [inputNumber=0] The input number to connect to.
         *  @returns {Tone.SignalBase} this
         */
        connect(node: AudioParam | AudioNode | Tone.Signal | Tone, outputNumber?: number, inputNumber?: number): Tone.SignalBase;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Convert an incoming signal between 0, 1 to an equal power gain scale.
     *
     *  @extends {Tone.SignalBase}
     *  @constructor
     *  @example
     * var eqPowGain = new Tone.EqualPowerGain();
     */
    class EqualPowerGain extends Tone.SignalBase {
        /**
         *  clean up
         *  @returns {Tone.EqualPowerGain} this
         */
        dispose(): Tone.EqualPowerGain;
        /**
         *  When signals connect to other signals or AudioParams,
         *  they take over the output value of that signal or AudioParam.
         *  For all other nodes, the behavior is the same as a default <code>connect</code>.
         *
         *  @override
         *  @param {AudioParam|AudioNode|Tone.Signal|Tone} node
         *  @param {number} [outputNumber=0] The output number to connect from.
         *  @param {number} [inputNumber=0] The input number to connect to.
         *  @returns {Tone.SignalBase} this
         */
        connect(node: AudioParam | AudioNode | Tone.Signal | Tone, outputNumber?: number, inputNumber?: number): Tone.SignalBase;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Maps a NormalRange [0, 1] to an AudioRange [-1, 1].
     *         See also Tone.AudioToGain.
     *
     *  @extends {Tone.SignalBase}
     *  @constructor
     *  @example
     * var g2a = new Tone.GainToAudio();
     */
    class GainToAudio extends Tone.SignalBase {
        /**
         *  clean up
         *  @returns {Tone.GainToAudio} this
         */
        dispose(): Tone.GainToAudio;
        /**
         *  When signals connect to other signals or AudioParams,
         *  they take over the output value of that signal or AudioParam.
         *  For all other nodes, the behavior is the same as a default <code>connect</code>.
         *
         *  @override
         *  @param {AudioParam|AudioNode|Tone.Signal|Tone} node
         *  @param {number} [outputNumber=0] The output number to connect from.
         *  @param {number} [inputNumber=0] The input number to connect to.
         *  @returns {Tone.SignalBase} this
         */
        connect(node: AudioParam | AudioNode | Tone.Signal | Tone, outputNumber?: number, inputNumber?: number): Tone.SignalBase;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Output 1 if the signal is greater than the value, otherwise outputs 0.
     *          can compare two signals or a signal and a number.
     *
     *  @constructor
     *  @extends {Tone.Signal}
     *  @param {number} [value=0] the value to compare to the incoming signal
     *  @example
     * var gt = new Tone.GreaterThan(2);
     * var sig = new Tone.Signal(4).connect(gt);
     * //output of gt is equal 1.
     */
    class GreaterThan extends Tone.Signal {
        constructor(value?: number);
        /**
         *  dispose method
         *  @returns {Tone.GreaterThan} this
         */
        dispose(): Tone.GreaterThan;
        /**
         * Return the current signal value at the given time.
         * @param  {Time} time When to get the signal value
         * @return {Number}
         */
        getValueAtTime(time: Time): number;
        /**
         *  The units of the parameter
         *  @type {Tone.Type}
         */
        units: Tone.Type;
        /**
         *  If the value should be converted or not
         *  @type {Boolean}
         */
        convert: boolean;
        /**
         * The current value of the parameter.
         * @memberOf Tone.Param#
         * @type {Number}
         * @name value
         */
        value: number;
        /**
         *  Schedules a parameter value change at the given time.
         *  @param {*}	value The value to set the signal.
         *  @param {Time}  time The time when the change should occur.
         *  @returns {Tone.Param} this
         *  @example
         * //set the frequency to "G4" in exactly 1 second from now.
         * freq.setValueAtTime("G4", "+1");
         */
        setValueAtTime(value: any, time: Time): Tone.Param;
        /**
         *  Creates a schedule point with the current value at the current time.
         *  This is useful for creating an automation anchor point in order to
         *  schedule changes from the current value.
         *
         *  @param {number=} now (Optionally) pass the now value in.
         *  @returns {Tone.Param} this
         */
        setRampPoint(now?: number): Tone.Param;
        /**
         *  Schedules a linear continuous change in parameter value from the
         *  previous scheduled parameter value to the given value.
         *
         *  @param  {number} value
         *  @param  {Time} endTime
         *  @returns {Tone.Param} this
         */
        linearRampToValueAtTime(value: number, endTime: Time): Tone.Param;
        /**
         *  Schedules an exponential continuous change in parameter value from
         *  the previous scheduled parameter value to the given value.
         *
         *  @param  {number} value
         *  @param  {Time} endTime
         *  @returns {Tone.Param} this
         */
        exponentialRampToValueAtTime(value: number, endTime: Time): Tone.Param;
        /**
         *  Schedules an exponential continuous change in parameter value from
         *  the current time and current value to the given value over the
         *  duration of the rampTime.
         *
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        exponentialRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Schedules an linear continuous change in parameter value from
         *  the current time and current value to the given value over the
         *  duration of the rampTime.
         *
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //linearly ramp to the value 4 over 3 seconds.
         * signal.linearRampTo(4, 3);
         */
        linearRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time. Since it
         *  is an exponential approach it will continue approaching after the ramp duration. The
         *  rampTime is the time that it takes to reach over 99% of the way towards the value.
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        targetRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time. Since it
         *  is an exponential approach it will continue approaching after the ramp duration. The
         *  rampTime is the time that it takes to reach over 99% of the way towards the value. This methods
         *  is similar to setTargetAtTime except the third argument is a time instead of a 'timeConstant'
         *  @param  {number} value   The value to ramp to.
         *  @param {Time}	time 	When the ramp should start.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        exponentialApproachValueAtTime(value: number, time: Time, rampTime: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time with
         *  a rate having the given time constant.
         *  @param {number} value
         *  @param {Time} startTime
         *  @param {number} timeConstant
         *  @returns {Tone.Param} this
         */
        setTargetAtTime(value: number, startTime: Time, timeConstant: number): Tone.Param;
        /**
         *  Sets an array of arbitrary parameter values starting at the given time
         *  for the given duration.
         *
         *  @param {Array} values
         *  @param {Time} startTime
         *  @param {Time} duration
         *  @param {NormalRange} [scaling=1] If the values in the curve should be scaled by some value
         *  @returns {Tone.Param} this
         */
        setValueCurveAtTime(values: Array, startTime: Time, duration: Time, scaling?: NormalRange): Tone.Param;
        /**
         *  Cancels all scheduled parameter changes with times greater than or
         *  equal to startTime.
         *
         *  @param  {Time} time
         *  @returns {Tone.Param} this
         */
        cancelScheduledValues(time: Time): Tone.Param;
        /**
         *  This is similar to [cancelScheduledValues](#cancelScheduledValues) except
         *  it holds the automated value at time until the next automated event.
         *  @param  {Time} time
         *  @returns {Tone.Param} this
         */
        cancelAndHoldAtTime(time: Time): Tone.Param;
        /**
         *  Ramps to the given value over the duration of the rampTime.
         *  Automatically selects the best ramp type (exponential or linear)
         *  depending on the `units` of the signal
         *
         *  @param  {number} value
         *  @param  {Time} rampTime 	The time that it takes the
         *                              value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //ramp to the value either linearly or exponentially
         * //depending on the "units" value of the signal
         * signal.rampTo(0, 10);
         *  @example
         * //schedule it to ramp starting at a specific time
         * signal.rampTo(0, 10, 5)
         */
        rampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  GreaterThanZero outputs 1 when the input is strictly greater than zero
     *
     *  @constructor
     *  @extends {Tone.SignalBase}
     *  @example
     * var gt0 = new Tone.GreaterThanZero();
     * var sig = new Tone.Signal(0.01).connect(gt0);
     * //the output of gt0 is 1.
     * sig.value = 0;
     * //the output of gt0 is 0.
     */
    class GreaterThanZero extends Tone.SignalBase {
        /**
         *  dispose method
         *  @returns {Tone.GreaterThanZero} this
         */
        dispose(): Tone.GreaterThanZero;
        /**
         *  When signals connect to other signals or AudioParams,
         *  they take over the output value of that signal or AudioParam.
         *  For all other nodes, the behavior is the same as a default <code>connect</code>.
         *
         *  @override
         *  @param {AudioParam|AudioNode|Tone.Signal|Tone} node
         *  @param {number} [outputNumber=0] The output number to connect from.
         *  @param {number} [inputNumber=0] The input number to connect to.
         *  @returns {Tone.SignalBase} this
         */
        connect(node: AudioParam | AudioNode | Tone.Signal | Tone, outputNumber?: number, inputNumber?: number): Tone.SignalBase;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Signal-rate modulo operator. Only works in AudioRange [-1, 1] and for modulus
     *         values in the NormalRange.
     *
     *  @constructor
     *  @extends {Tone.SignalBase}
     *  @param {NormalRange} modulus The modulus to apply.
     *  @example
     * var mod = new Tone.Modulo(0.2)
     * var sig = new Tone.Signal(0.5).connect(mod);
     * //mod outputs 0.1
     */
    class Modulo extends Tone.SignalBase {
        constructor(modulus: NormalRange);
        /**
         * The modulus value.
         * @memberOf Tone.Modulo#
         * @type {NormalRange}
         * @name value
         */
        value: NormalRange;
        /**
         * clean up
         *  @returns {Tone.Modulo} this
         */
        dispose(): Tone.Modulo;
        /**
         *  When signals connect to other signals or AudioParams,
         *  they take over the output value of that signal or AudioParam.
         *  For all other nodes, the behavior is the same as a default <code>connect</code>.
         *
         *  @override
         *  @param {AudioParam|AudioNode|Tone.Signal|Tone} node
         *  @param {number} [outputNumber=0] The output number to connect from.
         *  @param {number} [inputNumber=0] The input number to connect to.
         *  @returns {Tone.SignalBase} this
         */
        connect(node: AudioParam | AudioNode | Tone.Signal | Tone, outputNumber?: number, inputNumber?: number): Tone.SignalBase;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Multiply two incoming signals. Or, if a number is given in the constructor,
     *          multiplies the incoming signal by that value.
     *
     *  @constructor
     *  @extends {Tone.Signal}
     *  @param {number=} value Constant value to multiple. If no value is provided,
     *                         it will return the product of the first and second inputs
     *  @example
     * var mult = new Tone.Multiply();
     * var sigA = new Tone.Signal(3);
     * var sigB = new Tone.Signal(4);
     * sigA.connect(mult, 0, 0);
     * sigB.connect(mult, 0, 1);
     * //output of mult is 12.
     *  @example
     * var mult = new Tone.Multiply(10);
     * var sig = new Tone.Signal(2).connect(mult);
     * //the output of mult is 20.
     */
    class Multiply extends Tone.Signal {
        constructor(value?: number);
        /**
         *  clean up
         *  @returns {Tone.Multiply} this
         */
        dispose(): Tone.Multiply;
        /**
         * Return the current signal value at the given time.
         * @param  {Time} time When to get the signal value
         * @return {Number}
         */
        getValueAtTime(time: Time): number;
        /**
         *  The units of the parameter
         *  @type {Tone.Type}
         */
        units: Tone.Type;
        /**
         *  If the value should be converted or not
         *  @type {Boolean}
         */
        convert: boolean;
        /**
         * The current value of the parameter.
         * @memberOf Tone.Param#
         * @type {Number}
         * @name value
         */
        value: number;
        /**
         *  Schedules a parameter value change at the given time.
         *  @param {*}	value The value to set the signal.
         *  @param {Time}  time The time when the change should occur.
         *  @returns {Tone.Param} this
         *  @example
         * //set the frequency to "G4" in exactly 1 second from now.
         * freq.setValueAtTime("G4", "+1");
         */
        setValueAtTime(value: any, time: Time): Tone.Param;
        /**
         *  Creates a schedule point with the current value at the current time.
         *  This is useful for creating an automation anchor point in order to
         *  schedule changes from the current value.
         *
         *  @param {number=} now (Optionally) pass the now value in.
         *  @returns {Tone.Param} this
         */
        setRampPoint(now?: number): Tone.Param;
        /**
         *  Schedules a linear continuous change in parameter value from the
         *  previous scheduled parameter value to the given value.
         *
         *  @param  {number} value
         *  @param  {Time} endTime
         *  @returns {Tone.Param} this
         */
        linearRampToValueAtTime(value: number, endTime: Time): Tone.Param;
        /**
         *  Schedules an exponential continuous change in parameter value from
         *  the previous scheduled parameter value to the given value.
         *
         *  @param  {number} value
         *  @param  {Time} endTime
         *  @returns {Tone.Param} this
         */
        exponentialRampToValueAtTime(value: number, endTime: Time): Tone.Param;
        /**
         *  Schedules an exponential continuous change in parameter value from
         *  the current time and current value to the given value over the
         *  duration of the rampTime.
         *
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        exponentialRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Schedules an linear continuous change in parameter value from
         *  the current time and current value to the given value over the
         *  duration of the rampTime.
         *
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //linearly ramp to the value 4 over 3 seconds.
         * signal.linearRampTo(4, 3);
         */
        linearRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time. Since it
         *  is an exponential approach it will continue approaching after the ramp duration. The
         *  rampTime is the time that it takes to reach over 99% of the way towards the value.
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        targetRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time. Since it
         *  is an exponential approach it will continue approaching after the ramp duration. The
         *  rampTime is the time that it takes to reach over 99% of the way towards the value. This methods
         *  is similar to setTargetAtTime except the third argument is a time instead of a 'timeConstant'
         *  @param  {number} value   The value to ramp to.
         *  @param {Time}	time 	When the ramp should start.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        exponentialApproachValueAtTime(value: number, time: Time, rampTime: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time with
         *  a rate having the given time constant.
         *  @param {number} value
         *  @param {Time} startTime
         *  @param {number} timeConstant
         *  @returns {Tone.Param} this
         */
        setTargetAtTime(value: number, startTime: Time, timeConstant: number): Tone.Param;
        /**
         *  Sets an array of arbitrary parameter values starting at the given time
         *  for the given duration.
         *
         *  @param {Array} values
         *  @param {Time} startTime
         *  @param {Time} duration
         *  @param {NormalRange} [scaling=1] If the values in the curve should be scaled by some value
         *  @returns {Tone.Param} this
         */
        setValueCurveAtTime(values: Array, startTime: Time, duration: Time, scaling?: NormalRange): Tone.Param;
        /**
         *  Cancels all scheduled parameter changes with times greater than or
         *  equal to startTime.
         *
         *  @param  {Time} time
         *  @returns {Tone.Param} this
         */
        cancelScheduledValues(time: Time): Tone.Param;
        /**
         *  This is similar to [cancelScheduledValues](#cancelScheduledValues) except
         *  it holds the automated value at time until the next automated event.
         *  @param  {Time} time
         *  @returns {Tone.Param} this
         */
        cancelAndHoldAtTime(time: Time): Tone.Param;
        /**
         *  Ramps to the given value over the duration of the rampTime.
         *  Automatically selects the best ramp type (exponential or linear)
         *  depending on the `units` of the signal
         *
         *  @param  {number} value
         *  @param  {Time} rampTime 	The time that it takes the
         *                              value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //ramp to the value either linearly or exponentially
         * //depending on the "units" value of the signal
         * signal.rampTo(0, 10);
         *  @example
         * //schedule it to ramp starting at a specific time
         * signal.rampTo(0, 10, 5)
         */
        rampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Negate the incoming signal. i.e. an input signal of 10 will output -10
     *
     *  @constructor
     *  @extends {Tone.SignalBase}
     *  @example
     * var neg = new Tone.Negate();
     * var sig = new Tone.Signal(-2).connect(neg);
     * //output of neg is positive 2.
     */
    class Negate extends Tone.SignalBase {
        /**
         *  clean up
         *  @returns {Tone.Negate} this
         */
        dispose(): Tone.Negate;
        /**
         *  When signals connect to other signals or AudioParams,
         *  they take over the output value of that signal or AudioParam.
         *  For all other nodes, the behavior is the same as a default <code>connect</code>.
         *
         *  @override
         *  @param {AudioParam|AudioNode|Tone.Signal|Tone} node
         *  @param {number} [outputNumber=0] The output number to connect from.
         *  @param {number} [inputNumber=0] The input number to connect to.
         *  @returns {Tone.SignalBase} this
         */
        connect(node: AudioParam | AudioNode | Tone.Signal | Tone, outputNumber?: number, inputNumber?: number): Tone.SignalBase;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Normalize takes an input min and max and maps it linearly to NormalRange [0,1]
     *
     *  @extends {Tone.SignalBase}
     *  @constructor
     *  @param {number} inputMin the min input value
     *  @param {number} inputMax the max input value
     *  @example
     * var norm = new Tone.Normalize(2, 4);
     * var sig = new Tone.Signal(3).connect(norm);
     * //output of norm is 0.5.
     */
    class Normalize extends Tone.SignalBase {
        constructor(inputMin: number, inputMax: number);
        /**
         * The minimum value the input signal will reach.
         * @memberOf Tone.Normalize#
         * @type {number}
         * @name min
         */
        min: number;
        /**
         * The maximum value the input signal will reach.
         * @memberOf Tone.Normalize#
         * @type {number}
         * @name max
         */
        max: number;
        /**
         *  clean up
         *  @returns {Tone.Normalize} this
         */
        dispose(): Tone.Normalize;
        /**
         *  When signals connect to other signals or AudioParams,
         *  they take over the output value of that signal or AudioParam.
         *  For all other nodes, the behavior is the same as a default <code>connect</code>.
         *
         *  @override
         *  @param {AudioParam|AudioNode|Tone.Signal|Tone} node
         *  @param {number} [outputNumber=0] The output number to connect from.
         *  @param {number} [inputNumber=0] The input number to connect to.
         *  @returns {Tone.SignalBase} this
         */
        connect(node: AudioParam | AudioNode | Tone.Signal | Tone, outputNumber?: number, inputNumber?: number): Tone.SignalBase;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Pow applies an exponent to the incoming signal. The incoming signal
     *         must be AudioRange.
     *
     *  @extends {Tone.SignalBase}
     *  @constructor
     *  @param {Positive} exp The exponent to apply to the incoming signal, must be at least 2.
     *  @example
     * var pow = new Tone.Pow(2);
     * var sig = new Tone.Signal(0.5).connect(pow);
     * //output of pow is 0.25.
     */
    class Pow extends Tone.SignalBase {
        constructor(exp: Positive);
        /**
         * The value of the exponent.
         * @memberOf Tone.Pow#
         * @type {number}
         * @name value
         */
        value: number;
        /**
         *  Clean up.
         *  @returns {Tone.Pow} this
         */
        dispose(): Tone.Pow;
        /**
         *  When signals connect to other signals or AudioParams,
         *  they take over the output value of that signal or AudioParam.
         *  For all other nodes, the behavior is the same as a default <code>connect</code>.
         *
         *  @override
         *  @param {AudioParam|AudioNode|Tone.Signal|Tone} node
         *  @param {number} [outputNumber=0] The output number to connect from.
         *  @param {number} [inputNumber=0] The input number to connect to.
         *  @returns {Tone.SignalBase} this
         */
        connect(node: AudioParam | AudioNode | Tone.Signal | Tone, outputNumber?: number, inputNumber?: number): Tone.SignalBase;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Performs a linear scaling on an input signal.
     *          Scales a NormalRange input to between
     *          outputMin and outputMax.
     *
     *  @constructor
     *  @extends {Tone.SignalBase}
     *  @param {number} [outputMin=0] The output value when the input is 0.
     *  @param {number} [outputMax=1]	The output value when the input is 1.
     *  @example
     * var scale = new Tone.Scale(50, 100);
     * var signal = new Tone.Signal(0.5).connect(scale);
     * //the output of scale equals 75
     */
    class Scale extends Tone.SignalBase {
        constructor(outputMin?: number, outputMax?: number);
        /**
         * The minimum output value. This number is output when
         * the value input value is 0.
         * @memberOf Tone.Scale#
         * @type {number}
         * @name min
         */
        min: number;
        /**
         * The maximum output value. This number is output when
         * the value input value is 1.
         * @memberOf Tone.Scale#
         * @type {number}
         * @name max
         */
        max: number;
        /**
         *  Clean up.
         *  @returns {Tone.Scale} this
         */
        dispose(): Tone.Scale;
        /**
         *  When signals connect to other signals or AudioParams,
         *  they take over the output value of that signal or AudioParam.
         *  For all other nodes, the behavior is the same as a default <code>connect</code>.
         *
         *  @override
         *  @param {AudioParam|AudioNode|Tone.Signal|Tone} node
         *  @param {number} [outputNumber=0] The output number to connect from.
         *  @param {number} [inputNumber=0] The input number to connect to.
         *  @returns {Tone.SignalBase} this
         */
        connect(node: AudioParam | AudioNode | Tone.Signal | Tone, outputNumber?: number, inputNumber?: number): Tone.SignalBase;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Performs an exponential scaling on an input signal.
     *          Scales a NormalRange value [0,1] exponentially
     *          to the output range of outputMin to outputMax.
     *
     *  @constructor
     *  @extends {Tone.SignalBase}
     *  @param {number} [outputMin=0] The output value when the input is 0.
     *  @param {number} [outputMax=1]	The output value when the input is 1.
     *  @param {number} [exponent=2] The exponent which scales the incoming signal.
     *  @example
     * var scaleExp = new Tone.ScaleExp(0, 100, 2);
     * var signal = new Tone.Signal(0.5).connect(scaleExp);
     */
    class ScaleExp extends Tone.SignalBase {
        constructor(outputMin?: number, outputMax?: number, exponent?: number);
        /**
         * Instead of interpolating linearly between the <code>min</code> and
         * <code>max</code> values, setting the exponent will interpolate between
         * the two values with an exponential curve.
         * @memberOf Tone.ScaleExp#
         * @type {number}
         * @name exponent
         */
        exponent: number;
        /**
         * The minimum output value. This number is output when
         * the value input value is 0.
         * @memberOf Tone.ScaleExp#
         * @type {number}
         * @name min
         */
        min: number;
        /**
         * The maximum output value. This number is output when
         * the value input value is 1.
         * @memberOf Tone.ScaleExp#
         * @type {number}
         * @name max
         */
        max: number;
        /**
         *  Clean up.
         *  @returns {Tone.ScaleExp} this
         */
        dispose(): Tone.ScaleExp;
        /**
         *  When signals connect to other signals or AudioParams,
         *  they take over the output value of that signal or AudioParam.
         *  For all other nodes, the behavior is the same as a default <code>connect</code>.
         *
         *  @override
         *  @param {AudioParam|AudioNode|Tone.Signal|Tone} node
         *  @param {number} [outputNumber=0] The output number to connect from.
         *  @param {number} [inputNumber=0] The input number to connect to.
         *  @returns {Tone.SignalBase} this
         */
        connect(node: AudioParam | AudioNode | Tone.Signal | Tone, outputNumber?: number, inputNumber?: number): Tone.SignalBase;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  A signal is an audio-rate value. Tone.Signal is a core component of the library.
     *          Unlike a number, Signals can be scheduled with sample-level accuracy. Tone.Signal
     *          has all of the methods available to native Web Audio
     *          [AudioParam](http://webaudio.github.io/web-audio-api/#the-audioparam-interface)
     *          as well as additional conveniences. Read more about working with signals
     *          [here](https://github.com/Tonejs/Tone.js/wiki/Signals).
     *
     *  @constructor
     *  @extends {Tone.Param}
     *  @param {Number|AudioParam} [value] Initial value of the signal. If an AudioParam
     *                                     is passed in, that parameter will be wrapped
     *                                     and controlled by the Signal.
     *  @param {string} [units=Number] unit The units the signal is in.
     *  @example
     * var signal = new Tone.Signal(10);
     */
    class Signal extends Tone.Param {
        constructor(value?: number | AudioParam, units?: string);
        /**
         *  The default values
         *  @type  {Object}
         *  @static
         *  @const
         */
        static readonly defaults: any;
        /**
         * Return the current signal value at the given time.
         * @param  {Time} time When to get the signal value
         * @return {Number}
         */
        getValueAtTime(time: Time): number;
        /**
         *  dispose and disconnect
         *  @returns {Tone.Signal} this
         */
        dispose(): Tone.Signal;
        /**
         *  The units of the parameter
         *  @type {Tone.Type}
         */
        units: Tone.Type;
        /**
         *  If the value should be converted or not
         *  @type {Boolean}
         */
        convert: boolean;
        /**
         * The current value of the parameter.
         * @memberOf Tone.Param#
         * @type {Number}
         * @name value
         */
        value: number;
        /**
         *  Schedules a parameter value change at the given time.
         *  @param {*}	value The value to set the signal.
         *  @param {Time}  time The time when the change should occur.
         *  @returns {Tone.Param} this
         *  @example
         * //set the frequency to "G4" in exactly 1 second from now.
         * freq.setValueAtTime("G4", "+1");
         */
        setValueAtTime(value: any, time: Time): Tone.Param;
        /**
         *  Creates a schedule point with the current value at the current time.
         *  This is useful for creating an automation anchor point in order to
         *  schedule changes from the current value.
         *
         *  @param {number=} now (Optionally) pass the now value in.
         *  @returns {Tone.Param} this
         */
        setRampPoint(now?: number): Tone.Param;
        /**
         *  Schedules a linear continuous change in parameter value from the
         *  previous scheduled parameter value to the given value.
         *
         *  @param  {number} value
         *  @param  {Time} endTime
         *  @returns {Tone.Param} this
         */
        linearRampToValueAtTime(value: number, endTime: Time): Tone.Param;
        /**
         *  Schedules an exponential continuous change in parameter value from
         *  the previous scheduled parameter value to the given value.
         *
         *  @param  {number} value
         *  @param  {Time} endTime
         *  @returns {Tone.Param} this
         */
        exponentialRampToValueAtTime(value: number, endTime: Time): Tone.Param;
        /**
         *  Schedules an exponential continuous change in parameter value from
         *  the current time and current value to the given value over the
         *  duration of the rampTime.
         *
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        exponentialRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Schedules an linear continuous change in parameter value from
         *  the current time and current value to the given value over the
         *  duration of the rampTime.
         *
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //linearly ramp to the value 4 over 3 seconds.
         * signal.linearRampTo(4, 3);
         */
        linearRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time. Since it
         *  is an exponential approach it will continue approaching after the ramp duration. The
         *  rampTime is the time that it takes to reach over 99% of the way towards the value.
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        targetRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time. Since it
         *  is an exponential approach it will continue approaching after the ramp duration. The
         *  rampTime is the time that it takes to reach over 99% of the way towards the value. This methods
         *  is similar to setTargetAtTime except the third argument is a time instead of a 'timeConstant'
         *  @param  {number} value   The value to ramp to.
         *  @param {Time}	time 	When the ramp should start.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        exponentialApproachValueAtTime(value: number, time: Time, rampTime: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time with
         *  a rate having the given time constant.
         *  @param {number} value
         *  @param {Time} startTime
         *  @param {number} timeConstant
         *  @returns {Tone.Param} this
         */
        setTargetAtTime(value: number, startTime: Time, timeConstant: number): Tone.Param;
        /**
         *  Sets an array of arbitrary parameter values starting at the given time
         *  for the given duration.
         *
         *  @param {Array} values
         *  @param {Time} startTime
         *  @param {Time} duration
         *  @param {NormalRange} [scaling=1] If the values in the curve should be scaled by some value
         *  @returns {Tone.Param} this
         */
        setValueCurveAtTime(values: Array, startTime: Time, duration: Time, scaling?: NormalRange): Tone.Param;
        /**
         *  Cancels all scheduled parameter changes with times greater than or
         *  equal to startTime.
         *
         *  @param  {Time} time
         *  @returns {Tone.Param} this
         */
        cancelScheduledValues(time: Time): Tone.Param;
        /**
         *  This is similar to [cancelScheduledValues](#cancelScheduledValues) except
         *  it holds the automated value at time until the next automated event.
         *  @param  {Time} time
         *  @returns {Tone.Param} this
         */
        cancelAndHoldAtTime(time: Time): Tone.Param;
        /**
         *  Ramps to the given value over the duration of the rampTime.
         *  Automatically selects the best ramp type (exponential or linear)
         *  depending on the `units` of the signal
         *
         *  @param  {number} value
         *  @param  {Time} rampTime 	The time that it takes the
         *                              value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //ramp to the value either linearly or exponentially
         * //depending on the "units" value of the signal
         * signal.rampTo(0, 10);
         *  @example
         * //schedule it to ramp starting at a specific time
         * signal.rampTo(0, 10, 5)
         */
        rampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Base class for all Signals. Used Internally.
     *
     *  @constructor
     *  @extends {Tone}
     */
    class SignalBase extends Tone {
        /**
         *  When signals connect to other signals or AudioParams,
         *  they take over the output value of that signal or AudioParam.
         *  For all other nodes, the behavior is the same as a default <code>connect</code>.
         *
         *  @override
         *  @param {AudioParam|AudioNode|Tone.Signal|Tone} node
         *  @param {number} [outputNumber=0] The output number to connect from.
         *  @param {number} [inputNumber=0] The input number to connect to.
         *  @returns {Tone.SignalBase} this
         */
        connect(node: AudioParam | AudioNode | Tone.Signal | Tone, outputNumber?: number, inputNumber?: number): Tone.SignalBase;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Subtract the signal connected to <code>input[1]</code> from the signal connected
     *         to <code>input[0]</code>. If an argument is provided in the constructor, the
     *         signals <code>.value</code> will be subtracted from the incoming signal.
     *
     *  @extends {Tone.Signal}
     *  @constructor
     *  @param {number=} value The value to subtract from the incoming signal. If the value
     *                         is omitted, it will subtract the second signal from the first.
     *  @example
     * var sub = new Tone.Subtract(1);
     * var sig = new Tone.Signal(4).connect(sub);
     * //the output of sub is 3.
     *  @example
     * var sub = new Tone.Subtract();
     * var sigA = new Tone.Signal(10);
     * var sigB = new Tone.Signal(2.5);
     * sigA.connect(sub, 0, 0);
     * sigB.connect(sub, 0, 1);
     * //output of sub is 7.5
     */
    class Subtract extends Tone.Signal {
        constructor(value?: number);
        /**
         *  Clean up.
         *  @returns {Tone.SignalBase} this
         */
        dispose(): Tone.SignalBase;
        /**
         * Return the current signal value at the given time.
         * @param  {Time} time When to get the signal value
         * @return {Number}
         */
        getValueAtTime(time: Time): number;
        /**
         *  The units of the parameter
         *  @type {Tone.Type}
         */
        units: Tone.Type;
        /**
         *  If the value should be converted or not
         *  @type {Boolean}
         */
        convert: boolean;
        /**
         * The current value of the parameter.
         * @memberOf Tone.Param#
         * @type {Number}
         * @name value
         */
        value: number;
        /**
         *  Schedules a parameter value change at the given time.
         *  @param {*}	value The value to set the signal.
         *  @param {Time}  time The time when the change should occur.
         *  @returns {Tone.Param} this
         *  @example
         * //set the frequency to "G4" in exactly 1 second from now.
         * freq.setValueAtTime("G4", "+1");
         */
        setValueAtTime(value: any, time: Time): Tone.Param;
        /**
         *  Creates a schedule point with the current value at the current time.
         *  This is useful for creating an automation anchor point in order to
         *  schedule changes from the current value.
         *
         *  @param {number=} now (Optionally) pass the now value in.
         *  @returns {Tone.Param} this
         */
        setRampPoint(now?: number): Tone.Param;
        /**
         *  Schedules a linear continuous change in parameter value from the
         *  previous scheduled parameter value to the given value.
         *
         *  @param  {number} value
         *  @param  {Time} endTime
         *  @returns {Tone.Param} this
         */
        linearRampToValueAtTime(value: number, endTime: Time): Tone.Param;
        /**
         *  Schedules an exponential continuous change in parameter value from
         *  the previous scheduled parameter value to the given value.
         *
         *  @param  {number} value
         *  @param  {Time} endTime
         *  @returns {Tone.Param} this
         */
        exponentialRampToValueAtTime(value: number, endTime: Time): Tone.Param;
        /**
         *  Schedules an exponential continuous change in parameter value from
         *  the current time and current value to the given value over the
         *  duration of the rampTime.
         *
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        exponentialRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Schedules an linear continuous change in parameter value from
         *  the current time and current value to the given value over the
         *  duration of the rampTime.
         *
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //linearly ramp to the value 4 over 3 seconds.
         * signal.linearRampTo(4, 3);
         */
        linearRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time. Since it
         *  is an exponential approach it will continue approaching after the ramp duration. The
         *  rampTime is the time that it takes to reach over 99% of the way towards the value.
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        targetRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time. Since it
         *  is an exponential approach it will continue approaching after the ramp duration. The
         *  rampTime is the time that it takes to reach over 99% of the way towards the value. This methods
         *  is similar to setTargetAtTime except the third argument is a time instead of a 'timeConstant'
         *  @param  {number} value   The value to ramp to.
         *  @param {Time}	time 	When the ramp should start.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        exponentialApproachValueAtTime(value: number, time: Time, rampTime: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time with
         *  a rate having the given time constant.
         *  @param {number} value
         *  @param {Time} startTime
         *  @param {number} timeConstant
         *  @returns {Tone.Param} this
         */
        setTargetAtTime(value: number, startTime: Time, timeConstant: number): Tone.Param;
        /**
         *  Sets an array of arbitrary parameter values starting at the given time
         *  for the given duration.
         *
         *  @param {Array} values
         *  @param {Time} startTime
         *  @param {Time} duration
         *  @param {NormalRange} [scaling=1] If the values in the curve should be scaled by some value
         *  @returns {Tone.Param} this
         */
        setValueCurveAtTime(values: Array, startTime: Time, duration: Time, scaling?: NormalRange): Tone.Param;
        /**
         *  Cancels all scheduled parameter changes with times greater than or
         *  equal to startTime.
         *
         *  @param  {Time} time
         *  @returns {Tone.Param} this
         */
        cancelScheduledValues(time: Time): Tone.Param;
        /**
         *  This is similar to [cancelScheduledValues](#cancelScheduledValues) except
         *  it holds the automated value at time until the next automated event.
         *  @param  {Time} time
         *  @returns {Tone.Param} this
         */
        cancelAndHoldAtTime(time: Time): Tone.Param;
        /**
         *  Ramps to the given value over the duration of the rampTime.
         *  Automatically selects the best ramp type (exponential or linear)
         *  depending on the `units` of the signal
         *
         *  @param  {number} value
         *  @param  {Time} rampTime 	The time that it takes the
         *                              value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //ramp to the value either linearly or exponentially
         * //depending on the "units" value of the signal
         * signal.rampTo(0, 10);
         *  @example
         * //schedule it to ramp starting at a specific time
         * signal.rampTo(0, 10, 5)
         */
        rampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     * @class Tone.TickSignal extends Tone.Signal, but adds the capability
     *        to calculate the number of elapsed ticks. exponential and target curves
     *        are approximated with multiple linear ramps.
     *
     *        Thank you Bruno Dias, H. Sofia Pinto, and David M. Matos, for your [WAC paper](https://smartech.gatech.edu/bitstream/handle/1853/54588/WAC2016-49.pdf)
     *        describing integrating timing functions for tempo calculations.
     *
     * @param {Number} value The initial value of the signal
     * @extends {Tone.Signal}
     */
    class TickSignal extends Tone.Signal {
        constructor(value: number);
        /**
         *  Start exponentially approaching the target value at the given time with
         *  a rate having the given time constant.
         *  @param {number} value
         *  @param {Time} startTime
         *  @param {number} timeConstant
         *  @returns {Tone.TickSignal} this
         */
        setTargetAtTime(value: number, startTime: Time, timeConstant: number): Tone.TickSignal;
        /**
         *  Schedules an exponential continuous change in parameter value from
         *  the previous scheduled parameter value to the given value.
         *  @param  {number} value
         *  @param  {Time} endTime
         *  @returns {Tone.TickSignal} this
         */
        exponentialRampToValueAtTime(value: number, endTime: Time): Tone.TickSignal;
        /**
         * Returns the tick value at the time. Takes into account
         * any automation curves scheduled on the signal.
         * @param  {Time} time The time to get the tick count at
         * @return {Ticks}      The number of ticks which have elapsed at the time
         *                          given any automations.
         */
        getTicksAtTime(time: Time): Ticks;
        /**
         * Return the elapsed time of the number of ticks from the given time
         * @param {Ticks} ticks The number of ticks to calculate
         * @param  {Time} time The time to get the next tick from
         * @return {Seconds} The duration of the number of ticks from the given time in seconds
         */
        getDurationOfTicks(ticks: Ticks, time: Time): Seconds;
        /**
         * Given a tick, returns the time that tick occurs at.
         * @param  {Ticks} tick
         * @return {Time}      The time that the tick occurs.
         */
        getTimeOfTick(tick: Ticks): Time;
        /**
         * Convert some number of ticks their the duration in seconds accounting
         * for any automation curves starting at the given time.
         * @param  {Ticks} ticks The number of ticks to convert to seconds.
         * @param  {Time} [when=now]  When along the automation timeline to convert the ticks.
         * @return {Tone.Time}       The duration in seconds of the ticks.
         */
        ticksToTime(ticks: Ticks, when?: Time): Tone.Time;
        /**
         * The inverse of [ticksToTime](#tickstotime). Convert a duration in
         * seconds to the corresponding number of ticks accounting for any
         * automation curves starting at the given time.
         * @param  {Time} duration The time interval to convert to ticks.
         * @param  {Time} [when=now]     When along the automation timeline to convert the ticks.
         * @return {Tone.Ticks}          The duration in ticks.
         */
        timeToTicks(duration: Time, when?: Time): Tone.Ticks;
        /**
         * Return the current signal value at the given time.
         * @param  {Time} time When to get the signal value
         * @return {Number}
         */
        getValueAtTime(time: Time): number;
        /**
         *  dispose and disconnect
         *  @returns {Tone.Signal} this
         */
        dispose(): Tone.Signal;
        /**
         *  The units of the parameter
         *  @type {Tone.Type}
         */
        units: Tone.Type;
        /**
         *  If the value should be converted or not
         *  @type {Boolean}
         */
        convert: boolean;
        /**
         * The current value of the parameter.
         * @memberOf Tone.Param#
         * @type {Number}
         * @name value
         */
        value: number;
        /**
         *  Schedules a parameter value change at the given time.
         *  @param {*}	value The value to set the signal.
         *  @param {Time}  time The time when the change should occur.
         *  @returns {Tone.Param} this
         *  @example
         * //set the frequency to "G4" in exactly 1 second from now.
         * freq.setValueAtTime("G4", "+1");
         */
        setValueAtTime(value: any, time: Time): Tone.Param;
        /**
         *  Creates a schedule point with the current value at the current time.
         *  This is useful for creating an automation anchor point in order to
         *  schedule changes from the current value.
         *
         *  @param {number=} now (Optionally) pass the now value in.
         *  @returns {Tone.Param} this
         */
        setRampPoint(now?: number): Tone.Param;
        /**
         *  Schedules a linear continuous change in parameter value from the
         *  previous scheduled parameter value to the given value.
         *
         *  @param  {number} value
         *  @param  {Time} endTime
         *  @returns {Tone.Param} this
         */
        linearRampToValueAtTime(value: number, endTime: Time): Tone.Param;
        /**
         *  Schedules an exponential continuous change in parameter value from
         *  the current time and current value to the given value over the
         *  duration of the rampTime.
         *
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        exponentialRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Schedules an linear continuous change in parameter value from
         *  the current time and current value to the given value over the
         *  duration of the rampTime.
         *
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //linearly ramp to the value 4 over 3 seconds.
         * signal.linearRampTo(4, 3);
         */
        linearRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time. Since it
         *  is an exponential approach it will continue approaching after the ramp duration. The
         *  rampTime is the time that it takes to reach over 99% of the way towards the value.
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        targetRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time. Since it
         *  is an exponential approach it will continue approaching after the ramp duration. The
         *  rampTime is the time that it takes to reach over 99% of the way towards the value. This methods
         *  is similar to setTargetAtTime except the third argument is a time instead of a 'timeConstant'
         *  @param  {number} value   The value to ramp to.
         *  @param {Time}	time 	When the ramp should start.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        exponentialApproachValueAtTime(value: number, time: Time, rampTime: Time): Tone.Param;
        /**
         *  Sets an array of arbitrary parameter values starting at the given time
         *  for the given duration.
         *
         *  @param {Array} values
         *  @param {Time} startTime
         *  @param {Time} duration
         *  @param {NormalRange} [scaling=1] If the values in the curve should be scaled by some value
         *  @returns {Tone.Param} this
         */
        setValueCurveAtTime(values: Array, startTime: Time, duration: Time, scaling?: NormalRange): Tone.Param;
        /**
         *  Cancels all scheduled parameter changes with times greater than or
         *  equal to startTime.
         *
         *  @param  {Time} time
         *  @returns {Tone.Param} this
         */
        cancelScheduledValues(time: Time): Tone.Param;
        /**
         *  This is similar to [cancelScheduledValues](#cancelScheduledValues) except
         *  it holds the automated value at time until the next automated event.
         *  @param  {Time} time
         *  @returns {Tone.Param} this
         */
        cancelAndHoldAtTime(time: Time): Tone.Param;
        /**
         *  Ramps to the given value over the duration of the rampTime.
         *  Automatically selects the best ramp type (exponential or linear)
         *  depending on the `units` of the signal
         *
         *  @param  {number} value
         *  @param  {Time} rampTime 	The time that it takes the
         *                              value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //ramp to the value either linearly or exponentially
         * //depending on the "units" value of the signal
         * signal.rampTo(0, 10);
         *  @example
         * //schedule it to ramp starting at a specific time
         * signal.rampTo(0, 10, 5)
         */
        rampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     * @class Tone.TransportTimelineSignal extends Tone.Signal, but adds the ability to synchronize the signal to the signal to the Tone.Transport
     * @extends {Tone.Signal}
     */
    class TransportTimelineSignal extends Tone.Signal {
        /**
         *  Get the scheduled value at the given time. This will
         *  return the unconverted (raw) value.
         *  @param  {TransportTime}  time  The time in seconds.
         *  @return  {Number}  The scheduled value at the given time.
         */
        getValueAtTime(time: TransportTime): number;
        /**
         * Set the output of the signal at the given time
         * @param  {Number} value The value to change to at the given time
         * @param  {TransportTime} time  The time to change the signal
         * @return {Tone.TransportTimelineSignal}       this
         */
        setValueAtTime(value: number, time: TransportTime): Tone.TransportTimelineSignal;
        /**
         * Linear ramp to the given value from the previous scheduled point to the given value
         * @param  {Number} value The value to change to at the given time
         * @param  {TransportTime} time  The time to change the signal
         * @return {Tone.TransportTimelineSignal}       this
         */
        linearRampToValueAtTime(value: number, time: TransportTime): Tone.TransportTimelineSignal;
        /**
         * Exponential ramp to the given value from the previous scheduled point to the given value
         * @param  {Number} value The value to change to at the given time
         * @param  {TransportTime} time  The time to change the signal
         * @return {Tone.TransportTimelineSignal}       this
         */
        exponentialRampToValueAtTime(value: number, time: TransportTime): Tone.TransportTimelineSignal;
        /**
         *  Start exponentially approaching the target value at the given time with
         *  a rate having the given time constant.
         *  @param {number} value
         *  @param {TransportTime} startTime
         *  @param {number} timeConstant
         * @return {Tone.TransportTimelineSignal}       this
         */
        setTargetAtTime(value: number, startTime: TransportTime, timeConstant: number): Tone.TransportTimelineSignal;
        /**
         *  Cancels all scheduled parameter changes with times greater than or
         *  equal to startTime.
         *  @param  {TransportTime} startTime
         *  @returns {Tone.Param} this
         */
        cancelScheduledValues(startTime: TransportTime): Tone.Param;
        /**
         *  Set an array of arbitrary values starting at the given time for the given duration.
         *  @param {Float32Array} values
         *  @param {Time} startTime
         *  @param {Time} duration
         *  @param {NormalRange} [scaling=1] If the values in the curve should be scaled by some value
         *  @returns {Tone.Signal} this
         */
        setValueCurveAtTime(values: Float32Array, startTime: Time, duration: Time, scaling?: NormalRange): Tone.Signal;
        /**
         *  This is similar to [cancelScheduledValues](#cancelScheduledValues) except
         *  it holds the automated value at time until the next automated event.
         *  @param  {Time} time
         *  @returns {Tone.TransportTimelineSignal} this
         */
        cancelAndHoldAtTime(time: Time): Tone.TransportTimelineSignal;
        /**
         * Dispose and disconnect
         * @return {Tone.TransportTimelineSignal} this
         */
        dispose(): Tone.TransportTimelineSignal;
        /**
         *  The units of the parameter
         *  @type {Tone.Type}
         */
        units: Tone.Type;
        /**
         *  If the value should be converted or not
         *  @type {Boolean}
         */
        convert: boolean;
        /**
         * The current value of the parameter.
         * @memberOf Tone.Param#
         * @type {Number}
         * @name value
         */
        value: number;
        /**
         *  Creates a schedule point with the current value at the current time.
         *  This is useful for creating an automation anchor point in order to
         *  schedule changes from the current value.
         *
         *  @param {number=} now (Optionally) pass the now value in.
         *  @returns {Tone.Param} this
         */
        setRampPoint(now?: number): Tone.Param;
        /**
         *  Schedules an exponential continuous change in parameter value from
         *  the current time and current value to the given value over the
         *  duration of the rampTime.
         *
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        exponentialRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Schedules an linear continuous change in parameter value from
         *  the current time and current value to the given value over the
         *  duration of the rampTime.
         *
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //linearly ramp to the value 4 over 3 seconds.
         * signal.linearRampTo(4, 3);
         */
        linearRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time. Since it
         *  is an exponential approach it will continue approaching after the ramp duration. The
         *  rampTime is the time that it takes to reach over 99% of the way towards the value.
         *  @param  {number} value   The value to ramp to.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        targetRampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         *  Start exponentially approaching the target value at the given time. Since it
         *  is an exponential approach it will continue approaching after the ramp duration. The
         *  rampTime is the time that it takes to reach over 99% of the way towards the value. This methods
         *  is similar to setTargetAtTime except the third argument is a time instead of a 'timeConstant'
         *  @param  {number} value   The value to ramp to.
         *  @param {Time}	time 	When the ramp should start.
         *  @param  {Time} rampTime the time that it takes the
         *                               value to ramp from it's current value
         *  @returns {Tone.Param} this
         *  @example
         * //exponentially ramp to the value 2 over 4 seconds.
         * signal.exponentialRampTo(2, 4);
         */
        exponentialApproachValueAtTime(value: number, time: Time, rampTime: Time): Tone.Param;
        /**
         *  Ramps to the given value over the duration of the rampTime.
         *  Automatically selects the best ramp type (exponential or linear)
         *  depending on the `units` of the signal
         *
         *  @param  {number} value
         *  @param  {Time} rampTime 	The time that it takes the
         *                              value to ramp from it's current value
         *  @param {Time}	[startTime=now] 	When the ramp should start.
         *  @returns {Tone.Param} this
         *  @example
         * //ramp to the value either linearly or exponentially
         * //depending on the "units" value of the signal
         * signal.rampTo(0, 10);
         *  @example
         * //schedule it to ramp starting at a specific time
         * signal.rampTo(0, 10, 5)
         */
        rampTo(value: number, rampTime: Time, startTime?: Time): Tone.Param;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Wraps the native Web Audio API
     *         [WaveShaperNode](http://webaudio.github.io/web-audio-api/#the-waveshapernode-interface).
     *
     *  @extends {Tone.SignalBase}
     *  @constructor
     *  @param {function|Array|Number} mapping The function used to define the values.
     *                                    The mapping function should take two arguments:
     *                                    the first is the value at the current position
     *                                    and the second is the array position.
     *                                    If the argument is an array, that array will be
     *                                    set as the wave shaping function. The input
     *                                    signal is an AudioRange [-1, 1] value and the output
     *                                    signal can take on any numerical values.
     *
     *  @param {Number} [bufferLen=1024] The length of the WaveShaperNode buffer.
     *  @example
     * var timesTwo = new Tone.WaveShaper(function(val){
     * 	return val * 2;
     * }, 2048);
     *  @example
     * //a waveshaper can also be constructed with an array of values
     * var invert = new Tone.WaveShaper([1, -1]);
     */
    class WaveShaper extends Tone.SignalBase {
        constructor(mapping: ((...params: any[]) => any) | Array | number, bufferLen?: number);
        /**
         *  Uses a mapping function to set the value of the curve.
         *  @param {function} mapping The function used to define the values.
         *                            The mapping function take two arguments:
         *                            the first is the value at the current position
         *                            which goes from -1 to 1 over the number of elements
         *                            in the curve array. The second argument is the array position.
         *  @returns {Tone.WaveShaper} this
         *  @example
         * //map the input signal from [-1, 1] to [0, 10]
         * shaper.setMap(function(val, index){
         * 	return (val + 1) * 5;
         * })
         */
        setMap(mapping: (...params: any[]) => any): Tone.WaveShaper;
        /**
         * The array to set as the waveshaper curve. For linear curves
         * array length does not make much difference, but for complex curves
         * longer arrays will provide smoother interpolation.
         * @memberOf Tone.WaveShaper#
         * @type {Array}
         * @name curve
         */
        curve: Array;
        /**
         * Specifies what type of oversampling (if any) should be used when
         * applying the shaping curve. Can either be "none", "2x" or "4x".
         * @memberOf Tone.WaveShaper#
         * @type {string}
         * @name oversample
         */
        oversample: string;
        /**
         *  Clean up.
         *  @returns {Tone.WaveShaper} this
         */
        dispose(): Tone.WaveShaper;
        /**
         *  When signals connect to other signals or AudioParams,
         *  they take over the output value of that signal or AudioParam.
         *  For all other nodes, the behavior is the same as a default <code>connect</code>.
         *
         *  @override
         *  @param {AudioParam|AudioNode|Tone.Signal|Tone} node
         *  @param {number} [outputNumber=0] The output number to connect from.
         *  @param {number} [inputNumber=0] The input number to connect to.
         *  @returns {Tone.SignalBase} this
         */
        connect(node: AudioParam | AudioNode | Tone.Signal | Tone, outputNumber?: number, inputNumber?: number): Tone.SignalBase;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.Zero outputs 0's at audio-rate. The reason this has to be
     *         it's own class is that many browsers optimize out Tone.Signal
     *         with a value of 0 and will not process nodes further down the graph.
     *  @extends {Tone.SignalBase}
     */
    class Zero extends Tone.SignalBase {
        /**
         *  clean up
         *  @return  {Tone.Zero}  this
         */
        dispose(): Tone.Zero;
        /**
         *  When signals connect to other signals or AudioParams,
         *  they take over the output value of that signal or AudioParam.
         *  For all other nodes, the behavior is the same as a default <code>connect</code>.
         *
         *  @override
         *  @param {AudioParam|AudioNode|Tone.Signal|Tone} node
         *  @param {number} [outputNumber=0] The output number to connect from.
         *  @param {number} [inputNumber=0] The input number to connect to.
         *  @returns {Tone.SignalBase} this
         */
        connect(node: AudioParam | AudioNode | Tone.Signal | Tone, outputNumber?: number, inputNumber?: number): Tone.SignalBase;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  The AudioContext
         *  @type {Tone.Context}
         *  @name context
         *  @memberOf Tone#
         *  @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.AMOscillator
     *
     *  @extends {Tone.Oscillator}
     *  @constructor
     *  @param {Frequency} frequency The starting frequency of the oscillator.
     *  @param {String} type The type of the carrier oscillator.
     *  @param {String} modulationType The type of the modulator oscillator.
     *  @example
     * //a sine oscillator frequency-modulated by a square wave
     * var fmOsc = new Tone.AMOscillator("Ab3", "sine", "square").toMaster().start();
     */
    class AMOscillator extends Tone.Oscillator {
        constructor(frequency: Frequency, type: string, modulationType: string);
        /**
         *  The oscillator's frequency
         *  @type {Frequency}
         *  @signal
         */
        frequency: Frequency;
        /**
         *  The detune control signal.
         *  @type {Cents}
         *  @signal
         */
        detune: Cents;
        /**
         *  Harmonicity is the frequency ratio between the carrier and the modulator oscillators.
         *  A harmonicity of 1 gives both oscillators the same frequency.
         *  Harmonicity = 2 means a change of an octave.
         *  @type {Positive}
         *  @signal
         *  @example
         * //pitch the modulator an octave below carrier
         * synth.harmonicity.value = 0.5;
         */
        harmonicity: Positive;
        /**
         *  default values
         *  @static
         *  @type {Object}
         *  @const
         */
        static readonly defaults: any;
        /**
         * The type of the carrier oscillator
         * @memberOf Tone.AMOscillator#
         * @type {string}
         * @name type
         */
        type: string;
        /**
         * The oscillator type without the partialsCount appended to the end
         * @memberOf Tone.AMOscillator#
         * @type {string}
         * @name baseType
         * @example
         * osc.type = 'sine2'
         * osc.baseType //'sine'
         * osc.partialCount = 2
         */
        baseType: string;
        /**
         * 'partialCount' offers an alternative way to set the number of used partials.
         * When partialCount is 0, the maximum number of partials are used when representing
         * the waveform using the periodicWave. When 'partials' is set, this value is
         * not settable, but equals the length of the partials array.
         * @memberOf Tone.AMOscillator#
         * @type {Number}
         * @name partialCount
         */
        partialCount: number;
        /**
         * The type of the modulator oscillator
         * @memberOf Tone.AMOscillator#
         * @type {string}
         * @name modulationType
         */
        modulationType: string;
        /**
         * The phase of the oscillator in degrees.
         * @memberOf Tone.AMOscillator#
         * @type {number}
         * @name phase
         */
        phase: number;
        /**
         * The partials of the carrier waveform. A partial represents
         * the amplitude at a harmonic. The first harmonic is the
         * fundamental frequency, the second is the octave and so on
         * following the harmonic series.
         * Setting this value will automatically set the type to "custom".
         * The value is an empty array when the type is not "custom".
         * @memberOf Tone.AMOscillator#
         * @type {Array}
         * @name partials
         * @example
         * osc.partials = [1, 0.2, 0.01];
         */
        partials: Array;
        /**
         *  Clean up.
         *  @return {Tone.AMOscillator} this
         */
        dispose(): Tone.AMOscillator;
        /**
         *  Sync the signal to the Transport's bpm. Any changes to the transports bpm,
         *  will also affect the oscillators frequency.
         *  @returns {Tone.Oscillator} this
         *  @example
         * Tone.Transport.bpm.value = 120;
         * osc.frequency.value = 440;
         * //the ration between the bpm and the frequency will be maintained
         * osc.syncFrequency();
         * Tone.Transport.bpm.value = 240;
         * // the frequency of the oscillator is doubled to 880
         */
        syncFrequency(): Tone.Oscillator;
        /**
         *  Unsync the oscillator's frequency from the Transport.
         *  See Tone.Oscillator.syncFrequency
         *  @returns {Tone.Oscillator} this
         */
        unsyncFrequency(): Tone.Oscillator;
        /**
         * Mute the output.
         * @memberOf Tone.Source#
         * @type {boolean}
         * @name mute
         * @example
         * //mute the output
         * source.mute = true;
         */
        mute: boolean;
        /**
         * The fadeIn time of the amplitude envelope.
         * @memberOf Tone.Source#
         * @type {Time}
         * @name fadeIn
         */
        fadeIn: Time;
        /**
         * The fadeOut time of the amplitude envelope.
         * @memberOf Tone.Source#
         * @type {Time}
         * @name fadeOut
         */
        fadeOut: Time;
        /**
         * The volume of the output in decibels.
         * @type {Decibels}
         * @signal
         * @example
         * source.volume.value = -6;
         */
        volume: Decibels;
        /**
         *  Returns the playback state of the source, either "started" or "stopped".
         *  @type {Tone.State}
         *  @readOnly
         *  @memberOf Tone.Source#
         *  @name state
         */
        readonly state: Tone.State;
        /**
         *  Start the source at the specified time. If no time is given,
         *  start the source now.
         *  @param  {Time} [time=now] When the source should be started.
         *  @returns {Tone.Source} this
         *  @example
         * source.start("+0.5"); //starts the source 0.5 seconds from now
         */
        start(time?: Time): Tone.Source;
        /**
         *  Stop the source at the specified time. If no time is given,
         *  stop the source now.
         *  @param  {Time} [time=now] When the source should be stopped.
         *  @returns {Tone.Source} this
         *  @example
         * source.stop(); // stops the source immediately
         */
        stop(time?: Time): Tone.Source;
        /**
         *  Sync the source to the Transport so that all subsequent
         *  calls to `start` and `stop` are synced to the TransportTime
         *  instead of the AudioContext time.
         *
         *  @returns {Tone.Source} this
         *  @example
         * //sync the source so that it plays between 0 and 0.3 on the Transport's timeline
         * source.sync().start(0).stop(0.3);
         * //start the transport.
         * Tone.Transport.start();
         *
         *  @example
         * //start the transport with an offset and the sync'ed sources
         * //will start in the correct position
         * source.sync().start(0.1);
         * //the source will be invoked with an offset of 0.4
         * Tone.Transport.start("+0.5", 0.5);
         */
        sync(): Tone.Source;
        /**
         *  Unsync the source to the Transport. See Tone.Source.sync
         *  @returns {Tone.Source} this
         */
        unsync(): Tone.Source;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Wrapper around the native BufferSourceNode.
     *  @extends {Tone.AudioNode}
     *  @param  {AudioBuffer|Tone.Buffer}  buffer   The buffer to play
     *  @param  {Function}  onload  The callback to invoke when the
     *                               buffer is done playing.
     */
    class BufferSource extends Tone.AudioNode {
        constructor(buffer: AudioBuffer | Tone.Buffer, onload: (...params: any[]) => any);
        /**
         *  The callback to invoke after the
         *  buffer source is done playing.
         *  @type  {Function}
         */
        onended: (...params: any[]) => any;
        /**
         *  The playbackRate of the buffer
         *  @type {Positive}
         *  @signal
         */
        playbackRate: Positive;
        /**
         *  The fadeIn time of the amplitude envelope.
         *  @type {Time}
         */
        fadeIn: Time;
        /**
         *  The fadeOut time of the amplitude envelope.
         *  @type {Time}
         */
        fadeOut: Time;
        /**
         * The curve applied to the fades, either "linear" or "exponential"
         * @type {String}
         */
        curve: string;
        /**
         *  The defaults
         *  @const
         *  @type  {Object}
         */
        static readonly defaults: any;
        /**
         *  Returns the playback state of the source, either "started" or "stopped".
         *  @type {Tone.State}
         *  @readOnly
         *  @memberOf Tone.BufferSource#
         *  @name state
         */
        readonly state: Tone.State;
        /**
         *  Get the playback state at the given time
         *  @param  {Time}  time  The time to test the state at
         *  @return  {Tone.State}  The playback state.
         */
        getStateAtTime(time: Time): Tone.State;
        /**
         *  Start the buffer
         *  @param  {Time} [startTime=now] When the player should start.
         *  @param  {Time} [offset=0] The offset from the beginning of the sample
         *                                 to start at.
         *  @param  {Time=} duration How long the sample should play. If no duration
         *                                is given, it will default to the full length
         *                                of the sample (minus any offset)
         *  @param  {Gain}  [gain=1]  The gain to play the buffer back at.
         *  @return  {Tone.BufferSource}  this
         */
        start(startTime?: Time, offset?: Time, duration?: Time, gain?: Gain): Tone.BufferSource;
        /**
         *  Stop the buffer.
         *  @param  {Time=}  time         The time the buffer should stop.
         *  @return  {Tone.BufferSource}  this
         */
        stop(time?: Time): Tone.BufferSource;
        /**
         *  Cancel a scheduled stop event
         *  @return  {Tone.BufferSource}  this
         */
        cancelStop(): Tone.BufferSource;
        /**
         * If loop is true, the loop will start at this position.
         * @memberOf Tone.BufferSource#
         * @type {Time}
         * @name loopStart
         */
        loopStart: Time;
        /**
         * If loop is true, the loop will end at this position.
         * @memberOf Tone.BufferSource#
         * @type {Time}
         * @name loopEnd
         */
        loopEnd: Time;
        /**
         * The audio buffer belonging to the player.
         * @memberOf Tone.BufferSource#
         * @type {Tone.Buffer}
         * @name buffer
         */
        buffer: Tone.Buffer;
        /**
         * If the buffer should loop once it's over.
         * @memberOf Tone.BufferSource#
         * @type {Boolean}
         * @name loop
         */
        loop: boolean;
        /**
         *  Clean up.
         *  @return  {Tone.BufferSource}  this
         */
        dispose(): Tone.BufferSource;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.FMOscillator
     *
     *  @extends {Tone.Source}
     *  @constructor
     *  @param {Frequency} frequency The starting frequency of the oscillator.
     *  @param {String} type The type of the carrier oscillator.
     *  @param {String} modulationType The type of the modulator oscillator.
     *  @example
     * //a sine oscillator frequency-modulated by a square wave
     * var fmOsc = new Tone.FMOscillator("Ab3", "sine", "square").toMaster().start();
     */
    class FMOscillator extends Tone.Source {
        constructor(frequency: Frequency, type: string, modulationType: string);
        /**
         *  The oscillator's frequency
         *  @type {Frequency}
         *  @signal
         */
        frequency: Frequency;
        /**
         *  The detune control signal.
         *  @type {Cents}
         *  @signal
         */
        detune: Cents;
        /**
         *  The modulation index which is in essence the depth or amount of the modulation. In other terms it is the
         *  ratio of the frequency of the modulating signal (mf) to the amplitude of the
         *  modulating signal (ma) -- as in ma/mf.
         *	@type {Positive}
         *	@signal
         */
        modulationIndex: Positive;
        /**
         *  Harmonicity is the frequency ratio between the carrier and the modulator oscillators.
         *  A harmonicity of 1 gives both oscillators the same frequency.
         *  Harmonicity = 2 means a change of an octave.
         *  @type {Positive}
         *  @signal
         *  @example
         * //pitch the modulator an octave below carrier
         * synth.harmonicity.value = 0.5;
         */
        harmonicity: Positive;
        /**
         *  default values
         *  @static
         *  @type {Object}
         *  @const
         */
        static readonly defaults: any;
        /**
         * The type of the carrier oscillator
         * @memberOf Tone.FMOscillator#
         * @type {string}
         * @name type
         */
        type: string;
        /**
         * The oscillator type without the partialsCount appended to the end
         * @memberOf Tone.FMOscillator#
         * @type {string}
         * @name baseType
         * @example
         * osc.type = 'sine2'
         * osc.baseType //'sine'
         * osc.partialCount = 2
         */
        baseType: string;
        /**
         * 'partialCount' offers an alternative way to set the number of used partials.
         * When partialCount is 0, the maximum number of partials are used when representing
         * the waveform using the periodicWave. When 'partials' is set, this value is
         * not settable, but equals the length of the partials array.
         * @memberOf Tone.FMOscillator#
         * @type {Number}
         * @name partialCount
         */
        partialCount: number;
        /**
         * The type of the modulator oscillator
         * @memberOf Tone.FMOscillator#
         * @type {String}
         * @name modulationType
         */
        modulationType: string;
        /**
         * The phase of the oscillator in degrees.
         * @memberOf Tone.FMOscillator#
         * @type {number}
         * @name phase
         */
        phase: number;
        /**
         * The partials of the carrier waveform. A partial represents
         * the amplitude at a harmonic. The first harmonic is the
         * fundamental frequency, the second is the octave and so on
         * following the harmonic series.
         * Setting this value will automatically set the type to "custom".
         * The value is an empty array when the type is not "custom".
         * @memberOf Tone.FMOscillator#
         * @type {Array}
         * @name partials
         * @example
         * osc.partials = [1, 0.2, 0.01];
         */
        partials: Array;
        /**
         *  Clean up.
         *  @return {Tone.FMOscillator} this
         */
        dispose(): Tone.FMOscillator;
        /**
         * Mute the output.
         * @memberOf Tone.Source#
         * @type {boolean}
         * @name mute
         * @example
         * //mute the output
         * source.mute = true;
         */
        mute: boolean;
        /**
         * The fadeIn time of the amplitude envelope.
         * @memberOf Tone.Source#
         * @type {Time}
         * @name fadeIn
         */
        fadeIn: Time;
        /**
         * The fadeOut time of the amplitude envelope.
         * @memberOf Tone.Source#
         * @type {Time}
         * @name fadeOut
         */
        fadeOut: Time;
        /**
         * The volume of the output in decibels.
         * @type {Decibels}
         * @signal
         * @example
         * source.volume.value = -6;
         */
        volume: Decibels;
        /**
         *  Returns the playback state of the source, either "started" or "stopped".
         *  @type {Tone.State}
         *  @readOnly
         *  @memberOf Tone.Source#
         *  @name state
         */
        readonly state: Tone.State;
        /**
         *  Start the source at the specified time. If no time is given,
         *  start the source now.
         *  @param  {Time} [time=now] When the source should be started.
         *  @returns {Tone.Source} this
         *  @example
         * source.start("+0.5"); //starts the source 0.5 seconds from now
         */
        start(time?: Time): Tone.Source;
        /**
         *  Stop the source at the specified time. If no time is given,
         *  stop the source now.
         *  @param  {Time} [time=now] When the source should be stopped.
         *  @returns {Tone.Source} this
         *  @example
         * source.stop(); // stops the source immediately
         */
        stop(time?: Time): Tone.Source;
        /**
         *  Sync the source to the Transport so that all subsequent
         *  calls to `start` and `stop` are synced to the TransportTime
         *  instead of the AudioContext time.
         *
         *  @returns {Tone.Source} this
         *  @example
         * //sync the source so that it plays between 0 and 0.3 on the Transport's timeline
         * source.sync().start(0).stop(0.3);
         * //start the transport.
         * Tone.Transport.start();
         *
         *  @example
         * //start the transport with an offset and the sync'ed sources
         * //will start in the correct position
         * source.sync().start(0.1);
         * //the source will be invoked with an offset of 0.4
         * Tone.Transport.start("+0.5", 0.5);
         */
        sync(): Tone.Source;
        /**
         *  Unsync the source to the Transport. See Tone.Source.sync
         *  @returns {Tone.Source} this
         */
        unsync(): Tone.Source;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.FatOscillator is an array of oscillators with detune spread between the oscillators
     *
     *  @extends {Tone.Source}
     *  @constructor
     *  @param {Frequency} frequency The oscillator's frequency.
     *  @param {String} type The type of the oscillator.
     *  @param {Cents} spread The detune spread between the oscillators.
     *  @example
     * var fatOsc = new Tone.FatOscillator("Ab3", "sine", 40).toMaster().start();
     */
    class FatOscillator extends Tone.Source {
        constructor(frequency: Frequency, type: string, spread: Cents);
        /**
         *  The oscillator's frequency
         *  @type {Frequency}
         *  @signal
         */
        frequency: Frequency;
        /**
         *  The detune control signal.
         *  @type {Cents}
         *  @signal
         */
        detune: Cents;
        /**
         *  default values
         *  @static
         *  @type {Object}
         *  @const
         */
        static readonly defaults: any;
        /**
         * The type of the carrier oscillator
         * @memberOf Tone.FatOscillator#
         * @type {string}
         * @name type
         */
        type: string;
        /**
         * The detune spread between the oscillators. If "count" is
         * set to 3 oscillators and the "spread" is set to 40,
         * the three oscillators would be detuned like this: [-20, 0, 20]
         * for a total detune spread of 40 cents.
         * @memberOf Tone.FatOscillator#
         * @type {Cents}
         * @name spread
         */
        spread: Cents;
        /**
         * The number of detuned oscillators
         * @memberOf Tone.FatOscillator#
         * @type {Number}
         * @name count
         */
        count: number;
        /**
         * The phase of the oscillator in degrees.
         * @memberOf Tone.FatOscillator#
         * @type {Number}
         * @name phase
         */
        phase: number;
        /**
         * The oscillator type without the partialsCount appended to the end
         * @memberOf Tone.FatOscillator#
         * @type {string}
         * @name baseType
         * @example
         * osc.type = 'sine2'
         * osc.baseType //'sine'
         * osc.partialCount = 2
         */
        baseType: string;
        /**
         * The partials of the carrier waveform. A partial represents
         * the amplitude at a harmonic. The first harmonic is the
         * fundamental frequency, the second is the octave and so on
         * following the harmonic series.
         * Setting this value will automatically set the type to "custom".
         * The value is an empty array when the type is not "custom".
         * @memberOf Tone.FatOscillator#
         * @type {Array}
         * @name partials
         * @example
         * osc.partials = [1, 0.2, 0.01];
         */
        partials: Array;
        /**
         * 'partialCount' offers an alternative way to set the number of used partials.
         * When partialCount is 0, the maximum number of partials are used when representing
         * the waveform using the periodicWave. When 'partials' is set, this value is
         * not settable, but equals the length of the partials array.
         * @memberOf Tone.FatOscillator#
         * @type {Number}
         * @name partialCount
         */
        partialCount: number;
        /**
         *  Clean up.
         *  @return {Tone.FatOscillator} this
         */
        dispose(): Tone.FatOscillator;
        /**
         * Mute the output.
         * @memberOf Tone.Source#
         * @type {boolean}
         * @name mute
         * @example
         * //mute the output
         * source.mute = true;
         */
        mute: boolean;
        /**
         * The fadeIn time of the amplitude envelope.
         * @memberOf Tone.Source#
         * @type {Time}
         * @name fadeIn
         */
        fadeIn: Time;
        /**
         * The fadeOut time of the amplitude envelope.
         * @memberOf Tone.Source#
         * @type {Time}
         * @name fadeOut
         */
        fadeOut: Time;
        /**
         * The volume of the output in decibels.
         * @type {Decibels}
         * @signal
         * @example
         * source.volume.value = -6;
         */
        volume: Decibels;
        /**
         *  Returns the playback state of the source, either "started" or "stopped".
         *  @type {Tone.State}
         *  @readOnly
         *  @memberOf Tone.Source#
         *  @name state
         */
        readonly state: Tone.State;
        /**
         *  Start the source at the specified time. If no time is given,
         *  start the source now.
         *  @param  {Time} [time=now] When the source should be started.
         *  @returns {Tone.Source} this
         *  @example
         * source.start("+0.5"); //starts the source 0.5 seconds from now
         */
        start(time?: Time): Tone.Source;
        /**
         *  Stop the source at the specified time. If no time is given,
         *  stop the source now.
         *  @param  {Time} [time=now] When the source should be stopped.
         *  @returns {Tone.Source} this
         *  @example
         * source.stop(); // stops the source immediately
         */
        stop(time?: Time): Tone.Source;
        /**
         *  Sync the source to the Transport so that all subsequent
         *  calls to `start` and `stop` are synced to the TransportTime
         *  instead of the AudioContext time.
         *
         *  @returns {Tone.Source} this
         *  @example
         * //sync the source so that it plays between 0 and 0.3 on the Transport's timeline
         * source.sync().start(0).stop(0.3);
         * //start the transport.
         * Tone.Transport.start();
         *
         *  @example
         * //start the transport with an offset and the sync'ed sources
         * //will start in the correct position
         * source.sync().start(0.1);
         * //the source will be invoked with an offset of 0.4
         * Tone.Transport.start("+0.5", 0.5);
         */
        sync(): Tone.Source;
        /**
         *  Unsync the source to the Transport. See Tone.Source.sync
         *  @returns {Tone.Source} this
         */
        unsync(): Tone.Source;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     * @class Tone.GrainPlayer implements [granular synthesis](https://en.wikipedia.org/wiki/Granular_synthesis).
     *        Granular Synthesis enables you to adjust pitch and playback rate independently. The grainSize is the
     *        amount of time each small chunk of audio is played for and the overlap is the
     *        amount of crossfading transition time between successive grains.
     * @extends {Tone.Source}
     * @param {String|Tone.Buffer} url	The url to load, or the Tone.Buffer to play.
     * @param {Function=} callback The callback to invoke after the url is loaded.
     */
    class GrainPlayer extends Tone.Source {
        constructor(url: string | Tone.Buffer, callback?: (...params: any[]) => any);
        /**
         *  The audio buffer belonging to the player.
         *  @type  {Tone.Buffer}
         */
        buffer: Tone.Buffer;
        /**
         *  Adjust the pitch independently of the playbackRate.
         *  @type  {Cents}
         */
        detune: Cents;
        /**
         *  the default parameters
         *  @static
         *  @const
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         *  Play the buffer at the given startTime. Optionally add an offset
         *  and/or duration which will play the buffer from a position
         *  within the buffer for the given duration.
         *
         *  @param  {Time} [startTime=now] When the player should start.
         *  @param  {Time} [offset=0] The offset from the beginning of the sample
         *                                 to start at.
         *  @param  {Time=} duration How long the sample should play. If no duration
         *                                is given, it will default to the full length
         *                                of the sample (minus any offset)
         *  @returns {Tone.GrainPlayer} this
         *  @memberOf Tone.GrainPlayer#
         *  @method start
         *  @name start
         */
        start(startTime?: Time, offset?: Time, duration?: Time): Tone.GrainPlayer;
        /**
         * The playback rate of the sample
         * @memberOf Tone.GrainPlayer#
         * @type {Positive}
         * @name playbackRate
         */
        playbackRate: Positive;
        /**
         * The loop start time.
         * @memberOf Tone.GrainPlayer#
         * @type {Time}
         * @name loopStart
         */
        loopStart: Time;
        /**
         * The loop end time.
         * @memberOf Tone.GrainPlayer#
         * @type {Time}
         * @name loopEnd
         */
        loopEnd: Time;
        /**
         * The direction the buffer should play in
         * @memberOf Tone.GrainPlayer#
         * @type {boolean}
         * @name reverse
         */
        reverse: boolean;
        /**
         * The size of each chunk of audio that the
         * buffer is chopped into and played back at.
         * @memberOf Tone.GrainPlayer#
         * @type {Time}
         * @name grainSize
         */
        grainSize: Time;
        /**
         * This is the duration of the cross-fade between
         * sucessive grains.
         * @memberOf Tone.GrainPlayer#
         * @type {Time}
         * @name overlap
         */
        overlap: Time;
        /**
         * If all the buffer is loaded
         * @memberOf Tone.GrainPlayer#
         * @type {Boolean}
         * @name loaded
         * @readOnly
         */
        readonly loaded: boolean;
        /**
         * Clean up
         * @return {Tone.GrainPlayer} this
         */
        dispose(): Tone.GrainPlayer;
        /**
         * Mute the output.
         * @memberOf Tone.Source#
         * @type {boolean}
         * @name mute
         * @example
         * //mute the output
         * source.mute = true;
         */
        mute: boolean;
        /**
         * The fadeIn time of the amplitude envelope.
         * @memberOf Tone.Source#
         * @type {Time}
         * @name fadeIn
         */
        fadeIn: Time;
        /**
         * The fadeOut time of the amplitude envelope.
         * @memberOf Tone.Source#
         * @type {Time}
         * @name fadeOut
         */
        fadeOut: Time;
        /**
         * The volume of the output in decibels.
         * @type {Decibels}
         * @signal
         * @example
         * source.volume.value = -6;
         */
        volume: Decibels;
        /**
         *  Returns the playback state of the source, either "started" or "stopped".
         *  @type {Tone.State}
         *  @readOnly
         *  @memberOf Tone.Source#
         *  @name state
         */
        readonly state: Tone.State;
        /**
         *  Stop the source at the specified time. If no time is given,
         *  stop the source now.
         *  @param  {Time} [time=now] When the source should be stopped.
         *  @returns {Tone.Source} this
         *  @example
         * source.stop(); // stops the source immediately
         */
        stop(time?: Time): Tone.Source;
        /**
         *  Sync the source to the Transport so that all subsequent
         *  calls to `start` and `stop` are synced to the TransportTime
         *  instead of the AudioContext time.
         *
         *  @returns {Tone.Source} this
         *  @example
         * //sync the source so that it plays between 0 and 0.3 on the Transport's timeline
         * source.sync().start(0).stop(0.3);
         * //start the transport.
         * Tone.Transport.start();
         *
         *  @example
         * //start the transport with an offset and the sync'ed sources
         * //will start in the correct position
         * source.sync().start(0.1);
         * //the source will be invoked with an offset of 0.4
         * Tone.Transport.start("+0.5", 0.5);
         */
        sync(): Tone.Source;
        /**
         *  Unsync the source to the Transport. See Tone.Source.sync
         *  @returns {Tone.Source} this
         */
        unsync(): Tone.Source;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class  Tone.Noise is a noise generator. It uses looped noise buffers to save on performance.
     *          Tone.Noise supports the noise types: "pink", "white", and "brown". Read more about
     *          colors of noise on [Wikipedia](https://en.wikipedia.org/wiki/Colors_of_noise).
     *
     *  @constructor
     *  @extends {Tone.Source}
     *  @param {string} type the noise type (white|pink|brown)
     *  @example
     * //initialize the noise and start
     * var noise = new Tone.Noise("pink").start();
     *
     * //make an autofilter to shape the noise
     * var autoFilter = new Tone.AutoFilter({
     * 	"frequency" : "8m",
     * 	"min" : 800,
     * 	"max" : 15000
     * }).connect(Tone.Master);
     *
     * //connect the noise
     * noise.connect(autoFilter);
     * //start the autofilter LFO
     * autoFilter.start()
     */
    class Noise extends Tone.Source {
        constructor(type: string);
        /**
         *  The playback rate of the noise. Affects
         *  the "frequency" of the noise.
         *  @type {Positive}
         *  @signal
         */
        _playbackRate: Positive;
        /**
         *  the default parameters
         *
         *  @static
         *  @const
         *  @type {Object}
         */
        static readonly defaults: any;
        /**
         * The type of the noise. Can be "white", "brown", or "pink".
         * @memberOf Tone.Noise#
         * @type {string}
         * @name type
         * @example
         * noise.type = "white";
         */
        type: string;
        /**
         * Restarts the noise.
         * @param  {Time} time When to restart the noise.
         * @return {Tone.Noise}      this
         */
        restart(time: Time): Tone.Noise;
        /**
         *  Clean up.
         *  @returns {Tone.Noise} this
         */
        dispose(): Tone.Noise;
        /**
         * Mute the output.
         * @memberOf Tone.Source#
         * @type {boolean}
         * @name mute
         * @example
         * //mute the output
         * source.mute = true;
         */
        mute: boolean;
        /**
         * The fadeIn time of the amplitude envelope.
         * @memberOf Tone.Source#
         * @type {Time}
         * @name fadeIn
         */
        fadeIn: Time;
        /**
         * The fadeOut time of the amplitude envelope.
         * @memberOf Tone.Source#
         * @type {Time}
         * @name fadeOut
         */
        fadeOut: Time;
        /**
         * The volume of the output in decibels.
         * @type {Decibels}
         * @signal
         * @example
         * source.volume.value = -6;
         */
        volume: Decibels;
        /**
         *  Returns the playback state of the source, either "started" or "stopped".
         *  @type {Tone.State}
         *  @readOnly
         *  @memberOf Tone.Source#
         *  @name state
         */
        readonly state: Tone.State;
        /**
         *  Start the source at the specified time. If no time is given,
         *  start the source now.
         *  @param  {Time} [time=now] When the source should be started.
         *  @returns {Tone.Source} this
         *  @example
         * source.start("+0.5"); //starts the source 0.5 seconds from now
         */
        start(time?: Time): Tone.Source;
        /**
         *  Stop the source at the specified time. If no time is given,
         *  stop the source now.
         *  @param  {Time} [time=now] When the source should be stopped.
         *  @returns {Tone.Source} this
         *  @example
         * source.stop(); // stops the source immediately
         */
        stop(time?: Time): Tone.Source;
        /**
         *  Sync the source to the Transport so that all subsequent
         *  calls to `start` and `stop` are synced to the TransportTime
         *  instead of the AudioContext time.
         *
         *  @returns {Tone.Source} this
         *  @example
         * //sync the source so that it plays between 0 and 0.3 on the Transport's timeline
         * source.sync().start(0).stop(0.3);
         * //start the transport.
         * Tone.Transport.start();
         *
         *  @example
         * //start the transport with an offset and the sync'ed sources
         * //will start in the correct position
         * source.sync().start(0.1);
         * //the source will be invoked with an offset of 0.4
         * Tone.Transport.start("+0.5", 0.5);
         */
        sync(): Tone.Source;
        /**
         *  Unsync the source to the Transport. See Tone.Source.sync
         *  @returns {Tone.Source} this
         */
        unsync(): Tone.Source;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): Hertz;
        /**
         *  Convert a time representation into ticks.
         *  @param  {Time} time
         *  @return {Ticks}  the time in ticks
         */
        toTicks(time: Time): Ticks;
    }
    /**
     *  @class Tone.OmniOscillator aggregates Tone.Oscillator, Tone.PulseOscillator,
     *         Tone.PWMOscillator, Tone.FMOscillator, Tone.AMOscillator, and Tone.FatOscillator
     *         into one class. The oscillator class can be changed by setting the `type`.
     *         `omniOsc.type = "pwm"` will set it to the Tone.PWMOscillator. Prefixing
     *         any of the basic types ("sine", "square4", etc.) with "fm", "am", or "fat"
     *         will use the FMOscillator, AMOscillator or FatOscillator respectively.
     *         For example: `omniOsc.type = "fatsawtooth"` will create set the oscillator
     *         to a FatOscillator of type "sawtooth".
     *
     *  @extends {Tone.Source}
     *  @constructor
     *  @param {Frequency} frequency The initial frequency of the oscillator.
     *  @param {String} type The type of the oscillator.
     *  @example
     *  var omniOsc = new Tone.OmniOscillator("C#4", "pwm");
     */
    class OmniOscillator extends Tone.Source {
        constructor(frequency: Frequency, type: string);
        /**
         *  The frequency control.
         *  @type {Frequency}
         *  @signal
         */
        frequency: Frequency;
        /**
         *  The detune control
         *  @type {Cents}
         *  @signal
         */
        detune: Cents;
        /**
         *  default values
         *  @static
         *  @type {Object}
         *  @const
         */
        static readonly defaults: any;
        /**
         * The type of the oscillator. Can be any of the basic types: sine, square, triangle, sawtooth. Or
         * prefix the basic types with "fm", "am", or "fat" to use the FMOscillator, AMOscillator or FatOscillator
         * types. The oscillator could also be set to "pwm" or "pulse". All of the parameters of the
         * oscillator's class are accessible when the oscillator is set to that type, but throws an error
         * when it's not.
         *
         * @memberOf Tone.OmniOscillator#
         * @type {String}
         * @name type
         * @example
         * omniOsc.type = "pwm";
         * //modulationFrequency is parameter which is available
         * //only when the type is "pwm".
         * omniOsc.modulationFrequency.value = 0.5;
         * @example
         * //an square wave frequency modulated by a sawtooth
         * omniOsc.type = "fmsquare";
         * omniOsc.modulationType = "sawtooth";
         */
        type: string;
        /**
         * The partials of the waveform. A partial represents
         * the amplitude at a harmonic. The first harmonic is the
         * fundamental frequency, the second is the octave and so on
         * following the harmonic series.
         * Setting this value will automatically set the type to "custom".
         * The value is an empty array when the type is not "custom".
         * This is not available on "pwm" and "pulse" oscillator types.
         * @memberOf Tone.OmniOscillator#
         * @type {Array}
         * @name partials
         * @example
         * osc.partials = [1, 0.2, 0.01];
         */
        partials: Array;
        /**
         * The partial count of the oscillator. This is not available on "pwm" and "pulse" oscillator types.
         * @memberOf Tone.OmniOscillator#
         * @type {Number}
         * @name partialCount
         * @example
         * //set the maximum number of partials
         * osc.partialCount = 0;
         */
        partialCount: number;
        /**
         *  Set a member/attribute of the oscillator.
         *  @param {Object|String} params
         *  @param {number=} value
         *  @param {Time=} rampTime
         *  @returns {Tone.OmniOscillator} this
         */
        set(params: any | string, value?: number, rampTime?: Time): Tone.OmniOscillator;
        /**
         *  Get the object's attributes. Given no arguments get
         *  will return all available object properties and their corresponding
         *  values. Pass in a single attribute to retrieve or an array
         *  of attributes. The attribute strings can also include a "."
         *  to access deeper properties.
         *  @param {Array=|string|undefined} params the parameters to get, otherwise will return
         *  					                  all available.
         *  @returns {Object}
         */
        get(params: Array | string | undefined): any;
        /**
         * The phase of the oscillator in degrees.
         * @memberOf Tone.OmniOscillator#
         * @type {Degrees}
         * @name phase
         */
        phase: Degrees;
        /**
         * The source type of the oscillator.
         * @memberOf Tone.OmniOscillator#
         * @type {String}
         * @name sourceType
         * @example
         * var omniOsc = new Tone.OmniOscillator(440, "fmsquare");
         * omniOsc.sourceType // 'fm'
         */
        sourceType: string;
        /**
         * The base type of the oscillator.
         * @memberOf Tone.OmniOscillator#
         * @type {String}
         * @name baseType
         * @example
         * var omniOsc = new Tone.OmniOscillator(440, "fmsquare4");
         * omniOsc.sourceType // 'fm'
         * omniOsc.baseType //'square'
         * omniOsc.partialCount //4
         */
        baseType: string;
        /**
         * The width of the oscillator (only if the oscillator is set to "pulse")
         * @memberOf Tone.OmniOscillator#
         * @type {NormalRange}
         * @signal
         * @name width
         * @example
         * var omniOsc = new Tone.OmniOscillator(440, "pulse");
         * //can access the width attribute only if type === "pulse"
         * omniOsc.width.value = 0.2;
         */
        width: NormalRange;
        /**
         * The number of detuned oscillators
         * @memberOf Tone.OmniOscillator#
         * @type {Number}
         * @name count
         */
        count: number;
        /**
         * The detune spread between the oscillators. If "count" is
         * set to 3 oscillators and the "spread" is set to 40,
         * the three oscillators would be detuned like this: [-20, 0, 20]
         * for a total detune spread of 40 cents. See Tone.FatOscillator
         * for more info.
         * @memberOf Tone.OmniOscillator#
         * @type {Cents}
         * @name spread
         */
        spread: Cents;
        /**
         * The type of the modulator oscillator. Only if the oscillator
         * is set to "am" or "fm" types. see. Tone.AMOscillator or Tone.FMOscillator
         * for more info.
         * @memberOf Tone.OmniOscillator#
         * @type {String}
         * @name modulationType
         */
        modulationType: string;
        /**
         * The modulation index which is in essence the depth or amount of the modulation. In other terms it is the
         * ratio of the frequency of the modulating signal (mf) to the amplitude of the
         * modulating signal (ma) -- as in ma/mf.
         * See Tone.FMOscillator for more info.
         * @type {Positive}
         * @signal
         * @name modulationIndex
         * @memberof Tone.OmniOscillator#
         */
        modulationIndex: Positive;
        /**
         *  Harmonicity is the frequency ratio between the carrier and the modulator oscillators.
         *  A harmonicity of 1 gives both oscillators the same frequency.
         *  Harmonicity = 2 means a change of an octave. See Tone.AMOscillator or Tone.FMOscillator
         *  for more info.
         *  @memberOf Tone.OmniOscillator#
         *  @signal
         *  @type {Positive}
         *  @name harmonicity
         */
        harmonicity: Positive;
        /**
         * The modulationFrequency Signal of the oscillator
         * (only if the oscillator type is set to pwm). See
         * Tone.PWMOscillator for more info.
         * @memberOf Tone.OmniOscillator#
         * @type {Frequency}
         * @signal
         * @name modulationFrequency
         * @example
         * var omniOsc = new Tone.OmniOscillator(440, "pwm");
         * //can access the modulationFrequency attribute only if type === "pwm"
         * omniOsc.modulationFrequency.value = 0.2;
         */
        modulationFrequency: Frequency;
        /**
         *  Clean up.
         *  @return {Tone.OmniOscillator} this
         */
        dispose(): Tone.OmniOscillator;
        /**
         * Mute the output.
         * @memberOf Tone.Source#
         * @type {boolean}
         * @name mute
         * @example
         * //mute the output
         * source.mute = true;
         */
        mute: boolean;
        /**
         * The fadeIn time of the amplitude envelope.
         * @memberOf Tone.Source#
         * @type {Time}
         * @name fadeIn
         */
        fadeIn: Time;
        /**
         * The fadeOut time of the amplitude envelope.
         * @memberOf Tone.Source#
         * @type {Time}
         * @name fadeOut
         */
        fadeOut: Time;
        /**
         * The volume of the output in decibels.
         * @type {Decibels}
         * @signal
         * @example
         * source.volume.value = -6;
         */
        volume: Decibels;
        /**
         *  Returns the playback state of the source, either "started" or "stopped".
         *  @type {Tone.State}
         *  @readOnly
         *  @memberOf Tone.Source#
         *  @name state
         */
        readonly state: Tone.State;
        /**
         *  Start the source at the specified time. If no time is given,
         *  start the source now.
         *  @param  {Time} [time=now] When the source should be started.
         *  @returns {Tone.Source} this
         *  @example
         * source.start("+0.5"); //starts the source 0.5 seconds from now
         */
        start(time?: Time): Tone.Source;
        /**
         *  Stop the source at the specified time. If no time is given,
         *  stop the source now.
         *  @param  {Time} [time=now] When the source should be stopped.
         *  @returns {Tone.Source} this
         *  @example
         * source.stop(); // stops the source immediately
         */
        stop(time?: Time): Tone.Source;
        /**
         *  Sync the source to the Transport so that all subsequent
         *  calls to `start` and `stop` are synced to the TransportTime
         *  instead of the AudioContext time.
         *
         *  @returns {Tone.Source} this
         *  @example
         * //sync the source so that it plays between 0 and 0.3 on the Transport's timeline
         * source.sync().start(0).stop(0.3);
         * //start the transport.
         * Tone.Transport.start();
         *
         *  @example
         * //start the transport with an offset and the sync'ed sources
         * //will start in the correct position
         * source.sync().start(0.1);
         * //the source will be invoked with an offset of 0.4
         * Tone.Transport.start("+0.5", 0.5);
         */
        sync(): Tone.Source;
        /**
         *  Unsync the source to the Transport. See Tone.Source.sync
         *  @returns {Tone.Source} this
         */
        unsync(): Tone.Source;
        /**
         * Get the audio context belonging to this instance.
         * @type {Tone.Context}
         * @memberOf Tone.AudioNode#
         * @name context
         * @readOnly
         */
        readonly context: Tone.Context;
        /**
         *  channelCount is the number of channels used when up-mixing and down-mixing
         *  connections to any inputs to the node. The default value is 2 except for
         *  specific nodes where its value is specially determined.
         *
         *  @memberof Tone.AudioNode#
         *  @type {Number}
         *  @name channelCount
         *  @readOnly
         */
        readonly channelCount: number;
        /**
         *  channelCountMode determines how channels will be counted when up-mixing and
         *  down-mixing connections to any inputs to the node.
         *  The default value is "max". This attribute has no effect for nodes with no inputs.
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelCountMode
         *  @readOnly
         */
        readonly channelCountMode: string;
        /**
         *  channelInterpretation determines how individual channels will be treated
         *  when up-mixing and down-mixing connections to any inputs to the node.
         *  The default value is "speakers".
         *  @memberof Tone.AudioNode#
         *  @type {String}
         *  @name channelInterpretation
         *  @readOnly
         */
        readonly channelInterpretation: string;
        /**
         *  The number of inputs feeding into the AudioNode.
         *  For source nodes, this will be 0.
         *  @type {Number}
         *  @name numberOfInputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfInputs: number;
        /**
         *  The number of outputs coming out of the AudioNode.
         *  @type {Number}
         *  @name numberOfOutputs
         *  @memberof Tone.AudioNode#
         *  @readOnly
         */
        readonly numberOfOutputs: number;
        /**
         *  connect the output of a ToneNode to an AudioParam, AudioNode, or ToneNode
         *  @param  {Tone | AudioParam | AudioNode} unit
         *  @param {number} [outputNum=0] optionally which output to connect from
         *  @param {number} [inputNum=0] optionally which input to connect to
         *  @returns {Tone.AudioNode} this
         */
        connect(unit: Tone | AudioParam | AudioNode, outputNum?: number, inputNum?: number): Tone.AudioNode;
        /**
         *  disconnect the output
         *  @param {Number|AudioNode} output Either the output index to disconnect
         *                                   if the output is an array, or the
         *                                   node to disconnect from.
         *  @returns {Tone.AudioNode} this
         */
        disconnect(output: number | AudioNode): Tone.AudioNode;
        /**
         *  Connect the output of this node to the rest of the nodes in series.
         *  @example
         *  //connect a node to an effect, panVol and then to the master output
         *  node.chain(effect, panVol, Tone.Master);
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        chain(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  connect the output of this node to the rest of the nodes in parallel.
         *  @param {...(AudioParam|Tone|AudioNode)} nodes
         *  @returns {Tone.AudioNode} this
         */
        fan(...nodes: (AudioParam | Tone | AudioNode)[]): Tone.AudioNode;
        /**
         *  Connect 'this' to the master output. Shorthand for this.connect(Tone.Master)
         *  @returns {Tone.AudioNode} this
         *  @example
         * //connect an oscillator to the master output
         * var osc = new Tone.Oscillator().toMaster();
         */
        toMaster(): Tone.AudioNode;
        /**
         *  Send this signal to the channel name.
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Decibels} amount The amount of the source to send to the bus.
         *  @return {GainNode} The gain node which connects this node to the desired channel.
         *                     Can be used to adjust the levels of the send.
         *  @example
         * source.send("reverb", -12);
         */
        send(channelName: string, amount: Decibels): GainNode;
        /**
         *  Receive the input from the desired channelName to the input
         *
         *  @param  {String} channelName A named channel to send the signal to.
         *  @param  {Number=} channelNumber The channel to connect to
         *  @returns {Tone} this
         *  @example
         * reverbEffect.receive("reverb");
         */
        receive(channelName: string, channelNumber?: number): Tone;
        /**
         *  Convert Time into seconds.
         *
         *  Unlike the method which it overrides, this takes into account
         *  transporttime and musical notation.
         *
         *  Time : 1.40
         *  Notation: 4n or 1m or 2t
         *  Now Relative: +3n
         *
         *  @param  {Time} time
         *  @return {Seconds}
         */
        toSeconds(time: Time): Seconds;
        /**
         *  Convert a frequency representation into a number.
         *  @param  {Frequency} freq
         *  @return {Hertz}      the frequency in hertz
         */
        toFrequency(freq: Frequency): H