<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;
use DateTime;
use App\User;
use App\TraderMember;
use App\Exports\TraderMembersExport;
use App\Imports\TraderMembersImport;
use Log;

class TraderMemberController extends Controller
{
    public function __construct()
	{
	    $this->middleware('auth');
	}
    public function export($trader_id, $type = 'xlsx', Request $request) {
        return new TraderMembersExport($trader_id, $type, $request);
    }

    public function import($trader_id, Request $request) {
        $validator = Validator::make($request->all(), [
            'file' => 'required|max:5000|mimes:xlsx,xls,csv'
        ]);

        if($validator) {
            $dateTime = date('Ymd_His');
            $file = $request->file;
            $fileName = $dateTime . '-' . $file->getClientOriginalName();
            $savePath = public_path('uploads');
            $file->move($savePath, $fileName);

            $filePath = public_path('uploads') . '\\' . $fileName;
            Excel::import(new TraderMembersImport($trader_id), $filePath);

            return response()->json(['success' => 'File uploaded successful.']);
        }

        return response()->json(['errors' => $validator->errors->all()]);
    }

    /**
     * Display a listing of the trader-members.
     *
     * @return Illuminate\View\View
     */
    public function index(Request $request) {
        $trader_members = TraderMember::paginate(env('LIST_SIZE', 10));

        if($request->ajax()){
            $return_array = compact('trader_members');
            return json_encode($return_array);
        }

        return view('trader-members.index', compact('trader_members'));
    }

    /**
     * Show the form for creating a new trader-member.
     *
     * @return Illuminate\View\View
     */
    public function create(Request $request)
    {
        $trader_members = TraderMember::pluck('user_id')->all();

        if($request->ajax()){
            $return_array = compact('trader_members');
            return json_encode($return_array);
        }

        return view('trader-members.create', compact('trader_members'));
    }

    /**
     * Store a new trader-member in the storage.
     *
     * @param Illuminate\Http\Request $request
     *
     * @return Illuminate\Http\RedirectResponse | Illuminate\Routing\Redirector
     */
    public function store(Request $request)
    {
        $data = $this->getData($request);

        $trader_member=TraderMember::create($data)->id;
        if($trader_member) {
            return redirect()->route('trader-members.index')
                ->with('success_message', 'Member was successfully added.');
        } else {
            return back()->withInput()
                ->withErrors(['unexpected_error' => 'Unexpected error occurred while trying to process your request.']);

        }
    }

    /**
     * Display the specified trader-member.
     *
     * @param int $id
     *
     * @return Illuminate\View\View
     */
    public function show($id)
    {
        $trader_member = TraderMember::findOrFail($id);

        return view('trader-members.show', compact('trader_member'));
    }

    /**
     * Show the form for editing the specified trader-member.
     *
     * @param int $id
     *
     * @return Illuminate\View\View
     */
    public function edit($id)
    {
        $trader_member = TraderMember::findOrFail($id);

        return view('trader-members.edit', compact('trader_member'));
    }

    /**
     * Update the specified trader-member in the storage.
     *
     * @param int $id
     * @param Illuminate\Http\Request $request
     *
     * @return Illuminate\Http\RedirectResponse | Illuminate\Routing\Redirector
     */
    public function update($id, Request $request)
    {
        $data = $this->getData($request);

        $trader_member = TraderMember::findOrFail($id);
        $updated_trader_member = $trader_member->update($data);
        if($updated_trader_member) {
            return redirect()->route('trader-members.index')
                ->with('success_message', 'Member was successfully updated.');
        } else {
            return back()->withInput()
                ->withErrors(['unexpected_error' => 'Unexpected error occurred while trying to process your request.']);
        }
    }

    /**
     * Remove the specified trader-member from the storage.
     *
     * @param int $id
     *
     * @return Illuminate\Http\RedirectResponse | Illuminate\Routing\Redirector
     */
    public function destroy($id)
    {
        $trader_member = TraderMember::findOrFail($id);
        $deleted_trader_member = $trader_member->delete();
        if($deleted_trader_member) {
            return redirect()->route('trader-members.index')
                ->with('success_message', 'Member was successfully deleted.');
        } else {
            return back()->withInput()
                ->withErrors(['unexpected_error' => 'Unexpected error occurred while trying to process your request.']);
        }
    }

    public function storeUser(Request $request) {
        $user['username'] = $request->username;
        $user['username_canonical'] = $request->username;
        $user['password'] = Hash::make($request->password);
        $user['firstname'] = $request->firstname;
        $user['lastname'] = $request->lastname;
        $user['email'] = $request->email;
        $user['email_canonical'] = $request->email;
        $user['phone'] = $request->phone;
        $user['locked'] = 0;
        $user['enabled'] = 1;
        $user['user_id'] = 0;
        $user['expired'] = 0;
        $user['address'] = null;
        $user['credentials_expired'] = 0;
        $user['salt'] = Hash::make("password");
        $user['roles'] = serialize([]);
        $user['registration_date'] = new DateTime();
        $user['active'] = 1;
        $user['status'] = 6;
        $user['official_code'] = 'anonymous';
        $user['language'] = 'english';
        //create the use account and get the id
        $user_id = User::create($user)->id;
        if ($user_id) return $user_id;
        else return false;
    }

    /**
     * Store a newly created resource in storage using AJAX.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function storeAjax(Request $request)
    {
        Log::info($request->all());
        
        $validator = Validator::make($request->all(), [
            'username' => 'required',
            'firstname' => 'required',
            'lastname' => 'required',
            'email' => 'email|required',
            'phone' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors'=>$validator->errors()->all()]);
        }

        if (! $request->user_id) {
            // create user
            $validator = Validator::make($request->all(), [
                'password' => 'required|min:6',
                'confirm_password' => 'required|same:password'
            ]);

            if ($validator->fails()) {
                return response()->json(['errors'=>$validator->errors()->all()]);
            }

            $user_id = $this->storeUser($request);
            if (! $user_id) {
                return response()->json(['errors'=>'User creation failed']);
            }
        } else {
            // update user
            $user_id = $request->user_id;

            $member_user['firstname'] = $request->firstname;
            $member_user['lastname'] = $request->lastname;
            $member_user['email'] = $request->email;
            $member_user['phone'] = $request->phone;
            $member_user['username'] = $request->username;

            if ($request->password != null) {
                $validator = Validator::make($request->all(), [
                    'password' => 'required|min:6',
                    'confirm_password' => 'required|same:password'
                ]);

                if ($validator->fails()) {
                    return response()->json(['errors'=>$validator->errors()->all()]);
                }

                $member_user['password'] = Hash::make($request->password);
            }

            $user = User::findOrFail($user_id);
            if (! $user->update($member_user)) {
                return response()->json(['errors'=>'User update failed']);
            }
        }

        // create or update trader-member
        $trader_member = TraderMember::updateOrCreate(
            ['id' => $request->id],
            ['trader_id' => $request->trader_id, 'active' => $request->active, 'user_id' => $user_id,'type'=>$request->type]
        );

        return response()->json($trader_member->load('user'));
    }

    /**
     * Show the form for editing the specified trader-member.
     *
     * @param int $id
     *
     * @return Illuminate\View\View
     */
    public function editAjax($id)
    {
        $trader_member = TraderMember::findOrFail($id);
        return response()->json($trader_member->load('user'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Product  $product
     * @return \Illuminate\Http\Response
     */
    public function destroyAjax($id)
    {
        $trader_member = TraderMember::where('id', $id)->delete();
        return response()->json($trader_member);
    }
}
