<?php

namespace App\Exports;


use EloquentBuilder;
use Illuminate\Support\Facades\Auth;
use Illuminate\Contracts\Support\Responsable;
use PhpOffice\PhpSpreadsheet\Shared\Date;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use Maatwebsite\Excel\Excel;
use Maatwebsite\Excel\Concerns\WithColumnFormatting;
use Maatwebsite\Excel\Concerns\Exportable;
use Maatwebsite\Excel\Concerns\FromQuery;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use App\ProduceCollection;
use App\User;

class ProduceCollectionsExport implements Responsable, FromQuery, WithHeadings, WithMapping, WithColumnFormatting
{
    use Exportable;

    protected $request;

    /**
    * It's required to define the fileName within
    * the export class when making use of Responsable.
    */
    private $fileName = 'produce-collections.xlsx';

    /**
    * Optional Writer Type
    */
    private $writerType = Excel::XLSX;

    /**
    * Optional headers
    */
    private $headers = [
        'Content-Type' => 'text/csv',
    ];

    public function __construct($request)
    {
        $this->request = $request;
    }

    public function headings(): array
    {
        return [
            'ID',
            'Farmer Organisation',
            'Buyer',
            'Produce',
            'Quality',
            'Unit',
            'Price per Unit',
            'Quantity',
            'Status',
            'Collection Date',
        ];
    }

    public function map($collection): array
    {
        return [
            $collection->id,
            $collection->contract_detail->produce_item->trader->organisation_name,
            $collection->contract_detail->contract->trader->organisation_name,
            $collection->contract_detail->produce_item->produce->name,
            $collection->contract_detail->produce_item->quality,
            $collection->contract_detail->produce_item->produce_unit->name,
            $collection->contract_detail->contract->price_per_unit,
            $collection->quantity,
            $collection->status == 1 ? 'Approved' : 'Pending',
            Date::dateTimeToExcel($collection->created_at),
        ];
    }

    public function columnFormats(): array
    {
        return [
            'J' => NumberFormat::FORMAT_DATE_DDMMYYYY,
        ];
    }

    public function query()
    {
        if(Auth::user()->isTraderAdmin()) {
            $user_id = Auth::user()->id;
            $user = User::findOrFail($user_id);
            $trader = $user->getRelatedTrader();
            $trader_id = $trader->id;

            $query = ProduceCollection::whereHas('contract_details', function($query) use($trader_id) {
                $query->whereHas('contract', function($query) use($trader_id) {
                   $query->where('trader_id', $trader_id);
                });
            });

            return EloquentBuilder::to($query, request()->all());
        }

        return EloquentBuilder::to(ProduceCollection::class, $this->request->all());
    }
}
