<?php

namespace App\Exports;

use EloquentBuilder;
use Illuminate\Support\Facades\Auth;
use Illuminate\Contracts\Support\Responsable;
use PhpOffice\PhpSpreadsheet\Shared\Date;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use Maatwebsite\Excel\Excel;
use Maatwebsite\Excel\Concerns\Exportable;
use Maatwebsite\Excel\Concerns\FromQuery;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithColumnFormatting;
use App\FarmerProduceAggregation;
use App\Trader;
use App\TraderMember;
use App\User;

class ProduceAggregationsExport implements Responsable, FromQuery, WithHeadings, WithMapping, WithColumnFormatting
{
    use Exportable;

    protected $request;

    /**
    * It's required to define the fileName within
    * the export class when making use of Responsable.
    */
    private $fileName = 'aggregations.xlsx';

    /**
    * Optional Writer Type
    */
    private $writerType = Excel::XLSX;

    /**
    * Optional headers
    */
    private $headers = [
        'Content-Type' => 'text/csv',
    ];

    public function __construct($request)
    {
        $this->request = $request;
    }

    public function headings(): array
    {
        return [
            'Organisation',
            'Farmer ID',
            'Farmer Name',
            'Produce',
            'Units',
            'Price per Unit',
            'Quantity',
            'Aggregation Date',
            'Approved',
            'Credited Amount',
        ];
    }

    public function map($aggregation): array
    {
        return [
            $aggregation->farmer->traderMember->trader->organisation_name,
            $aggregation->farmer_id,
            $aggregation->farmer->firstname . ' ' . $aggregation->farmer->lastname,
            $aggregation->produce_item->produce->name,
            $aggregation->produce_item->produce_unit->name,
            $aggregation->price_per_unit,
            $aggregation->quantity,
            Date::dateTimeToExcel($aggregation->created_at),
            $aggregation->approved == 1 ? 'YES' : 'NO',
            $aggregation->approved == 1 ? $aggregation->quantity * $aggregation->price_per_unit : '-',
        ];
    }

    public function columnFormats(): array
    {
        return [
            'H' => NumberFormat::FORMAT_DATE_DDMMYYYY,
        ];
    }

    public function query()
    {
        if(Auth::user()->isTraderAdmin()) {
            $user_id = Auth::user()->id;
            $user = User::findOrFail($user_id);
            $trader = $user->getRelatedTrader();

            $farmers = $trader->users;
            $farmer_ids = $farmers->pluck('id');

            $query = FarmerProduceAggregation::whereIn('farmer_id', $farmer_ids);
            return EloquentBuilder::to($query, request()->all());
        }

        return EloquentBuilder::to(FarmerProduceAggregation::class, $this->request->all());
    }
}
