(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory() :
    typeof define === 'function' && define.amd ? define(factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, global.resolveURI = factory());
})(this, (function () { 'use strict';

    // Matches the scheme of a URL, eg "http://"
    const schemeRegex = /^[\w+.-]+:\/\//;
    /**
     * Matches the parts of a URL:
     * 1. Scheme, including ":", guaranteed.
     * 2. User/password, including "@", optional.
     * 3. Host, guaranteed.
     * 4. Port, including ":", optional.
     * 5. Path, including "/", optional.
     */
    const urlRegex = /^([\w+.-]+:)\/\/([^@]*@)?([^:/]*)(:\d+)?(\/[^#?]*)?/;
    function isAbsoluteUrl(input) {
        return schemeRegex.test(input);
    }
    function isSchemeRelativeUrl(input) {
        return input.startsWith('//');
    }
    function isAbsolutePath(input) {
        return input.startsWith('/');
    }
    function parseAbsoluteUrl(input) {
        const match = urlRegex.exec(input);
        return {
            scheme: match[1],
            user: match[2] || '',
            host: match[3],
            port: match[4] || '',
            path: match[5] || '/',
            relativePath: false,
        };
    }
    function parseUrl(input) {
        if (isSchemeRelativeUrl(input)) {
            const url = parseAbsoluteUrl('http:' + input);
            url.scheme = '';
            return url;
        }
        if (isAbsolutePath(input)) {
            const url = parseAbsoluteUrl('http://foo.com' + input);
            url.scheme = '';
            url.host = '';
            return url;
        }
        if (!isAbsoluteUrl(input)) {
            const url = parseAbsoluteUrl('http://foo.com/' + input);
            url.scheme = '';
            url.host = '';
            url.relativePath = true;
            return url;
        }
        return parseAbsoluteUrl(input);
    }
    function stripPathFilename(path) {
        // If a path ends with a parent directory "..", then it's a relative path with excess parent
        // paths. It's not a file, so we can't strip it.
        if (path.endsWith('/..'))
            return path;
        const index = path.lastIndexOf('/');
        return path.slice(0, index + 1);
    }
    function mergePaths(url, base) {
        // If we're not a relative path, then we're an absolute path, and it doesn't matter what base is.
        if (!url.relativePath)
            return;
        normalizePath(base);
        // If the path is just a "/", then it was an empty path to begin with (remember, we're a relative
        // path).
        if (url.path === '/') {
            url.path = base.path;
        }
        else {
            // Resolution happens relative to the base path's directory, not the file.
            url.path = stripPathFilename(base.path) + url.path;
        }
        // If the base path is absolute, then our path is now absolute too.
        url.relativePath = base.relativePath;
    }
    /**
     * The path can have empty directories "//", unneeded parents "foo/..", or current directory
     * "foo/.". We need to normalize to a standard representation.
     */
    function normalizePath(url) {
        const { relativePath } = url;
        const pieces = url.path.split('/');
        // We need to preserve the first piece always, so that we output a leading slash. The item at
        // pieces[0] is an empty string.
        let pointer = 1;
        // Positive is the number of real directories we've output, used for popping a parent directory.
        // Eg, "foo/bar/.." will have a positive 2, and we can decrement to be left with just "foo".
        let positive = 0;
        // We need to keep a trailing slash if we encounter an empty directory (eg, splitting "foo/" will
        // generate `["foo", ""]` pieces). And, if we pop a parent directory. But once we encounter a
        // real directory, we won't need to append, unless the other conditions happen again.
        let addTrailingSlash = false;
        for (let i = 1; i < pieces.length; i++) {
            const piece = pieces[i];
            // An empty directory, could be a trailing slash, or just a double "//" in the path.
            if (!piece) {
                addTrailingSlash = true;
                continue;
            }
            // If we encounter a real directory, then we don't need to append anymore.
            addTrailingSlash = false;
            // A current directory, which we can always drop.
            if (piece === '.')
                continue;
            // A parent directory, we need to see if there are any real directories we can pop. Else, we
            // have an excess of parents, and we'll need to keep the "..".
            if (piece === '..') {
                if (positive) {
                    addTrailingSlash = true;
                    positive--;
                    pointer--;
                }
                else if (relativePath) {
                    // If we're in a relativePath, then we need to keep the excess parents. Else, in an absolute
                    // URL, protocol relative URL, or an absolute path, we don't need to keep excess.
                    pieces[pointer++] = piece;
                }
                continue;
            }
            // We've encountered a real directory. Move it to the next insertion pointer, which accounts for
            // any popped or dropped directories.
            pieces[pointer++] = piece;
            positive++;
        }
        let path = '';
        for (let i = 1; i < pointer; i++) {
            path += '/' + pieces[i];
        }
        if (!path || (addTrailingSlash && !path.endsWith('/..'))) {
            path += '/';
        }
        url.path = path;
    }
    /**
     * Attempts to resolve `input` URL/path relative to `base`.
     */
    function resolve(input, base) {
        if (!input && !base)
            return '';
        const url = parseUrl(input);
        // If we have a base, and the input isn't already an absolute URL, then we need to merge.
        if (base && !url.scheme) {
            const baseUrl = parseUrl(base);
            url.scheme = baseUrl.scheme;
            // If there's no host, then we were just a path.
            if (!url.host || baseUrl.scheme === 'file:') {
                // The host, user, and port are joined, you can't copy one without the others.
                url.user = baseUrl.user;
                url.host = baseUrl.host;
                url.port = baseUrl.port;
            }
            mergePaths(url, baseUrl);
        }
        normalizePath(url);
        // If the input (and base, if there was one) are both relative, then we need to output a relative.
        if (url.relativePath) {
            // The first char is always a "/".
            const path = url.path.slice(1);
            if (!path)
                return '.';
            // If base started with a leading ".", or there is no base and input started with a ".", then we
            // need to ensure that the relative path starts with a ".". We don't know if relative starts
            // with a "..", though, so check before prepending.
            const keepRelative = (base || input).startsWith('.');
            return !keepRelative || path.startsWith('.') ? path : './' + path;
        }
        // If there's no host (and no scheme/user/port), then we need to output an absolute path.
        if (!url.scheme && !url.host)
            return url.path;
        // We're outputting either an absolute URL, or a protocol relative one.
        return `${url.scheme}//${url.user}${url.host}${url.port}${url.path}`;
    }

    return resolve;

}));
//# sourceMappingURL=resolve-uri.umd.js.map
